package com.jz.tool.token.configuration;

import java.io.IOException;
import java.net.URISyntaxException;
import java.util.Arrays;
import java.util.ResourceBundle;

import org.springframework.beans.BeansException;
import org.springframework.beans.MutablePropertyValues;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.beans.factory.support.BeanDefinitionRegistryPostProcessor;
import org.springframework.beans.factory.support.GenericBeanDefinition;
import org.springframework.boot.context.embedded.FilterRegistrationBean;
import org.springframework.context.annotation.Configuration;

import com.jz.common.utils.text.StringTools;
import com.jz.tool.token.annotation.WebTokenFilter;
import com.jz.tool.token.exception.TokenBeansException;
import com.jz.tool.token.proxy.TokenFilterCglibProxy;

@Configuration
public class TokenBeanDefinitionRegisterProcessor implements BeanDefinitionRegistryPostProcessor {

	private static final String tokenEnabledKey = "com.jz.tools.token.enabled";

	@Override
	public void postProcessBeanFactory(ConfigurableListableBeanFactory beanFactory) throws BeansException {
	}

	@Override
	public void postProcessBeanDefinitionRegistry(BeanDefinitionRegistry registry) throws BeansException {
		ResourceBundle resourceBundle = ResourceBundle.getBundle("application");
		if (resourceBundle.containsKey(tokenEnabledKey) && "false".equals(resourceBundle.getString(tokenEnabledKey))) {
			return;
		}
		// 获取filter class
		Class<?> tokenFilterClass = this.getTokenFilterClass();
		if (null == tokenFilterClass)
			return;
		// 获取注解
		WebTokenFilter tokenFilter = tokenFilterClass.getAnnotation(WebTokenFilter.class);
		// 创建动态代理
		TokenFilterCglibProxy cglibProxy = new TokenFilterCglibProxy(tokenFilterClass);
		// 获取filter名称
		String beanName = tokenFilterClass.getSimpleName().substring(0, 1).toLowerCase();
		beanName = StringTools.ternary(tokenFilter.value(), beanName + tokenFilterClass.getSimpleName().substring(1));
		// 创建注册对象
		GenericBeanDefinition beanDefinition = new GenericBeanDefinition();
		beanDefinition.setBeanClass(FilterRegistrationBean.class);
		MutablePropertyValues propertyValues = beanDefinition.getPropertyValues();
		propertyValues.add("filter", cglibProxy.getDynamicProxyInstance());
		propertyValues.add("order", tokenFilter.order());
		propertyValues.add("urlPatterns", Arrays.asList(tokenFilter.urlPatterns()));
		propertyValues.add("name", beanName);
		registry.registerBeanDefinition(beanName, beanDefinition);
	}

	private Class<?> getTokenFilterClass() {
		try {
			return DiscoveryTokenFilter.discoveryMappingUrl("com.jz");
		} catch (ClassNotFoundException | IOException | URISyntaxException e) {
			throw new TokenBeansException("Cann't found token filter class");
		}
	}

}
