package com.jz.jar.business.proxy.common;

import java.lang.reflect.Type;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.collections.MapUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.gson.reflect.TypeToken;
import com.jz.common.utils.collection.ArrayMapTools;
import com.jz.common.utils.collection.OrderTool;
import com.jz.common.utils.json.GsonTools;
import com.jz.common.utils.text.StringTools;
import com.jz.jar.business.bean.AliyunBean;
import com.jz.jar.business.request.UserRequest;
import com.jz.jar.business.request.WorksRequest;
import com.jz.jar.business.wrapper.UserWrapper;
import com.jz.jar.business.wrapper.WWorkTopicTheme;
import com.jz.jar.business.wrapper.WorksInfoWrapper;
import com.jz.jar.franchise.service.CourseTomatoTopicService;
import com.jz.jar.media.enums.BrandEnum;
import com.jz.jar.media.service.WorksInfoService;
import com.jz.jar.media.service.WorksLikeService;
import com.jz.jooq.franchise.tables.pojos.CourseTomatoTopic;
import com.jz.jooq.media.tables.pojos.WorksInfo;

@Lazy
@Service
public class WorksInfoProxy {
	
	private static final Type pictureType = new TypeToken<List<String>>() {}.getType();
	
	@Autowired
	private CourseTomatoTopicService courseTomatoTopicService;
	@Autowired
	private WorksInfoService worksInfoService;
	@Autowired
	private WorksLikeService worksLikeService;
	@Autowired
	private UserProxy userProxy;

	public List<WorksInfoWrapper> getSimpleWorksInfo(BrandEnum brand, WorksRequest request) {
		if (ArrayMapTools.isEmpty(request.getWorksId()))
			return null;
		List<WorksInfo> worksInfo = worksInfoService.findSimpleWorksInfo(request.getWorksId());
		if (ArrayMapTools.isEmpty(worksInfo))
			return null;
		Map<String, UserWrapper> uid2User = null;
		Map<String, Integer> likeCount = null;
		List<String> likes = null;
		if (request.getWithExtended()) {
			// get user info
			Set<String> puids = worksInfo.stream().map(WorksInfo::getPuid).collect(Collectors.toSet());
			uid2User = userProxy.getSimpleParentInfoHas(brand, UserRequest.of(puids));
			// get art like cnt
			likeCount = worksLikeService.getLikeCount(request.getWorksId());
			// get like art ids
			if (null != request.getUid())
				likes = worksLikeService.screenLike(request.getWorksId(), request.getUid());
		}
		// package wrapper
		List<WorksInfoWrapper> wrappers = Lists.newArrayList();
		for (WorksInfo workInfo : worksInfo) {
			WorksInfoWrapper wrapper = WorksInfoWrapper.of(workInfo);
			wrapper.setPic(AliyunBean.getImagesUrl(workInfo.getPic()));
			if (request.getWithExtended()) {
				// set user info
				wrapper.setUser(uid2User.get(workInfo.getPuid()));
				// set like cnt
				wrapper.setLikeCnt(MapUtils.getInteger(likeCount, workInfo.getId(), 0));
				// set is like
				wrapper.setIsLike(null != likes && likes.contains(workInfo.getId()));
			}
			wrappers.add(wrapper);
		}
		return OrderTool.keepOrder(request.getWorksId(), wrappers, WorksInfoWrapper::getId);
	}

	public Map<String, WorksInfoWrapper> getSimpleWorksInfoHas(BrandEnum brand, WorksRequest request) {
		List<WorksInfoWrapper> artInfo = getSimpleWorksInfo(brand, request);
		if (ArrayMapTools.isEmpty(artInfo))
			return Maps.newHashMap();
		Map<String, WorksInfoWrapper> id2Art = Maps.newHashMap();
		artInfo.forEach(it -> id2Art.put(it.getId(), it));
		return id2Art;
	}
	
	public WorksInfoWrapper getWorkInfo(BrandEnum brand, WorksRequest request) {
		WorksInfo workInfo = worksInfoService.getWorksInfo(request.getWorkId(), brand);
		if (null == workInfo)
			return null;
		boolean isMaster = workInfo.getPuid().equals(request.getUid());
		if (workInfo.getIsOpen() <= 0 && !isMaster)
			return null; // is private & not master
		WorksInfoWrapper wrapper = WorksInfoWrapper.of(workInfo);
		// set resource info
		wrapper.setPic(AliyunBean.getImagesUrl(workInfo.getPic()));
		// set pictures
		if (StringTools.isNotEmptyAndBlank(workInfo.getPictures())) {
			List<String> pics = GsonTools.gson.fromJson(workInfo.getPictures(), pictureType);
			// 可展示图片
			List<String> pictures = Lists.newArrayList();
			pics.forEach(it -> pictures.add(AliyunBean.getImagesUrl(it)));
			wrapper.setPictures(pictures);
		}

		// 作品专题
		CourseTomatoTopic topic = courseTomatoTopicService.getTopic(brand.name(), workInfo.getTid());
		if (null != topic)
			wrapper.setTheme(WWorkTopicTheme.of(topic).setPic(AliyunBean.getImagesUrl(topic.getPic())));

		// like count
		wrapper.setLikeCnt(worksLikeService.countForWorkId(workInfo.getId()));
		// current user is like
		wrapper.setIsLike(worksLikeService.existForIdAndUid(workInfo.getId(), request.getUid()));
		// get user info
		wrapper.setUser(userProxy.getSimpleParentInfo(brand, UserRequest.of(request.getUid(), workInfo.getPuid())));
		// get student info
		if (request.getWithStudent())
			wrapper.setStudent(userProxy.getStudentInfo(workInfo.getSuid(), true, brand));
		return wrapper;
	}
}
