package com.jz.jar.media.repository;

import java.util.Collection;
import java.util.List;

import org.jooq.Condition;
import org.jooq.TableOnConditionStep;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Repository;

import com.jz.jar.media.enums.BrandEnum;
import com.jz.jar.media.enums.SourceStatus;
import com.jz.jooq.media.Tables;
import com.jz.jooq.media.tables.pojos.ArtInfo;

@Lazy
@Repository
public class ArtInfoRepository extends MediaBaseRepository {

	private static final com.jz.jooq.media.tables.ArtInfo AI = Tables.ART_INFO;
	private static final com.jz.jooq.media.tables.ArtThemeRelation ATR = Tables.ART_THEME_RELATION;

	public ArtInfo getArtInfo(String id, BrandEnum brand) {
		return mediaCtx.selectFrom(AI).where(AI.ID.eq(id).and(AI.BRAND.eq(brand.name()))).fetchAnyInto(ArtInfo.class);
	}

	public ArtInfo getSimpleArtInfo(String id, BrandEnum brand) {
		return mediaCtx.select(AI.ID, AI.SCHOOL_ID, AI.LESSON_ID, AI.PUID, AI.SUID, AI.IS_OPEN).from(AI)
				.where(AI.ID.eq(id).and(AI.BRAND.eq(brand.name()))).fetchAnyInto(ArtInfo.class);
	}

	public void updateOpenInfo(String id, BrandEnum brand, int isOpen) {
		mediaCtx.update(AI).set(AI.IS_OPEN, isOpen).set(AI.LAST_UPDATE, System.currentTimeMillis())
				.where(AI.ID.eq(id).and(AI.BRAND.eq(brand.name()))).execute();
	}

	private Condition getCondition(BrandEnum brand, Collection<String> uids) {
		return AI.BRAND.eq(brand.name()).and(AI.PUID.in(uids)).and(AI.IS_OPEN.ge(1))
				.and(AI.STATUS.eq(SourceStatus.online.getCode()));
	}

	public int countOpenNewArt(BrandEnum brand, Collection<String> uids, long timestamp) {
		return mediaCtx.fetchCount(AI, getCondition(brand, uids).and(AI.CREATE_TIME.gt(timestamp)));
	}

	public int countOpenArt(BrandEnum brand, Collection<String> uids) {
		return mediaCtx.fetchCount(AI, getCondition(brand, uids));
	}

	public List<String> findOpenArtIds(BrandEnum brand, Collection<String> uids, int start, int size) {
		return mediaCtx.select(AI.ID).from(AI).where(getCondition(brand, uids)).orderBy(AI.CREATE_TIME.desc())
				.limit(start, size).fetchInto(String.class);
	}

	public List<ArtInfo> findSimpleArtInfo(Collection<String> artIds) {
		return mediaCtx.select(AI.ID, AI.PUID, AI.RESOURCE).from(AI).where(AI.ID.in(artIds)).fetchInto(ArtInfo.class);
	}
	
	private Condition getRecommonCondition(BrandEnum brand) {
		return AI.BRAND.eq(brand.name()).and(AI.IS_OPEN.ge(1)).and(AI.STATUS.eq(SourceStatus.online.getCode()));
	}
	
	public int countRecommonArts(BrandEnum brand) {
		return mediaCtx.fetchCount(AI, getRecommonCondition(brand));
	}
	
	public List<String> findRecommonArtIds(BrandEnum brand, int start, int size) {
		return mediaCtx.select(AI.ID).from(AI).where(getRecommonCondition(brand)).orderBy(AI.CREATE_TIME.desc())
				.limit(start, size).fetchInto(String.class);
	}
	
	/********************************************************************/
	/*********************** art theme relation *************************/
	/********************************************************************/
	
	private TableOnConditionStep getJoinTable() {
		return ATR.join(AI).on(ATR.ART_ID.eq(AI.ID));
	}
	
	private Condition getRelationCondition(BrandEnum brand, String themeId) {
		return ATR.THEME_ID.eq(themeId).and(AI.BRAND.eq(brand.name())).and(AI.IS_OPEN.ge(1))
				.and(AI.STATUS.eq(SourceStatus.online.getCode()));
	}
	
	public int countArts(BrandEnum brand, String themeId) {
		return mediaCtx.fetchCount(getJoinTable(), getRelationCondition(brand, themeId));
	}

	public List<String> getArtIds(BrandEnum brand, String themeId, boolean isHot, int start, int size) {
		return mediaCtx.select(ATR.ART_ID).from(getJoinTable()).where(getRelationCondition(brand, themeId))
				.orderBy(isHot ? ATR.LIKE_CNT.desc() : ATR.UPLOAD_TIME.desc()).limit(start, size)
				.fetchInto(String.class);
	}
	
	public List<String> getAllThemeIds(String artId) {
		return mediaCtx.select(ATR.THEME_ID).from(ATR).where(ATR.ART_ID.eq(artId)).fetchInto(String.class);
	}
}
