package com.jz.jar.media.repository;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.jooq.Condition;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Repository;

import com.jz.common.utils.text.StringTools;
import com.jz.jar.media.enums.AuditStatus;
import com.jz.jar.media.enums.BrandEnum;
import com.jz.jooq.media.Tables;
import com.jz.jooq.media.tables.pojos.WorksInfo;
import com.jz.jooq.media.tables.records.WorksInfoRecord;

@Lazy
@Repository
public class WorksInfoRepository extends MediaBaseRepository implements BaseConditionRepository<WorksInfoRecord> {

	private static final com.jz.jooq.media.tables.WorksInfo WI = Tables.WORKS_INFO;

	/********************************************************************/
	/**************************** condition *****************************/
	/********************************************************************/

	private Condition getOpenOnlineCondition(BrandEnum brand) {
		return WI.IS_OPEN.ge(1).and(WI.STATUS.eq(AuditStatus.online.getCode())).and(WI.BRAND.eq(brand.name()));
	}

	private Condition getOpenOnlineForPuidCondition(BrandEnum brand, Collection<String> uids) {
		return getOpenOnlineCondition(brand).and(WI.PUID.in(uids));
	}

	private Condition getOpenForTopicCondition(BrandEnum brand, String topicId) {
		return getOpenOnlineCondition(brand).and(WI.TID.eq(topicId));
	}

	private Condition getRecommonCondition(BrandEnum brand) {
		return getOpenOnlineCondition(brand).and(WI.IS_RECOMM.eq(1));
	}

	/********************************************************************/
	/**************************** works info ****************************/
	/********************************************************************/

	public boolean existWorksInfo(String id, BrandEnum brand) {
		return mediaCtx.fetchExists(WI, WI.ID.eq(id).and(WI.BRAND.eq(brand.name())));
	}

	public WorksInfo getWorksInfo(String id, BrandEnum brand) {
		return mediaCtx.selectFrom(WI).where(WI.ID.eq(id).and(WI.BRAND.eq(brand.name()))).fetchAnyInto(WorksInfo.class);
	}

	public WorksInfo getSimpleWorksInfo(String id, BrandEnum brand) {
		return mediaCtx.select(WI.ID, WI.SCHOOL_ID, WI.LESSON_ID, WI.PUID, WI.SUID, WI.IS_OPEN).from(WI)
				.where(WI.ID.eq(id).and(WI.BRAND.eq(brand.name()))).fetchAnyInto(WorksInfo.class);
	}

	public List<WorksInfo> findSimpleWorksInfo(Collection<String> artIds) {
		return mediaCtx.select(WI.ID, WI.PUID, WI.SUID, WI.PIC, WI.IS_OPEN).from(WI).where(WI.ID.in(artIds))
				.fetchInto(WorksInfo.class);
	}

	public void updateOpenInfo(String id, BrandEnum brand, int isOpen, int addLikeCnt) {
		mediaCtx.update(WI).set(WI.IS_OPEN, isOpen).set(WI.LIKE_CNT, WI.LIKE_CNT.add(addLikeCnt))
				.set(WI.LAST_UPDATE, System.currentTimeMillis()).where(WI.ID.eq(id).and(WI.BRAND.eq(brand.name())))
				.execute();
	}

	public int countOpenNewWorks(BrandEnum brand, Collection<String> uids, long timestamp) {
		return mediaCtx.fetchCount(WI, getOpenOnlineForPuidCondition(brand, uids).and(WI.CREATE_TIME.gt(timestamp)));
	}

	public int countOpenWorks(BrandEnum brand, Collection<String> uids) {
		return mediaCtx.fetchCount(WI, getOpenOnlineForPuidCondition(brand, uids));
	}

	public List<String> findOpenWorksId(BrandEnum brand, Collection<String> uids, int start, int size) {
		return mediaCtx.select(WI.ID).from(WI).where(getOpenOnlineForPuidCondition(brand, uids))
				.orderBy(WI.CREATE_TIME.desc()).limit(start, size).fetchInto(String.class);
	}

	public int countRecommonWorks(BrandEnum brand) {
		return mediaCtx.fetchCount(WI, getRecommonCondition(brand));
	}

	public List<String> findRecommonWorksId(BrandEnum brand, int start, int size) {
		return mediaCtx.select(WI.ID).from(WI).where(getRecommonCondition(brand)).orderBy(WI.CREATE_TIME.desc())
				.limit(start, size).fetchInto(String.class);
	}

	/** 获取存在作品的学校及课程 */
	public List<WorksInfo> findExistWorksInfo(Map<String, Collection<String>> school2LessonIds, String suid) {
		Condition condition = getOrWhereCondition(WI.SCHOOL_ID, WI.LESSON_ID, school2LessonIds);
		return mediaCtx.select(WI.SCHOOL_ID, WI.LESSON_ID, WI.ID).from(WI)
				.where(condition.and(WI.SUID.eq(suid)).and(WI.STATUS.eq(AuditStatus.online.getCode())))
				.fetchInto(WorksInfo.class);
	}

	/** 学员作品列表 */
	private Condition getStudentCondition(BrandEnum brand, String puid, String suid) {
		Condition condition = WI.BRAND.eq(brand.name()).and(WI.PUID.eq(puid))
				.and(WI.STATUS.eq(AuditStatus.online.getCode()));
		if (StringTools.isNotEmpty(suid))
			return condition.and(WI.SUID.eq(suid));
		return condition;
	}

	public int countStudentWorks(BrandEnum brand, String puid, String suid) {
		return mediaCtx.fetchCount(WI, getStudentCondition(brand, puid, suid));
	}

	public List<String> findStudentWorksId(BrandEnum brand, String puid, String suid, int start, int size) {
		return mediaCtx.select(WI.ID).from(WI).where(getStudentCondition(brand, puid, suid))
				.orderBy(WI.CREATE_TIME.desc()).limit(start, size).fetchInto(String.class);
	}

	public List<String> findAllStudentWorksId(BrandEnum brand, String puid, String suid) {
		return mediaCtx.select(WI.ID).from(WI).where(getStudentCondition(brand, puid, suid)).fetchInto(String.class);
	}

	public int countOpenWorksForTopic(BrandEnum brand, String topicId) {
		return mediaCtx.fetchCount(WI, getOpenForTopicCondition(brand, topicId));
	}

	public List<String> getOpenWorkIdsForTopic(BrandEnum brand, String tid, boolean isHot, int start, int size) {
		return mediaCtx.select(WI.ID).from(WI).where(getOpenForTopicCondition(brand, tid))
				.orderBy(isHot ? WI.LIKE_CNT.desc() : WI.LAST_UPDATE.desc()).limit(start, size).fetchInto(String.class);
	}

	public void addLikeCnt(String id, BrandEnum brand, int addLikeCnt) {
		mediaCtx.update(WI).set(WI.LIKE_CNT, WI.LIKE_CNT.add(addLikeCnt))
				.where(WI.ID.eq(id).and(WI.BRAND.eq(brand.name()))).execute();
	}
}
