package com.jz.gymchina.jar.resources.repository;

import java.util.Collection;
import java.util.List;

import org.jooq.Condition;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Repository;

import com.google.common.collect.Lists;
import com.jz.gymchina.jar.resources.enums.ResourceEnum;
import com.jz.jooq.gymchina.resources.Tables;
import com.jz.jooq.gymchina.resources.tables.pojos.GymDirectory;
import com.jz.jooq.gymchina.resources.tables.records.GymDirectoryRecord;

/**
 * @Title DirectoryRepository
 * @Package org.gymchina.resources.repository
 * @author tangjunfeng
 * @date 2019年3月11日 下午12:08:29
 * @version V1.0
 */
@Lazy
@Repository
public class DirectoryRepository extends ResourceBaseRepository {

	private static com.jz.jooq.gymchina.resources.tables.GymDirectory DIR = Tables.GYM_DIRECTORY;

	public List<GymDirectory> getOnlineDirectories() {
		return resourcesCtx.select(DIR.ID, DIR.PARENT_ID).from(DIR).where(DIR.STATUS.eq(ResourceEnum.online.name()))
				.fetchInto(GymDirectory.class);
	}

	private Condition getWhereCondition(String dirId) {
		Condition condition = DIR.STATUS.eq(ResourceEnum.online.name());
		if (null == dirId)
			return condition.and(DIR.RANK_IDX.eq(1));
		return condition.and(DIR.PARENT_ID.eq(dirId));
	}

	public int countChildrenDirectories(String dirId) {
		return resourcesCtx.fetchCount(DIR, getWhereCondition(dirId));
	}

	public List<GymDirectory> getChildrenDirectories(String dirId) {
		return resourcesCtx.select(DIR.ID, DIR.NAME, DIR.LAST_UPDATE).from(DIR).where(getWhereCondition(dirId))
				.orderBy(DIR.NAME.asc()).fetchInto(GymDirectory.class);
	}

	public boolean existDirectory(String dirId) {
		return resourcesCtx.fetchExists(DIR, DIR.ID.eq(dirId).and(DIR.STATUS.eq(ResourceEnum.online.name())));
	}

	public int countExistDirectory(Collection<String> dirIds) {
		return resourcesCtx.fetchCount(DIR, DIR.ID.in(dirIds).and(DIR.STATUS.eq(ResourceEnum.online.name())));
	}

	public GymDirectory getDirectory(String dirId) {
		return resourcesCtx.selectFrom(DIR).where(DIR.ID.eq(dirId).and(DIR.STATUS.eq(ResourceEnum.online.name())))
				.fetchAnyInto(GymDirectory.class);
	}

	public List<GymDirectory> getDirectories(Collection<String> dirIds) {
		return resourcesCtx.selectFrom(DIR).where(DIR.ID.in(dirIds).and(DIR.STATUS.eq(ResourceEnum.online.name())))
				.fetchInto(GymDirectory.class);
	}

	public Integer getDirectoryRank(String dirId) {
		return resourcesCtx.select(DIR.RANK_IDX).from(DIR)
				.where(DIR.ID.eq(dirId).and(DIR.STATUS.eq(ResourceEnum.online.name()))).fetchAnyInto(Integer.class);
	}

	public void addDirectory(String dirId, String name, GymDirectory parentDir, String uid) {
		GymDirectoryRecord directory = new GymDirectoryRecord();
		directory.setId(dirId);
		directory.setName(name);
		if (null == parentDir) {
			directory.setRankIdx(1);
		} else { // 存在父级
			directory.setParentId(parentDir.getId());
			directory.setRankIdx(parentDir.getRankIdx() + 1);
		}
		directory.setStatus(ResourceEnum.online.name());
		directory.setUid(uid);
		long timeMillis = System.currentTimeMillis();
		directory.setCreateTime(timeMillis);
		directory.setLastUpdate(timeMillis);
		resourcesCtx.insertInto(DIR).set(directory).execute();
	}

	public void updateDirectoryName(String dirId, String name, String uid) {
		resourcesCtx.update(DIR).set(DIR.NAME, name).set(DIR.UID, uid).set(DIR.LAST_UPDATE, System.currentTimeMillis())
				.where(DIR.ID.eq(dirId)).execute();
	}

	public void deleteDirectory(String dirId, String uid) {
		resourcesCtx.update(DIR).set(DIR.STATUS, ResourceEnum.delete.name()).set(DIR.UID, uid)
				.set(DIR.LAST_UPDATE, System.currentTimeMillis()).where(DIR.ID.eq(dirId)).execute();
	}

	public void batchInsertDirectories(Collection<GymDirectory> directories) {
		List<GymDirectoryRecord> records = Lists.newArrayList();
		for (GymDirectory directory : directories) {
			records.add(resourcesCtx.newRecord(DIR, directory));
		}
		resourcesCtx.batchInsert(records).execute();
	}
}
