package com.jz.gymchina.jar.resources.cache.ram;

import java.util.List;
import java.util.Map;

import javax.annotation.PostConstruct;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.jz.common.utils.collection.ArrayMapTools;
import com.jz.common.utils.text.StringTools;
import com.jz.gymchina.jar.resources.common.GymchinaThreadPool;
import com.jz.gymchina.jar.resources.service.DirectoryService;
import com.jz.jooq.gymchina.resources.tables.pojos.GymDirectory;

/**
 * @Title RAMDirectoryCache
 * @Package com.jz.gymchina.jar.resources.cache.ram
 * @author tangjunfeng
 * @date 2019年3月12日 下午7:01:48
 * @version V1.0
 */
@Lazy
@Component
public class RAMDirectoryCache {

	@Autowired
	private DirectoryService directoryService;

	private static final List<String> root = Lists.newArrayList();
	private static final Map<String, List<String>> parent2Children = Maps.newHashMap();
	private static final Map<String, String> directoryId2ParentDirectoryId = Maps.newHashMap();

	private void clearCache() {
		root.clear();
		parent2Children.clear();
		directoryId2ParentDirectoryId.clear();
	}

	private <T> void clearPut(List<T> cache, List<T> temp) {
		cache.clear();
		if (ArrayMapTools.isNotEmpty(temp))
			cache.addAll(temp);
	}

	private <K, V> void clearPut(Map<K, V> cache, Map<K, V> temp) {
		cache.clear();
		if (ArrayMapTools.isNotEmpty(temp))
			cache.putAll(temp);
	}

	public void syncRefreshCacheInfo() {
		GymchinaThreadPool.defaultThreadPool.execute(new Runnable() {
			@Override
			public void run() {
				refreshCacheInfo();
			}
		});
	}

	@PostConstruct
	public void refreshCacheInfo() {
		List<GymDirectory> directories = directoryService.getOnlineDirectories();
		if (ArrayMapTools.isEmpty(directories)) {
			this.clearCache();
			return;
		}

		List<String> tmpRoot = Lists.newArrayList();
		Map<String, List<String>> tmpParent2Children = Maps.newHashMap();
		Map<String, String> tmpId2ParentId = Maps.newHashMap();

		for (GymDirectory directory : directories) {
			if (StringTools.isEmptyAndBlank(directory.getParentId())) {
				tmpRoot.add(directory.getId());
			} else {
				if (!tmpParent2Children.containsKey(directory.getParentId()))
					tmpParent2Children.put(directory.getParentId(), Lists.newArrayList());
				tmpParent2Children.get(directory.getParentId()).add(directory.getId());
			}
			tmpId2ParentId.put(directory.getId(), directory.getParentId());
		}

		this.clearPut(root, tmpRoot);
		this.clearPut(parent2Children, tmpParent2Children);
		this.clearPut(directoryId2ParentDirectoryId, tmpId2ParentId);
	}

	public List<String> getPosterities(String directoryId) {
		List<String> posterities = Lists.newArrayList();
		this.getPosterities(directoryId, posterities);
		return posterities;
	}

	private void getPosterities(String directoryId, List<String> posterities) {
		if (!parent2Children.containsKey(directoryId))
			return;
		List<String> children = parent2Children.get(directoryId);
		posterities.addAll(children);
		children.forEach(child -> getPosterities(child, posterities));
	}

	public List<String> getAncestors(String directoryId) {
		List<String> ancestors = Lists.newArrayList();
		this.getAncestors(directoryId, ancestors);
		return ancestors;
	}

	private void getAncestors(String directoryId, List<String> ancestors) {
		if (!directoryId2ParentDirectoryId.containsKey(directoryId))
			return;
		String parentDirectoryId = directoryId2ParentDirectoryId.get(directoryId);
		if (StringTools.isEmptyAndBlank(parentDirectoryId))
			return;
		ancestors.add(parentDirectoryId);
		this.getAncestors(parentDirectoryId, ancestors);
	}

	public String getRoot(String directoryId) {
		if (!directoryId2ParentDirectoryId.containsKey(directoryId))
			return directoryId;
		String parentId = directoryId2ParentDirectoryId.get(directoryId);
		if (StringTools.isEmpty(parentId))
			return directoryId;
		return this.getRoot(parentId);
	}
}
