package com.jz.service;

import java.util.Calendar;

import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import com.aliyuncs.DefaultAcsClient;
import com.aliyuncs.IAcsClient;
import com.aliyuncs.dysmsapi.model.v20170525.SendSmsRequest;
import com.aliyuncs.dysmsapi.model.v20170525.SendSmsResponse;
import com.aliyuncs.http.MethodType;
import com.aliyuncs.profile.DefaultProfile;
import com.aliyuncs.profile.IClientProfile;
import com.jz.common.utils.text.StringTools;
import com.jz.jooq.oss.tables.pojos.SmsVerificationCode;
import com.jz.jooq.oss.tables.records.SmsVerificationCodeRecord;
import com.jz.oss.jar.service.SmsVerificationCodeService;
import com.jz.sms.response.SmsCommonResponse;

/**
 * @Title SmsService
 * @Package com.jz.sms.service
 * @author tangjunfeng
 * @date 2018年8月6日 下午4:16:08
 * @version V1.0
 */
@Service
public class SmsService {

	private static final Logger logger = LoggerFactory.getLogger(SmsService.class);

	// aliyun短信API产品域名（接口地址固定，无需修改）
	private static final String product = "Dysmsapi", domain = "dysmsapi.aliyuncs.com", endpointName = "cn-beijing";
	// 短信有效期（分钟）
	private static final int usefulLifeMinute = 5;
	
	@Value("${aliyun.access.key.id}")
	private String accessKeyId;
	@Value("${aliyun.access.key.secret}")
	private String accessKeySecret;
	@Value("${aliyun.sms.template.code}")
	private String templateCode;
	@Value("${jz.environment}")
	private String env;

	@Autowired
	private SmsVerificationCodeService smsVerificationCodeService;

	/** 发送验证码 */
	public Pair<Boolean, String> sendCode(String phone, String app) throws Exception {
		// 5分钟之后的验证码使用同一个
		SmsVerificationCode code = smsVerificationCodeService.getValidCode(phone, app, System.currentTimeMillis());
		if (null != code) {
			return this.invokeAliyun(phone, code.getCode());
		}

		// 生成一个临时验证码
		String tempCode = StringTools.RandomString.numbers(6);
		Pair<Boolean, String> pair = this.invokeAliyun(phone, tempCode);
		if (!pair.getKey()) // false
			return pair;
		// 发送成功,存储验证码
		Calendar calendar = Calendar.getInstance();
		SmsVerificationCodeRecord codeRecord = new SmsVerificationCodeRecord();
		codeRecord.setPhone(phone);
		codeRecord.setApp(app);
		codeRecord.setCode(tempCode);
		codeRecord.setCreateTime(calendar.getTimeInMillis());
		// 验证码有效期5分钟
		calendar.add(Calendar.MINUTE, usefulLifeMinute);
		codeRecord.setExpiry(calendar.getTimeInMillis());
		smsVerificationCodeService.save(codeRecord);
		return pair;
	}

	private Pair<Boolean, String> invokeAliyun(String phone, String tempCode) throws Exception {
		String msg = this.sendSms(phone, getSignName(), tempCode, templateCode);
		if (null == msg)
			return Pair.of(true, tempCode);
		logger.error("[" + phone + "] send sms error : " + msg);
		return Pair.of(false, msg);
	}

	private String getSignName() {
		return "PlayABC";
	}

	private String sendSms(String phone, String signName, String code, String template) throws Exception {
		// 初始化ascClient,暂时不支持多region（请勿修改）
		IClientProfile profile = DefaultProfile.getProfile(endpointName, accessKeyId, accessKeySecret);
		DefaultProfile.addEndpoint(endpointName, endpointName, product, domain);
		IAcsClient acsClient = new DefaultAcsClient(profile);
		// 组装请求对象
		SendSmsRequest request = new SendSmsRequest();
		// 使用post提交
		request.setMethod(MethodType.POST);
		// 必填:待发送手机号。支持以逗号分隔的形式进行批量调用，批量上限为1000个手机号码,批量调用相对于单条调用及时性稍有延迟,验证码类型的短信推荐使用单条调用的方式；发送国际/港澳台消息时，接收号码格式为00+国际区号+号码，如“0085200000000”
		request.setPhoneNumbers(phone);
		// 必填:短信签名-可在短信控制台中找到
		request.setSignName(signName);
		// 必填:短信模板-可在短信控制台中找到，发送国际/港澳台消息时，请使用国际/港澳台短信模版
		request.setTemplateCode(template);
		// 可选:模板中的变量替换JSON串,如模板内容为"亲爱的${name},您的验证码为${code}"时,此处的值为
		// 友情提示:如果JSON中需要带换行符,请参照标准的JSON协议对换行符的要求,比如短信内容中包含\r\n的情况在JSON中需要表示成\\r\\n,否则会导致JSON在服务端解析失败
		request.setTemplateParam("{\"code\":\"" + code + "\"}");
		// 可选-上行短信扩展码(扩展码字段控制在7位或以下，无特殊需求用户请忽略此字段)
		// request.setSmsUpExtendCode("90997");
		// 可选:outId为提供给业务方扩展字段,最终在短信回执消息中将此值带回给调用者
		// request.setOutId("yourOutId");
		// 请求失败这里会抛ClientException异常
		SendSmsResponse response = acsClient.getAcsResponse(request);
		if (null == response)
			return "系统错误";
		if ("OK".equalsIgnoreCase(response.getCode())) {
			return null;
		}
		return response.getMessage();
	}
	
	/** 校验验证码 */
	public SmsCommonResponse checkCode(String phone, String code, String app) {
		// 永久有效的验证码 测试环境有效
		if ("test".equalsIgnoreCase(env) && Core.forever_valid.equals(code))
			return SmsCommonResponse.ok();
		// 校验验证码
		long timeMillis = System.currentTimeMillis();
		SmsVerificationCode verificationCode = smsVerificationCodeService.getValidCode(phone, app, timeMillis);
		if (null == verificationCode)
			return SmsCommonResponse.failed("验证码已过期");
		if (verificationCode.getCode().equals(code))
			return SmsCommonResponse.ok();
		return SmsCommonResponse.failed("验证码错误");
	}
}
