package com.jz.service;

import java.util.Calendar;

import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import com.jz.beans.AliyunClient;
import com.jz.common.utils.text.StringTools;
import com.jz.jooq.oss.tables.pojos.SmsVerificationCode;
import com.jz.jooq.oss.tables.records.SmsVerificationCodeRecord;
import com.jz.oss.jar.service.SmsVerificationCodeService;
import com.jz.sms.enums.Sign;
import com.jz.sms.response.SmsCommonResponse;

/**
 * @Title SmsService
 * @Package com.jz.sms.service
 * @author tangjunfeng
 * @date 2018年8月6日 下午4:16:08
 * @version V1.0
 */
@Service
public class SmsService {

	private static final Logger logger = LoggerFactory.getLogger(SmsService.class);

	// 短信有效期（分钟）
	private static final int usefulLifeMinute = 5;

	@Value("${aliyun.access.key.id}")
	private String accessKeyId;
	@Value("${aliyun.access.key.secret}")
	private String accessKeySecret;
	@Value("${aliyun.sms.template.code}")
	private String templateCode;
	@Value("${jz.environment}")
	private String env;

	@Autowired
	private SmsVerificationCodeService smsVerificationCodeService;

	/** 发送验证码 */
	public Pair<Boolean, String> sendCode(String phone, String app, Sign sign, String model) throws Exception {
		// 5分钟之后的验证码使用同一个
		SmsVerificationCode code = smsVerificationCodeService.getValidCode(phone, app, model, System.currentTimeMillis());
		if (null != code) {
			// 一分钟一条短信
			if ((System.currentTimeMillis() - code.getLastSend()) / 1000 <= 60)
				return Pair.of(false, "发送太过频繁");
			Pair<Boolean, String> result = this.invokeAliyun(phone, code.getCode(), sign);
			if (!result.getKey()) // false
				return result;
			// 修改最近一次发送时间
			smsVerificationCodeService.updateLastSend(code.getId());
			return result;
		}

		// 生成一个临时验证码
		String tempCode = StringTools.RandomString.numbers(6);
		Pair<Boolean, String> pair = this.invokeAliyun(phone, tempCode, sign);
		if (!pair.getKey()) // false
			return pair;
		// 发送成功,存储验证码
		Calendar calendar = Calendar.getInstance();
		SmsVerificationCodeRecord codeRecord = new SmsVerificationCodeRecord();
		codeRecord.setPhone(phone);
		codeRecord.setApp(app);
		codeRecord.setModel(model);
		codeRecord.setCode(tempCode);
		codeRecord.setLastSend(calendar.getTimeInMillis());
		codeRecord.setCreateTime(calendar.getTimeInMillis());
		// 验证码有效期5分钟
		calendar.add(Calendar.MINUTE, usefulLifeMinute);
		codeRecord.setExpiry(calendar.getTimeInMillis());
		smsVerificationCodeService.save(codeRecord);
		return pair;
	}

	private Pair<Boolean, String> invokeAliyun(String phone, String tempCode, Sign sign) throws Exception {
		AliyunClient aliyunClient = AliyunClient.of(phone, sign.used, templateCode);
		aliyunClient.addTemplateParam("code", tempCode);
		String msg = AliyunInvokeService.sendSms(accessKeyId, accessKeySecret, aliyunClient);
		if (null == msg)
			return Pair.of(true, tempCode);
		logger.error("[" + phone + "] send sms error : " + msg);
		return Pair.of(false, msg);
	}

	/** 校验验证码 */
	public SmsCommonResponse checkCode(String phone, String code, String app, String model) {
		// 永久有效的验证码 测试环境有效
		if ("test".equalsIgnoreCase(env) && Core.forever_valid.equals(code))
			return SmsCommonResponse.ok();
		// 校验验证码
		long timeMillis = System.currentTimeMillis();
		SmsVerificationCode verificationCode = smsVerificationCodeService.getValidCode(phone, app, model, timeMillis);
		if (null == verificationCode)
			return SmsCommonResponse.failed("验证码已过期");
		if (null != verificationCode.getModel() && StringTools.isEmptyAndBlank(model))
			return SmsCommonResponse.failed("无效验证码");
		if (verificationCode.getCode().equals(code))
			return SmsCommonResponse.ok();
		return SmsCommonResponse.failed("验证码错误");
	}
}
