package com.jz.qcloud.tools;

import java.net.URL;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import com.jz.qcloud.beans.QcloudConfig;
import com.qcloud.cos.COSClient;
import com.qcloud.cos.ClientConfig;
import com.qcloud.cos.auth.BasicCOSCredentials;
import com.qcloud.cos.auth.COSCredentials;
import com.qcloud.cos.http.HttpMethodName;
import com.qcloud.cos.http.HttpProtocol;
import com.qcloud.cos.region.Region;

/**
 * @Title 针对于腾讯云对象存获取预签名url
 * @author shaolianjie
 */
public class PresignedUrlTools {

	private QcloudConfig qcloudConfig;

	private PresignedUrlTools() {
	}

	private PresignedUrlTools(QcloudConfig qcloudConfig) {
		super();
		this.qcloudConfig = qcloudConfig;
	}

	public static PresignedUrlTools getInstance(QcloudConfig qcloudConfig) {
		return new PresignedUrlTools(qcloudConfig);
	}

	private COSClient createCOSClient() {
		// 1 初始化用户身份信息（secretId, secretKey）。
		// SECRETID和SECRETKEY请登录访问管理控制台 https://console.cloud.tencent.com/cam/capi
		// 进行查看和管理
		COSCredentials cred = new BasicCOSCredentials(qcloudConfig.getSecretId(), qcloudConfig.getSecretKey());
		// 2 设置 bucket 的地域, COS 地域的简称请参照
		// https://cloud.tencent.com/document/product/436/6224
		// clientConfig 中包含了设置 region, https(默认 http), 超时, 代理等 set 方法, 使用可参见源码或者常见问题
		// Java SDK 部分。
		Region region = new Region(qcloudConfig.getRegion());
		ClientConfig clientConfig = new ClientConfig(region);
		// 这里建议设置使用 https 协议
		// 从 5.6.54 版本开始，默认使用了 https
		clientConfig.setHttpProtocol(HttpProtocol.https);
		// 3 生成 cos 客户端。
		return new COSClient(cred, clientConfig);
	}

	/**
	 * 获取预上传地址
	 * 
	 * @param identity 放到header中的identity
	 * @param basePath cos文件夹
	 * @param fileId   cos文件名不包含后缀
	 * @param ext      cos文件后缀名
	 * @param needPost 是否要使用post请求，否则使用put请求
	 */
	public String genPresignedUrl(String identity, String path, boolean needPost) {
		COSClient cosClient = createCOSClient();
		String key = path;

		// 设置签名过期时间(可选), 若未进行设置则默认使用 ClientConfig 中的签名过期时间(1小时)
		Date expirationDate = new Date(System.currentTimeMillis() + 7200 * 1000);

		// 填写本次请求的参数，需与实际请求相同，能够防止用户篡改此签名的 HTTP 请求的参数
		Map<String, String> params = new HashMap<String, String>();

		// 填写本次请求的头部，需与实际请求相同，能够防止用户篡改此签名的 HTTP 请求的头部
		Map<String, String> headers = new HashMap<String, String>();
		headers.put("identity", identity);

		// 请求的 HTTP 方法，上传请求用 PUT，下载请求用 GET，删除请求用 DELETE
		HttpMethodName method = needPost ? HttpMethodName.POST : HttpMethodName.PUT;

		URL url = cosClient.generatePresignedUrl(qcloudConfig.getBucketName(), key, expirationDate, method, headers,
				params);

		cosClient.shutdown();

		return url.toString();
	}

}