package com.jz.qcloud.tools;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.TreeMap;

import org.apache.commons.lang3.StringUtils;

import com.jz.common.utils.jz.IdTools;
import com.jz.common.utils.text.StringTools;
import com.jz.qcloud.beans.QcloudConfig;
import com.jz.qcloud.beans.TempUploadInfo;
import com.jz.qcloud.beans.UploadConfig;
import com.tencent.cloud.CosStsClient;
import com.tencent.cloud.Response;

/**
 * @Title 提供上传到对象存储的临时凭证
 * @author shaolianjie
 */
public class StsTools {

	private QcloudConfig qcloudConfig;

	private StsTools(QcloudConfig qcloudConfig) {
		this.qcloudConfig = qcloudConfig;
	}

	public static StsTools getInstance(QcloudConfig qcloudConfig) {
		return new StsTools(qcloudConfig);
	}

	/** 获取临时上传token */
	public TempUploadInfo getTemporaryUploadToken(String identity, String clientFileName, UploadConfig uploadConfig)
			throws Exception {
		TreeMap<String, Object> config = new TreeMap<String, Object>();
		// 云 api 密钥 SecretId
		config.put("secretId", qcloudConfig.getSecretId());
		// 云 api 密钥 SecretKey
		config.put("secretKey", qcloudConfig.getSecretKey());

		// 临时密钥有效时长，单位是秒
		config.put("durationSeconds", 7200);

		// 换成你的 bucket
		config.put("bucket", qcloudConfig.getBucketName());
		// 换成 bucket 所在地区
		config.put("region", qcloudConfig.getRegion());

		// 可以通过 allowPrefixes 指定前缀数组, 例子： a.jpg 或者 a/* 或者 * (使用通配符*存在重大安全风险, 请谨慎评估使用)
		config.put("allowPrefixes", new String[] { uploadConfig.getBasePath() });

		// 密钥的权限列表。简单上传和分片需要以下的权限，其他权限列表请看
		// https://cloud.tencent.com/document/product/436/31923
		String[] allowActions = new String[] {
				// 简单上传
				"name/cos:PutObject", "name/cos:PostObject",
				// 分片上传
				"name/cos:InitiateMultipartUpload", "name/cos:ListMultipartUploads", "name/cos:ListParts",
				"name/cos:UploadPart", "name/cos:CompleteMultipartUpload" };
		config.put("allowActions", allowActions);

		Response response = CosStsClient.getCredential(config);

		String fileId = this.getFileId(identity, clientFileName, uploadConfig);
		String path = uploadConfig.getSaveDir();
		if (!path.startsWith(uploadConfig.getBasePath())) {
			path = uploadConfig.getBasePath() + "/" + path;
		}
		if (!path.endsWith("/")) {
			path += "/";
		}
		path += fileId;
		if (StringUtils.isNotEmpty(uploadConfig.getExt())) {
			path += "." + uploadConfig.getExt();
		}

		String host = "https://" + qcloudConfig.getBucketName() + ".cos." + qcloudConfig.getRegion() + ".myqcloud.com";

		TempUploadInfo info = TempUploadInfo.of();
		info.setTmpSecretId(response.credentials.tmpSecretId).setTmpSecretKey(response.credentials.tmpSecretKey)
				.setSessionToken(response.credentials.sessionToken).setHost(host).setCosPath(path)
				.setExpire(response.expiredTime);
		if (StringUtils.isNotEmpty(uploadConfig.getCallback())) {
			info.setCallback(this.getCallbackBody(uploadConfig.getCallbackBody(),
					StringTools.ternary(uploadConfig.getCdnHost(), host), fileId, path, uploadConfig.getData(),
					clientFileName));
		}
		return info;
	}

	private String getFileId(String identity, String clientFileName, UploadConfig tokenConfig) {
		return StringTools.ternary(tokenConfig.getPrefix())
				+ IdTools.encode(identity + StringTools.RandomString.random(10))
				+ StringTools.ternary(tokenConfig.getSuffix());
	}

	private String getCallbackBody(String callbackBody, String host, String fileId, String savePath, String data,
			String clientFileName) throws UnsupportedEncodingException {
		// 拼接host
		if (StringTools.isNotEmptyAndBlank(host)) {
			callbackBody += "&host=" + host;
		}
		callbackBody += "&fileId=" + fileId + "&path=" + URLEncoder.encode(savePath, "UTF-8") + "&fileName="
				+ URLEncoder.encode(clientFileName, "UTF-8");
		callbackBody += "&mimeType=${mimeType}&imageInfo.format=${imageInfo.format}&imgHeight=${imageInfo.height}&imgWidth=${imageInfo.width}";
		callbackBody += "&data=" + StringTools.ternary(data);
		return callbackBody;
	}

}
