package com.jz.jar.oa.repository;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.jooq.Condition;
import org.jooq.UpdateSetMoreStep;
import org.jooq.impl.DSL;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Repository;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.jz.jooq.oa.Tables;
import com.jz.jooq.oa.tables.pojos.User;
import com.jz.jooq.oa.tables.records.UserRecord;

@Lazy
@Repository
public class OAUserRepository extends OABaseRepository {

	private static final com.jz.jooq.oa.tables.User U = Tables.USER;

	public void updateUserStatus(String uid, int status) {
		oaCtx.update(U).set(U.STATUS, status).where(U.UID.eq(uid)).execute();
	}

	public User getUser(String uid) {
		return oaCtx.selectFrom(U).where(U.UID.eq(uid).and(U.STATUS.eq(1))).fetchAnyInto(User.class);
	}

	public User getUserInfo(String uid) {
		return oaCtx.selectFrom(U).where(U.UID.eq(uid)).fetchAnyInto(User.class);
	}

	public List<User> mutGetUsers(Collection<String> uids) {
		return oaCtx.selectFrom(U).where(U.UID.in(uids).and(U.STATUS.eq(1))).fetchInto(User.class);
	}

	public List<User> mutGetAllUsers(Collection<String> uids) {
		return oaCtx.selectFrom(U).where(U.UID.in(uids)).fetchInto(User.class);
	}

	public List<User> multiGetSimpleUser(Collection<String> uids) {
		return oaCtx.select(U.UID, U.CHINESE_NAME, U.ENGLISH_NAME).from(U).where(U.UID.in(uids).and(U.STATUS.eq(1)))
				.fetchInto(User.class);
	}

	public Map<String, Integer> getStaffNumByDeptIds(Collection<String> deptIds) {
		List<Map<String, Object>> list = oaCtx.select(U.DEPT_ID.as("deptId"), DSL.count().as("cnt")).from(U)
				.where(U.DEPT_ID.in(deptIds).and(U.STATUS.eq(1))).groupBy(U.DEPT_ID).fetchMaps();
		Map<String, Integer> rsMap = Maps.newHashMap();
		for (Map<String, Object> map : list) {
			rsMap.put(MapUtils.getString(map, "deptId"), MapUtils.getIntValue(map, "cnt"));
		}
		return rsMap;
	}

	private Condition getConditions(String departmentId, String companyId, String workAddrId, String keyword,
			Collection<String> filterWorkAddrIds, Collection<String> filterCompanyIds, Collection<String> allowUids,
			Collection<String> filterDeptIds, String minJoinDate, String maxJoinDate, Integer status) {
		Condition condition = DSL.trueCondition();
		if (allowUids != null) {
			condition = condition.and(U.UID.in(allowUids));
		}
		if (filterWorkAddrIds != null && filterCompanyIds != null) {
			condition = condition.and(U.WORK_ADDR_ID.in(filterWorkAddrIds)).and(U.COMPANY_ID.in(filterCompanyIds));
		}
		if (StringUtils.isNotEmpty(departmentId)) {
			condition = condition.and(U.DEPT_ID.eq(departmentId));
		}
		if (StringUtils.isNotEmpty(companyId)) {
			condition = condition.and(U.COMPANY_ID.eq(companyId));
		}
		if (StringUtils.isNotEmpty(workAddrId)) {
			condition = condition.and(U.WORK_ADDR_ID.eq(workAddrId));
		}
		if (filterDeptIds != null) {
			condition = condition.and(U.DEPT_ID.in(filterDeptIds));
		}
		if (StringUtils.isNotEmpty(keyword)) {
			condition = condition.and(U.CHINESE_NAME.like("%" + keyword + "%")
					.or(U.ENGLISH_NAME.like("%" + keyword + "%")).or(U.PHONE.like("%" + keyword + "%")));
		}
		if (StringUtils.isNotEmpty(maxJoinDate) && StringUtils.isNotEmpty(minJoinDate)) {
			condition = condition.and(U.JOIN_DATE.between(minJoinDate, maxJoinDate));
		}
		if (status != null) {
			condition = condition.and(U.STATUS.eq(status));
		}
		return condition;
	}

	public int cntUsers(Collection<String> allowUids, Integer status, String departmentId, String companyId,
			String workAddrId, String keyword, Collection<String> filterDeptIds, String minJoinDate,
			String maxJoinDate) {
		return oaCtx.fetchCount(U, this.getConditions(departmentId, companyId, workAddrId, keyword, null, null,
				allowUids, filterDeptIds, minJoinDate, maxJoinDate, status));
	}

	public List<User> filterUsersByConditionsLimit(Collection<String> allowUids, Integer status, String departmentId,
			String companyId, String workAddrId, String keyword, Collection<String> filterDeptIds, String minJoinDate,
			String maxJoinDate, int start, int size) {
		return oaCtx.selectFrom(U)
				.where(this.getConditions(departmentId, companyId, workAddrId, keyword, null, null, allowUids,
						filterDeptIds, minJoinDate, maxJoinDate, status))
				.orderBy(U.ATTENDANCE_ID.desc()).limit(start, size).fetchInto(User.class);
	}

	public List<String> filterUids(String keyword) {
		return oaCtx.select(U.UID).from(U)
				.where(StringUtils.isEmpty(keyword) ? DSL.trueCondition()
						: (U.CHINESE_NAME.like("%" + keyword + "%")).or(U.ENGLISH_NAME.like("%" + keyword + "%")))
				.fetchInto(String.class);
	}

	public void createUser(UserRecord record) {
		oaCtx.insertInto(U).set(record).onDuplicateKeyUpdate().set(record).execute();
	}

	public void updateUser(String fuid, String chineseName, String englishName, String phone, int status,
			String cardType, String cardBak, String nationality, String country, String fertility, String residenceType,
			String residenceAddress, String contactAddress, String landline, String emergencyContact,
			String emergencyPhone, Integer insurance, Integer childFree, Integer trainAgreement, String joinDate,
			Integer isDirect, int attendanceId, String childs, Integer calSalary) {
		oaCtx.update(U).set(U.CHINESE_NAME, chineseName).set(U.ENGLISH_NAME, englishName).set(U.PHONE, phone)
				.set(U.STATUS, status).set(U.CARD_TYPE, cardType).set(U.CARD_BAK, cardBak)
				.set(U.NATIONALITY, nationality).set(U.COUNTRY, country).set(U.FERTILITY, fertility)
				.set(U.RESIDENCE_TYPE, residenceType).set(U.RESIDENCE_ADDRESS, residenceAddress)
				.set(U.CONTACT_ADDRESS, contactAddress).set(U.LANDLINE, landline)
				.set(U.EMERGENCY_CONTACT, emergencyContact).set(U.EMERGENCY_PHONE, emergencyPhone)
				.set(U.INSURANCE, insurance).set(U.CHILD_FREE, childFree).set(U.TRAIN_AGREEMENT, trainAgreement)
				.set(U.JOIN_DATE, joinDate).set(U.IS_DIRECT, isDirect).set(U.ATTENDANCE_ID, attendanceId)
				.set(U.CHILDS, childs).set(U.CAL_SALARY, calSalary).where(U.UID.eq(fuid)).execute();
	}

	public void updateUserPositionInfo(String fuid, String joinChannel, String referrer, String officialJoinDate,
			String contractStart, String contractEnd, String contractType, String contractNature, String deptId,
			String positionId, String superiorId, String relations, String workAddrId, String companyId,
			String laborUnion, String costCenter, String trailStart, String trailEnd, String resignationDate,
			String resignationWay, String internStart, String internEnd, String positionStart, String positionEnd) {
		oaCtx.update(U).set(U.JOIN_CHANNEL, joinChannel).set(U.REFERRER, referrer)
				.set(U.OFFICIAL_JOIN_DATE, officialJoinDate).set(U.CONTRACT_START, contractStart)
				.set(U.CONTRACT_END, contractEnd).set(U.CONTRACT_TYPE, contractType)
				.set(U.CONTRACT_NATURE, contractNature).set(U.DEPT_ID, deptId).set(U.POSITION_ID, positionId)
				.set(U.SUPERIOR_ID, superiorId).set(U.RELATIONS, relations).set(U.WORK_ADDR_ID, workAddrId)
				.set(U.COMPANY_ID, companyId).set(U.LABOR_UNION, laborUnion).set(U.COST_CENTER, costCenter)
				.set(U.TRAIL_START, trailStart).set(U.TRAIL_END, trailEnd).set(U.RESIGNATION_DATE, resignationDate)
				.set(U.RESIGNATION_WAY, resignationWay).set(U.INTERN_START, internStart).set(U.INTERN_END, internEnd)
				.set(U.POSITION_START, positionStart).set(U.POSITION_END, positionEnd).where(U.UID.eq(fuid)).execute();
	}

	public void updateBackgroundSurvey(String fuid, String backgroundSurvey) {
		oaCtx.update(U).set(U.BACKGROUND_SURVEY, backgroundSurvey).where(U.UID.eq(fuid)).execute();
	}

	public void updateUserAccountInfo(String fuid, String bankAccount, String bank, String bankArea, String fund,
			String socialSecurityType, String socialSecurityArea, String tax, Integer annualLeaveDays,
			String certificate, Integer oldLeaveDays, String leaveAdjustDate, Integer calAttendance) {
		UpdateSetMoreStep<UserRecord> moreStep = oaCtx.update(U).set(U.BANK_ACCOUNT, bankAccount).set(U.BANK, bank)
				.set(U.BANK_AREA, bankArea).set(U.FUND, fund).set(U.SOCIAL_SECURITY_TYPE, socialSecurityType)
				.set(U.SOCIAL_SECURITY_AREA, socialSecurityArea).set(U.TAX, tax)
				.set(U.ANNUAL_LEAVE_DAYS, annualLeaveDays).set(U.CERTIFICATE, certificate)
				.set(U.CAL_ATTENDANCE, calAttendance);
		if (StringUtils.isNotEmpty(leaveAdjustDate) && oldLeaveDays != annualLeaveDays) {
			moreStep.set(U.OLD_LEAVE_DAYS, oldLeaveDays).set(U.LEAVE_ADJUST_DATE, leaveAdjustDate);
		}
		moreStep.where(U.UID.eq(fuid)).execute();
	}

	public void backInOffice(String fuid, String joinDate, String officialJoinDate, String contractStart,
			String contractEnd, String contractType, String contractNature) {
		String _null = null;
		oaCtx.update(U).set(U.JOIN_DATE, joinDate).set(U.OFFICIAL_JOIN_DATE, officialJoinDate)
				.set(U.CONTRACT_START, contractStart).set(U.CONTRACT_END, contractEnd)
				.set(U.CONTRACT_TYPE, contractType).set(U.CONTRACT_NATURE, contractNature)
				.set(U.RESIGNATION_DATE, _null).set(U.STATUS, 1).where(U.UID.eq(fuid)).execute();
	}

	public Integer getOldLeaveDays(String fuid) {
		return oaCtx.select(U.ANNUAL_LEAVE_DAYS).from(U).where(U.UID.eq(fuid)).fetchAnyInto(Integer.class);
	}

	public List<String> filterUidsByConditionsLimit(String uid, String companyId, String keyword, int start, int size) {
		return oaCtx.select(U.UID).from(U)
				.where(this.getConditions(null, companyId, null, keyword, null, null, null, null, null, null, null))
				.orderBy(U.JOIN_DATE.asc(), U.UID.desc()).limit(start, size).fetchInto(String.class);
	}

	public int getMaxAttendanceId() {
		return oaCtx.select(DSL.max(U.ATTENDANCE_ID)).from(U).fetchAnyInto(Integer.class);
	}

	public List<String> getExistPhones(Collection<String> phones) {
		return oaCtx.select(U.PHONE).from(U).where(U.PHONE.in(phones)).fetchInto(String.class);
	}

	public List<User> filterSimpleUsers(String uid, Integer status, String departmentId, String companyId,
			String workAddrId, String keyword, Collection<String> filterDeptIds, String minJoinDate,
			String maxJoinDate) {
		return oaCtx.select(U.UID, U.CHINESE_NAME, U.ENGLISH_NAME, U.PHONE).from(U)
				.where(this.getConditions(departmentId, companyId, workAddrId, keyword, null, null, null, filterDeptIds,
						minJoinDate, maxJoinDate, status))
				.fetchInto(User.class);
	}

	public void updateUserBackInfo(String fuid, int haveTeacherCert, String teacherCert) {
		oaCtx.update(U).set(U.HAVE_TEACHER_CERT, haveTeacherCert).set(U.TEACHER_CERT, teacherCert).where(U.UID.eq(fuid))
				.execute();
	}

	public List<String> filterUidsByHrSettings(List<String> allUids,
			List<Pair<String, String>> workAddrid2CompanyIdPair) {
		List<Condition> conditions = Lists.newArrayList();
		for (Pair<String, String> p : workAddrid2CompanyIdPair) {
			conditions.add(U.WORK_ADDR_ID.eq(p.getLeft()).and(U.COMPANY_ID.eq(p.getRight())));
		}
		return oaCtx.select(U.UID).from(U).where(U.UID.in(allUids).and(DSL.or(conditions))).fetchInto(String.class);
	}

	private Condition getConditionsForSalary(String companyId, String workAddrId, String keyword,
			Collection<String> allowUids, Collection<String> filterDeptIds, String minJoinDate, String maxJoinDate,
			Integer status, String today, String currentMonth) {
		Condition condition = ((U.STATUS.eq(1).and(U.JOIN_DATE.le(today)))
				.or(U.STATUS.eq(2).and(U.RESIGNATION_DATE.ge(currentMonth)))).and(U.CAL_SALARY.eq(1));
		if (allowUids != null) {
			condition = condition.and(U.UID.in(allowUids));
		}
		if (StringUtils.isNotEmpty(companyId)) {
			condition = condition.and(U.COMPANY_ID.eq(companyId));
		}
		if (StringUtils.isNotEmpty(workAddrId)) {
			condition = condition.and(U.WORK_ADDR_ID.eq(workAddrId));
		}
		if (filterDeptIds != null) {
			condition = condition.and(U.DEPT_ID.in(filterDeptIds));
		}
		if (StringUtils.isNotEmpty(keyword)) {
			condition = condition.and(U.CHINESE_NAME.like("%" + keyword + "%")
					.or(U.ENGLISH_NAME.like("%" + keyword + "%")).or(U.PHONE.like("%" + keyword + "%")));
		}
		if (StringUtils.isNotEmpty(maxJoinDate) && StringUtils.isNotEmpty(minJoinDate)) {
			condition = condition.and(U.JOIN_DATE.between(minJoinDate, maxJoinDate));
		}
		if (status != null) {
			condition = condition.and(U.STATUS.eq(status));
		}
		return condition;
	}

	public int cntUserForSalarySetting(List<String> allowUids, String companyId, String workAddrId, String keyword,
			List<String> filterDeptIds, Integer status, String minJoinDate, String maxJoinDate, String today,
			String currentMonth) {
		return oaCtx.fetchCount(U, this.getConditionsForSalary(companyId, workAddrId, keyword, allowUids, filterDeptIds,
				minJoinDate, maxJoinDate, status, today, currentMonth));
	}

	public List<User> filterUserForSalarySetting(List<String> allowUids, String companyId, String workAddrId,
			String keyword, List<String> filterDeptIds, Integer status, String minJoinDate, String maxJoinDate,
			String today, String currentMonth, int start, int size) {
		return oaCtx
				.select(U.UID, U.CHINESE_NAME, U.ENGLISH_NAME, U.PHONE, U.DEPT_ID, U.POSITION_ID, U.JOIN_DATE,
						U.CONTRACT_NATURE)
				.from(U)
				.where(this.getConditionsForSalary(companyId, workAddrId, keyword, allowUids, filterDeptIds,
						minJoinDate, maxJoinDate, status, today, currentMonth))
				.orderBy(U.JOIN_DATE.desc()).limit(start, size).fetchInto(User.class);
	}

	public List<String> getUidsByHrSettings(List<Pair<String, String>> workAddrid2CompanyIdPair) {
		List<Condition> conditions = Lists.newArrayList();
		for (Pair<String, String> p : workAddrid2CompanyIdPair) {
			conditions.add(U.WORK_ADDR_ID.eq(p.getLeft()).and(U.COMPANY_ID.eq(p.getRight())));
		}
		return oaCtx.select(U.UID).from(U).where(DSL.or(conditions)).fetchInto(String.class);
	}

	public List<String> getUidsNeedEdit() {
		return oaCtx.select(U.UID).from(U).where(U.COMPANY_ID.isNull()).fetchInto(String.class);
	}

	public List<String> filterUidsForBreakRuleLogs(List<String> allowUids, String companyId, String workAddrId,
			String keyword, List<String> filterDeptIds, Integer status) {
		return oaCtx.select(U.UID).from(U).where(getConditions(null, companyId, workAddrId, keyword, null, null,
				allowUids, filterDeptIds, null, null, status)).fetchInto(String.class);
	}

}
