package com.jz.jar.oa.proxy;

import java.util.Calendar;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.UnaryOperator;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.jz.common.utils.collection.ArrayMapTools;
import com.jz.common.utils.datetime.TimeTools;
import com.jz.common.utils.text.StringTools;
import com.jz.jar.oa.cache.WorkflowIdentifyCache;
import com.jz.jar.oa.enmus.WorkflowType;
import com.jz.jar.oa.finals.OAFinial;
import com.jz.jar.oa.service.OACompanyService;
import com.jz.jar.oa.service.OAPositionService;
import com.jz.jar.oa.service.OAUserService;
import com.jz.jar.oa.service.OAWorkAddressService;
import com.jz.jar.oa.service.WorkflowSettingService;
import com.jz.jar.oa.wrapper.PositionWrapper;
import com.jz.jar.oa.wrapper.WorkflowHandler;
import com.jz.jooq.oa.tables.pojos.Company;
import com.jz.jooq.oa.tables.pojos.User;
import com.jz.jooq.oa.tables.pojos.WorkAddress;
import com.jz.jooq.oa.tables.pojos.WorkflowSetting;

@Lazy
@Service
public class WorkflowProxy {

	@Autowired
	private WorkflowSettingService workflowSettingService;
	@Autowired
	private OACompanyService companyService;
	@Autowired
	private OAUserService userService;
	@Autowired
	private OAUserProxy userProxy;
	@Autowired
	private OAWorkAddressService workAddressService;
	@Autowired
	private OAPositionService positionService;
	@Autowired
	private WorkflowIdentifyCache workflowIdentifyCache;

	/** 获取流程处理人信息 */
	public List<WorkflowHandler> getWorkflowHandlers(String uid, List<String> companyIds, String wfid, String hr,
			UnaryOperator<List<WorkflowSetting>>[] unaryOperators) {
		User user = userService.getUser(uid);
		if (null == user)
			return null;
		// 获取用户信息
		WorkAddress workAddress = workAddressService.getWorkAddress(user.getWorkAddrId());
		if (null == workAddress || StringTools.isEmpty(workAddress.getTimekeeper()))
			throw new NullPointerException("无效的工作地点信息");
		List<Company> companies = companyService.multiGetCompanies(companyIds);
		if (ArrayMapTools.isEmpty(companies))
			throw new NullPointerException("无效的公司信息");
		for (Company c : companies) {
			if (StringTools.isAnyEmpty(c.getCashier(), c.getAccounter(), c.getAssetsKeeperIt(), c.getAssetsKeeperWork(),
					c.getLaw()))
				throw new NullPointerException("公司信息不完整");
		}
		List<WorkflowSetting> settings = getWorkflowSetting(wfid, unaryOperators);
		// 配置节点信息
		Company tempCompany = null;
		List<WorkflowHandler> handlers = Lists.newArrayList();
		for (WorkflowSetting workflowSetting : settings) {
			tempCompany = companies.get(workflowSetting.getCompanyIdx());
			this.addWorkflowHandler(user, tempCompany, workAddress, handlers, workflowSetting, hr);
		}
		
		if (handlers.isEmpty())
			return null;
		Set<String> uids = Sets.newHashSet();
		handlers.forEach(it -> uids.add(it.getUid()));
		Map<String, User> users = userService.mutiGetUserMap(uids);
		if (ArrayMapTools.isEmpty(users))
			return null;
		Set<String> positionIds = users.values().stream().map(User::getPositionId).collect(Collectors.toSet());
		Map<String, PositionWrapper> positions = positionService.mutiGetSimplePositionMap(positionIds);
		if (null == positions)
			positions = Maps.newHashMap();
		List<WorkflowHandler> resultHandlers = Lists.newArrayList(), carbonCopy = Lists.newArrayList();
		int i = 0;
		for (WorkflowHandler handler : handlers) {
			if (StringTools.isEmpty(handler.getUid()) || !users.containsKey(handler.getUid()))
				throw new NullPointerException("Cann't found workflow handler for [" + uid + "], [" + wfid + "], ["
						+ handler.getHandleName() + "]");
			i++;
			User tempUser = users.get(handler.getUid());
			handler.setName(tempUser.getChineseName());
			// 审核节点
			if (handler.getIsAddPosition() && positions.containsKey(tempUser.getPositionId())) {
				String positionName = positions.get(tempUser.getPositionId()).getName();
				handler.setHandleName(handler.getHandleName() + "(" + positionName + ")");
			}
			if (handler.getIsCarbonCopy()) {
				carbonCopy.add(handler);
				continue;
			}
			// 非抄送添加用户执行步骤，* 10 用于加签排序
			// 这个10在 com.jz.oa.service.UserWorkflowService.addSignature 中使用，调整需要两边调整
			handler.setStep(i * 10);
			resultHandlers.add(handler);
		}

		// 手动抄送用户(用于查看所有的流程) 登录账号：19911112222 [放在这块是因为oa.user表中没有这个用户]
		WorkflowSetting firstSetting = settings.get(0);
		firstSetting.setCarbonCopy(1);
		carbonCopy.add(WorkflowHandler.of("oa_admin", firstSetting));

		if (!carbonCopy.isEmpty())
			resultHandlers.addAll(carbonCopy);
		return resultHandlers;
	}
	
	/** 获取流程审核节点配置信息 */
	private List<WorkflowSetting> getWorkflowSetting(String wfid, UnaryOperator<List<WorkflowSetting>> operators[]) {
		// 获取流程
		List<WorkflowSetting> settings = workflowSettingService.getWorkflowSetting(wfid);
		if (ArrayMapTools.isEmpty(settings))
			throw new NullPointerException("无法找到流程配置信息 : [" + wfid + "]");
		if (ArrayMapTools.isEmpty(operators))
			return settings;
		// 动态更新
		for (UnaryOperator<List<WorkflowSetting>> operator : operators) {
			if (null == operator)
				continue;
			settings = operator.apply(settings);
			if (ArrayMapTools.isEmpty(settings))
				throw new NullPointerException("无法找到流程配置信息 : [" + wfid + "]");
		}
		return settings;
	}
	
	/** 设置审核人 */
	private void addWorkflowHandler(User user, Company company, WorkAddress workAddress, List<WorkflowHandler> handlers,
			WorkflowSetting workflowSetting, String hr) {
		if (WorkflowType.superior.equalsName(workflowSetting.getType())) {
			if (workflowSetting.getCarbonCopy() > 0) { // 只能抄送直属上级 --> CEO（中间的）
				List<String> carbonCopyUids = userProxy.getLeadersWithoutCEO(user.getSuperiorId());
				if (ArrayMapTools.isNotEmpty(carbonCopyUids))
					carbonCopyUids.forEach(it -> handlers.add(WorkflowHandler.of(it, workflowSetting)));
				return;
			}
			// 添加上级审核 都是两个人:直属上级+ceo(如果两个人是同一个人，则只有ceo一个人)
			this.addSuperiorHandlers(user, workflowSetting, handlers);
		} else if (WorkflowType.finance.equalsName(workflowSetting.getType())) {
			this.addFinanceHandlers(company.getAccounter(), workflowSetting, handlers);
		} else if (WorkflowType.administration_countersign.equalsName(workflowSetting.getType())) {
			this.addAdmCountersign(workAddress.getTimekeeper(), workflowSetting, handlers);
		} else if (WorkflowType.hr_countersign.equalsName(workflowSetting.getType())) {
			this.addHrCountersign(hr, workflowSetting, handlers);
		} else if (WorkflowType.law.equalsName(workflowSetting.getType())) {
			this.addLawHandler(company.getLaw(), workflowSetting, handlers);
		} else if (WorkflowType.hr.equalsName(workflowSetting.getType())) {
			handlers.add(WorkflowHandler.of(hr, workflowSetting));
		} else if (WorkflowType.cashier.equalsName(workflowSetting.getType())) {
			handlers.add(WorkflowHandler.of(company.getCashier(), workflowSetting));
		} else if (WorkflowType.assets_it.equalsName(workflowSetting.getType())) {
			handlers.add(WorkflowHandler.of(company.getAssetsKeeperIt(), workflowSetting));
		} else if (WorkflowType.assets_work.equalsName(workflowSetting.getType())) {
			handlers.add(WorkflowHandler.of(company.getAssetsKeeperWork(), workflowSetting));
		} else if (WorkflowType.attendancer.equalsName(workflowSetting.getType())) {
			handlers.add(WorkflowHandler.of(workAddress.getTimekeeper(), workflowSetting));
		} else if (WorkflowType.cao.equalsName(workflowSetting.getType())) {
			this.addCaoHandler(workflowSetting, handlers);
		} else if (WorkflowType.ceo.equalsName(workflowSetting.getType())) {
			handlers.add(WorkflowHandler.of(OAFinial.CEO_UID, workflowSetting));
		} else if (WorkflowType.self.equalsName(workflowSetting.getType())) {
			handlers.add(WorkflowHandler.of(user.getUid(), workflowSetting));
		}
	}

	// 添加上级审核人（汇报线的 1 --> CEO - 1/CEO）
	private void addSuperiorHandlers(User user, WorkflowSetting workflowSetting, List<WorkflowHandler> handlers) {
		if (StringTools.isEmptyAndBlank(user.getSuperiorId()))
			return;
		// 直属上级
		handlers.add(WorkflowHandler.of(user.getSuperiorId(), workflowSetting, true));
		if (OAFinial.CEO_UID.equals(user.getSuperiorId()))
			return;
		// 只是第一个审核人拥有扩展信息
		if (workflowSetting.getIsExt() > 0)
			workflowSetting.setIsExt(0);
		List<String> leaders = userProxy.getLeadersWithoutCEO(user.getSuperiorId());
		if (ArrayMapTools.isEmpty(leaders))
			return;
		for (String superior : leaders) {
			handlers.add(WorkflowHandler.of(superior, workflowSetting, true));
		}
	}

	// 添加财务审核人（所属公司的会计 --> CEO - 1）
	private void addFinanceHandlers(String accounter, WorkflowSetting workflowSetting, List<WorkflowHandler> handlers) {
		handlers.add(WorkflowHandler.of(accounter, workflowSetting, true));
		List<String> financeLeaders = userProxy.getLeadersWithoutCEO(accounter);
		if (ArrayMapTools.isEmpty(financeLeaders))
			return;
		// 只是第一个审核人拥有扩展信息
		if (workflowSetting.getIsExt() > 0)
			workflowSetting.setIsExt(0);
		for (String leader : financeLeaders)
			handlers.add(WorkflowHandler.of(leader, workflowSetting, true));
	}

	// 添加行政会签
	private void addAdmCountersign(String adm, WorkflowSetting workflowSetting, List<WorkflowHandler> handlers) {
		handlers.add(WorkflowHandler.of(adm, workflowSetting, true));
		this.addCaoHandler(workflowSetting, handlers);
	}

	// 添加行政总监
	private void addCaoHandler(WorkflowSetting workflowSetting, List<WorkflowHandler> handlers) {
		User cao = userService.getUserByPosition(OAFinial.CAO_PID);
		if (null == cao)
			return;
		handlers.add(WorkflowHandler.of(cao.getUid(), workflowSetting, true));
	}

	// 添加人事会签
	private void addHrCountersign(String hrId, WorkflowSetting workflowSetting, List<WorkflowHandler> handlers) {
		handlers.add(WorkflowHandler.of(hrId, workflowSetting, true));
		User cho = userService.getUserByPosition(OAFinial.CHO_PID);
		if (null == cho)
			return;
		handlers.add(WorkflowHandler.of(cho.getUid(), workflowSetting, true));
	}

	// 添加法务审核
	private void addLawHandler(String lawId, WorkflowSetting workflowSetting, List<WorkflowHandler> handlers) {
		handlers.add(WorkflowHandler.of(lawId, workflowSetting, true));
		User lawUser = userService.getSimpleUser(lawId);
		if (null == lawUser || StringTools.isEmptyAndBlank(lawUser.getSuperiorId()))
			return;
		handlers.add(WorkflowHandler.of(lawUser.getSuperiorId(), workflowSetting, true));
	}

	/** 获取流程id */
	public String getIdentify() {
		long timeInMillis = Calendar.getInstance().getTimeInMillis();
		String today = TimeTools.getDateTime(timeInMillis, "yyyyMMddHHmmssSSS");
		String identity = workflowIdentifyCache.getWorkflowIdentity();
		return today + this.append(identity, 25 - today.length());
	}

	private String append(String a, int length) {
		if (a.length() >= length)
			return a;
		return append("0" + a, length);
	}
}
