package com.jz.jar.media.repository;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.jooq.Condition;
import org.jooq.TableOnConditionStep;
import org.jooq.impl.DSL;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Repository;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.jz.common.utils.collection.ArrayMapTools;
import com.jz.jar.media.enums.DbField;
import com.jz.jooq.media.Tables;
import com.jz.jooq.media.tables.pojos.PlaylistRelation;

@Lazy
@Repository
public class PlaylistRelationRepository extends MediaBaseRepository {

	private static final com.jz.jooq.media.tables.PlaylistRelation PR = Tables.PLAYLIST_RELATION;
	private static final com.jz.jooq.media.tables.Playlist PL = Tables.PLAYLIST;

	/** 获取级别的所有分类 */
	public List<String> findTopicsForLevel(String lid) {
		return mediaCtx.selectDistinct(PR.CID).from(PR).where(PR.LID.eq(lid)).fetchInto(String.class);
	}

	/** 获取当前专辑所有的cid */
	public List<String> getTopicsForPlaylist(String pid) {
		return mediaCtx.selectDistinct(PR.CID).from(PR).where(PR.PID.eq(pid)).fetchInto(String.class);
	}
	
	private TableOnConditionStep getOnlinePlaylistTable() {
		return PR.join(PL).on(PR.PID.eq(PL.PID));
	}
	
	private Condition getOnlinePlaylistCondition(String lid, Collection<String> cidArray) {
		return PR.LID.eq(lid).and(PR.CID.in(cidArray)).and(PL.STATUS.eq(DbField.online.name()));
	}
	
	/** 获取当前级别及分类下所有的专辑数量 */
	public int countPlaylistForLevelAndTopic(String lid, Collection<String> cidArray) {
		return mediaCtx.select(DSL.countDistinct(PR.PID)).from(getOnlinePlaylistTable())
				.where(getOnlinePlaylistCondition(lid, cidArray)).fetchAnyInto(int.class);
	}

	/** 获取当前级别及分类下所有的专辑id */
	public List<String> findPlaylistForLevelAndTopic(String lid, Collection<String> cidArray, int start, int size) {
		return mediaCtx.selectDistinct(PR.PID).from(getOnlinePlaylistTable())
				.where(getOnlinePlaylistCondition(lid, cidArray)).orderBy(PR.PID.asc()).limit(start, size)
				.fetchInto(String.class);
	}
	
	/** 获取当前级别及分类下所有的专辑id */
	public List<String> findPlaylistForLevelAndTopic(String lid, String cid) {
		return mediaCtx.selectDistinct(PR.PID).from(getOnlinePlaylistTable())
				.where(getOnlinePlaylistCondition(lid, Arrays.asList(cid))).orderBy(PR.PID.asc())
				.fetchInto(String.class);
	}
	
	/********************************************************************/
	/**************************** 专项练习 *******************************/
	/********************************************************************/
	
	/** 获取专项练习的分类id */
	public List<String> getSpecialTopics(String lid) {
		return mediaCtx.selectDistinct(PR.CID).from(PR).where(PR.LID.eq(lid).and(PR.IS_SPECIAL.eq(1)))
				.fetchInto(String.class);
	}
	
	public boolean isSpecial(String lid, String pid) {
		return mediaCtx.fetchExists(PR, PR.PID.eq(pid).and(PR.LID.eq(lid)).and(PR.IS_SPECIAL.eq(1)));
	}
	
	/** 获取专项练习的专辑数量 */
	public int countSpecialPlaylist(String lid, String cid) {
		return mediaCtx.fetchCount(PR, PR.LID.eq(lid).and(PR.CID.eq(cid)).and(PR.IS_SPECIAL.eq(1)));
	}
	
	/** 获取专项练习的分类及专辑id */
	public Map<String, List<String>> findSpecialTopicToPids(String lid) {
		List<PlaylistRelation> relations = mediaCtx.select(PR.CID, PR.PID).from(PR)
				.where(PR.LID.eq(lid).and(PR.IS_SPECIAL.eq(1))).orderBy(PR.SPECIAL_TIME.asc())
				.fetchInto(PlaylistRelation.class);
		if (ArrayMapTools.isEmpty(relations))
			return null;
		Map<String, List<String>> topicToPids = Maps.newHashMap();
		relations.forEach(it -> {
			if (!topicToPids.containsKey(it.getCid()))
				topicToPids.put(it.getCid(), Lists.newArrayList());
			topicToPids.get(it.getCid()).add(it.getPid());
		});
		return topicToPids;
	}
	
	/** 根据级别和专辑id获取分类 */
	public List<String> findPlaylistTopic(String lid, Collection<String> pidArray) {
		return mediaCtx.selectDistinct(PR.CID).from(PR).where(PR.PID.in(pidArray).and(PR.LID.eq(lid))).fetchInto(String.class);
	}
	
	/** 根据级别和分类列表获取最大播放数的专辑id */
	public Map<String, Collection<String>> findMaxWatchCntPlaylistForCids(String lid, Collection<String> cidArray, int cnt) {
		List<PlaylistRelation> relations = mediaCtx.select(PR.PID, PR.CID).from(getOnlinePlaylistTable())
				.where(getOnlinePlaylistCondition(lid, cidArray)).orderBy(PR.WATCH_CNT.desc())
				.fetchInto(PlaylistRelation.class);
		if (ArrayMapTools.isEmpty(relations))
			return Maps.newHashMap();
		Map<String, Collection<String>> has = Maps.newHashMap();
		relations.forEach(it -> {
			if (!has.containsKey(it.getCid())) {
				has.put(it.getCid(), Lists.newArrayList());
			}
			if (has.get(it.getCid()).size() < cnt) {
				has.get(it.getCid()).add(it.getPid());
			}
		});
		return has;
	}
	
	/** 根据级别和专辑id获取分类 */
	public Map<String, String> findPlaylistTopicForPidAndLevel(String lid, Collection<String> pidArray) {
		List<PlaylistRelation> relations = mediaCtx.select(PR.PID, PR.CID).from(PR)
				.where(PR.PID.in(pidArray).and(PR.LID.eq(lid))).fetchInto(PlaylistRelation.class);
		if (ArrayMapTools.isEmpty(relations))
			return Maps.newHashMap();
		Map<String, String> has = Maps.newHashMap();
		relations.forEach(it -> has.put(it.getPid(), it.getCid()));
		return has;
	}
	
	/********************************************************************/
	/**************************** 首页展示 *******************************/
	/********************************************************************/
	
	/** 根据级别和分类获取需要在首页展示的专辑id */
	public List<PlaylistRelation> findHomeShowPlaylistForLevelAndTopics(String lid, Collection<String> cids) {
		return mediaCtx.select(PR.PID, PR.CID).from(PR).where(PR.LID.eq(lid).and(PR.CID.in(cids)).and(PR.HOME_SHOW.eq(1)))
				.orderBy(PR.SEQ.asc()).fetchInto(PlaylistRelation.class);
	}
}
