package com.jz.jar.media.repository;

import java.util.Collection;
import java.util.List;

import org.jooq.Condition;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Repository;

import com.jz.jar.media.enums.AuditStatus;
import com.jz.jar.media.enums.BrandEnum;
import com.jz.jooq.media.Tables;
import com.jz.jooq.media.tables.pojos.WorksInfo;
import com.jz.jooq.media.tables.records.WorksInfoRecord;

@Lazy
@Repository
public class WorksInfoRepository extends MediaBaseRepository implements BaseConditionRepository<WorksInfoRecord> {

	private static final com.jz.jooq.media.tables.WorksInfo WI = Tables.WORKS_INFO;
	private static final com.jz.jooq.media.tables.WorksCity WC = Tables.WORKS_CITY;
	private static final com.jz.jooq.media.tables.WorksAge WA = Tables.WORKS_AGE;

	/********************************************************************/
	/**************************** condition *****************************/
	/********************************************************************/

	private Condition getOpenOnlineCondition(BrandEnum brand) {
		return WI.BRAND.eq(brand.name()).and(WI.STATUS.eq(AuditStatus.online.getCode())).and(WI.IS_OPEN.ge(1));
	}

	private Condition getOpenForTopicCondition(BrandEnum brand, String topicId) {
		return getOpenOnlineCondition(brand).and(WI.TID.eq(topicId));
	}

	private Condition getRecommonCondition(BrandEnum brand) {
		return getOpenOnlineCondition(brand).and(WI.IS_RECOMM.eq(1));
	}

	/********************************************************************/
	/**************************** works info ****************************/
	/********************************************************************/

	public boolean isExistWorksInfo(String id, BrandEnum brand) {
		return mediaCtx.fetchExists(WI, WI.ID.eq(id).and(WI.BRAND.eq(brand.name())));
	}

	public WorksInfo getWorksInfo(String id, BrandEnum brand) {
		return mediaCtx.selectFrom(WI).where(WI.ID.eq(id).and(WI.BRAND.eq(brand.name()))).fetchAnyInto(WorksInfo.class);
	}

	public WorksInfo getSimpleWorksInfo(String id, BrandEnum brand) {
		return mediaCtx.select(WI.ID, WI.SCHOOL_ID, WI.LESSON_ID, WI.PUID, WI.SUID, WI.IS_OPEN, WI.IS_COLLECTIVE)
				.from(WI).where(WI.ID.eq(id).and(WI.BRAND.eq(brand.name()))).fetchAnyInto(WorksInfo.class);
	}

	public List<WorksInfo> findSimpleWorksInfo(Collection<String> artIds) {
		return mediaCtx.selectFrom(WI).where(WI.ID.in(artIds)).fetchInto(WorksInfo.class);
	}

	public void updateOpenInfo(String id, BrandEnum brand, int isOpen, int addLikeCnt) {
		mediaCtx.update(WI).set(WI.IS_OPEN, isOpen).set(WI.LIKE_CNT, WI.LIKE_CNT.add(addLikeCnt))
				.set(WI.LAST_UPDATE, System.currentTimeMillis()).where(WI.ID.eq(id).and(WI.BRAND.eq(brand.name())))
				.execute();
	}

	public int countOpenNewWorks(BrandEnum brand, Collection<String> workIds, long timestamp) {
		return mediaCtx.fetchCount(WI,
				WI.ID.in(workIds).and(getOpenOnlineCondition(brand)).and(WI.CREATE_TIME.gt(timestamp)));
	}

	public int countRecommonWorks(BrandEnum brand) {
		return mediaCtx.fetchCount(WI, getRecommonCondition(brand));
	}

	public List<String> findRecommonWorksId(BrandEnum brand, int start, int size) {
		return mediaCtx.select(WI.ID).from(WI).where(getRecommonCondition(brand))
				.orderBy(WI.RECOMM_SEQ.desc(), WI.CREATE_TIME.desc()).limit(start, size).fetchInto(String.class);
	}

	public int countOpenWorksForTopic(BrandEnum brand, String topicId) {
		return mediaCtx.fetchCount(WI, getOpenForTopicCondition(brand, topicId));
	}

	public List<String> getOpenWorkIdsForTopic(BrandEnum brand, String tid, boolean isHot, int start, int size) {
		return mediaCtx.select(WI.ID).from(WI).where(getOpenForTopicCondition(brand, tid))
				.orderBy(isHot ? WI.LIKE_CNT.desc() : WI.LAST_UPDATE.desc(), WI.CREATE_TIME.desc()).limit(start, size)
				.fetchInto(String.class);
	}

	public void addLikeCnt(String id, BrandEnum brand, int addLikeCnt) {
		mediaCtx.update(WI).set(WI.LIKE_CNT, WI.LIKE_CNT.add(addLikeCnt))
				.where(WI.ID.eq(id).and(WI.BRAND.eq(brand.name()))).execute();
	}

	private Condition getFilterOpenWorkCondition(BrandEnum brand, Collection<String> wids) {
		return WI.ID.in(wids).and(getOpenOnlineCondition(brand));
	}

	public List<String> filterOpenWorkIds(BrandEnum brand, Collection<String> wids, boolean isHot, int start,
			int size) {
		return mediaCtx.select(WI.ID).from(WI).where(getFilterOpenWorkCondition(brand, wids))
				.orderBy(isHot ? WI.LIKE_CNT.desc() : WI.LAST_UPDATE.desc(), WI.CREATE_TIME.desc()).limit(start, size)
				.fetchInto(String.class);
	}

	public int countFilterOpenWorkIds(BrandEnum brand, Collection<String> wids) {
		return mediaCtx.fetchCount(WI, getFilterOpenWorkCondition(brand, wids));
	}

	public WorksInfo getWorksInfo(String schoolId, String lessonId, String suid) {
		return mediaCtx.selectFrom(WI)
				.where(WI.SCHOOL_ID.eq(schoolId).and(WI.LESSON_ID.eq(lessonId)).and(WI.SUID.eq(suid)))
				.orderBy(WI.STATUS.asc()).limit(1).fetchAnyInto(WorksInfo.class);
	}

	public void createWorkInfo(String id, String brandId, String schoolId, String lessonId, String puid, String suid,
			String title, String appraise, String pic, String pictures, String audio, Integer audioDuration, String wid,
			String material, String tid, int isOpen, int status, String operator, int isCollective, int selfUpload) {
		long current = System.currentTimeMillis();
		mediaCtx.insertInto(WI, WI.ID, WI.BRAND, WI.SCHOOL_ID, WI.LESSON_ID, WI.PUID, WI.SUID, WI.TITLE, WI.APPRAISE,
				WI.PIC, WI.PICTURES, WI.AUDIO, WI.AUDIO_DURATION, WI.WID, WI.MATERIAL, WI.TID, WI.IS_OPEN, WI.STATUS,
				WI.LAST_UPDATE, WI.CREATE_TIME, WI.OPERATER, WI.IS_COLLECTIVE, WI.SELF_UPLOAD)
				.values(id, brandId, schoolId, lessonId, puid, suid, title, appraise, pic, pictures, audio,
						audioDuration, wid, material, tid, isOpen, status, current, current, operator, isCollective,
						selfUpload)
				.onDuplicateKeyIgnore().execute();
	}

	public void updateWorkInfo(String id, String schoolId, String lessonId, String title, String appraise, String pic,
			String pictures, String audio, Integer audioDuration, String wid, int status, String rejectReason) {
		mediaCtx.update(WI).set(WI.TITLE, title).set(WI.APPRAISE, appraise).set(WI.PIC, pic).set(WI.PICTURES, pictures)
				.set(WI.AUDIO, audio).set(WI.AUDIO_DURATION, audioDuration).set(WI.WID, wid).set(WI.STATUS, status)
				.set(WI.REJECT_REASON, rejectReason).set(WI.LAST_UPDATE, System.currentTimeMillis())
				.where(WI.ID.eq(id).and(WI.SCHOOL_ID.eq(schoolId)).and(WI.LESSON_ID.eq(lessonId))).execute();
	}

	public void updateWorkInfoPic(String id, String pic, String pictures) {
		mediaCtx.update(WI).set(WI.PIC, pic).set(WI.PICTURES, pictures).set(WI.LAST_UPDATE, System.currentTimeMillis())
				.where(WI.ID.eq(id)).execute();
	}

	public int countWorksCity(String city) {
		return mediaCtx.fetchCount(WC, WC.CITY.eq(city));
	}

	public List<String> findWorksCityIds(String city, int start, int size) {
		return mediaCtx.select(WC.WORK_ID).from(WC).where(WC.CITY.eq(city)).orderBy(WC.CREATE_TIME.desc())
				.limit(start, size).fetchInto(String.class);
	}

	public int countWorksAge(int age) {
		return mediaCtx.fetchCount(WA, WA.AGE.eq(age));
	}

	public List<String> findWorksAgeIds(int age, int start, int size) {
		return mediaCtx.select(WA.WORK_ID).from(WA).where(WA.AGE.eq(age)).orderBy(WA.CREATE_TIME.desc())
				.limit(start, size).fetchInto(String.class);
	}
}
