package com.jz.jar.media.repository;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.jooq.Condition;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Repository;

import com.jz.common.utils.text.StringTools;
import com.jz.jar.media.enums.BrandEnum;
import com.jz.jooq.media.Tables;
import com.jz.jooq.media.tables.pojos.WorksAuthor;
import com.jz.jooq.media.tables.records.WorksAuthorRecord;

@Lazy
@Repository
public class WorksAuthorRepository extends MediaBaseRepository implements BaseConditionRepository<WorksAuthorRecord> {

	private static final com.jz.jooq.media.tables.WorksAuthor WA = Tables.WORKS_AUTHOR;

	private Condition getWorkIdCondition(BrandEnum brand, String workId) {
		return WA.WORK_ID.eq(workId).and(WA.BRAND.eq(brand.name()));
	}

	public List<String> getWorksAuthors(BrandEnum brand, String workId) {
		return mediaCtx.select(WA.SUID).from(WA).where(getWorkIdCondition(brand, workId)).fetchInto(String.class);
	}

	public List<String> getWorksAuthorParents(BrandEnum brand, String workId) {
		return mediaCtx.select(WA.PUID).from(WA).where(getWorkIdCondition(brand, workId)).fetchInto(String.class);
	}

	public boolean isWorksAuthor(BrandEnum brand, String workId, String suid) {
		return mediaCtx.fetchExists(WA, getWorkIdCondition(brand, workId).and(WA.SUID.eq(suid)));
	}

	public boolean isWorksAuthorParent(BrandEnum brand, String workId, String puid) {
		return mediaCtx.fetchExists(WA, getWorkIdCondition(brand, workId).and(WA.PUID.eq(puid)));
	}

	public String getWorksAuthor(BrandEnum brand, String workId) {
		return mediaCtx.select(WA.SUID).from(WA).where(getWorkIdCondition(brand, workId)).fetchAnyInto(String.class);
	}

	public String getWorksAuthorParent(BrandEnum brand, String workId) {
		return mediaCtx.select(WA.PUID).from(WA).where(getWorkIdCondition(brand, workId)).fetchAnyInto(String.class);
	}

	public List<WorksAuthor> findWorkAuthor(BrandEnum brand, Collection<String> workIds) {
		return mediaCtx.selectFrom(WA).where(WA.WORK_ID.in(workIds).and(WA.BRAND.eq(brand.name())))
				.fetchInto(WorksAuthor.class);
	}

	/** 学员作品列表 */
	private Condition getStudentCondition(BrandEnum brand, Collection<String> puids, String suid, Integer isOpen) {
		Condition condition = WA.BRAND.eq(brand.name()).and(WA.PUID.in(puids));
		if (StringTools.isNotEmptyAndBlank(suid))
			condition = condition.and(WA.SUID.eq(suid));
		if (null != isOpen)
			condition = condition.and(WA.IS_OPEN.eq(isOpen));
		return condition;
	}

	public int countStudentWorks(BrandEnum brand, Collection<String> puids, String suid, Integer isOpen) {
		return mediaCtx.fetchCount(WA, getStudentCondition(brand, puids, suid, isOpen));
	}

	public List<String> findStudentWorksId(BrandEnum brand, Collection<String> puids, String suid, Integer isOpen,
			int start, int size) {
		return mediaCtx.select(WA.WORK_ID).from(WA).where(getStudentCondition(brand, puids, suid, isOpen))
				.orderBy(WA.ID.desc()).limit(start, size).fetchInto(String.class);
	}

	public List<String> findAllStudentWorksId(BrandEnum brand, Collection<String> puids, String suid, Integer isOpen) {
		return mediaCtx.select(WA.WORK_ID).from(WA).where(getStudentCondition(brand, puids, suid, isOpen))
				.fetchInto(String.class);
	}

	/** 获取存在作品的学校及课程 */
	public List<WorksAuthor> findExistWorksInfo(Map<String, Collection<String>> school2LessonIds, String suid) {
		Condition condition = getOrWhereCondition(WA.SCHOOL_ID, WA.LESSON_ID, school2LessonIds);
		return mediaCtx.select(WA.SCHOOL_ID, WA.LESSON_ID, WA.WORK_ID, WA.IS_COLLECTIVE).from(WA)
				.where(condition.and(WA.SUID.eq(suid))).fetchInto(WorksAuthor.class);
	}

	public void updateOpenInfo(String workId, int isOpen) {
		mediaCtx.update(WA).set(WA.IS_OPEN, isOpen).where(WA.WORK_ID.eq(workId)).execute();
	}

	public void createWorkAuthor(String brandId, String id, String schoolId, String lessonId, String puid, String suid,
			int isCollective) {
		mediaCtx.insertInto(WA, WA.BRAND, WA.WORK_ID, WA.SCHOOL_ID, WA.LESSON_ID, WA.PUID, WA.SUID, WA.IS_COLLECTIVE)
				.values(brandId, id, schoolId, lessonId, puid, suid, isCollective).onDuplicateKeyIgnore().execute();
	}
}
