package com.jz.jar.media.repository;

import java.util.Collection;
import java.util.List;

import org.jooq.Condition;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Repository;

import com.jz.jar.media.enums.AuditStatus;
import com.jz.jar.media.enums.BrandEnum;
import com.jz.jooq.media.Tables;
import com.jz.jooq.media.tables.pojos.WorksInfo;
import com.jz.jooq.media.tables.records.WorksInfoRecord;

@Lazy
@Repository
public class WorksInfoRepository extends MediaBaseRepository implements BaseConditionRepository<WorksInfoRecord> {

	private static final com.jz.jooq.media.tables.WorksInfo WI = Tables.WORKS_INFO;

	/********************************************************************/
	/**************************** condition *****************************/
	/********************************************************************/

	private Condition getOpenOnlineCondition(BrandEnum brand) {
		return WI.BRAND.eq(brand.name()).and(WI.STATUS.eq(AuditStatus.online.getCode())).and(WI.IS_OPEN.ge(1));
	}

	private Condition getOpenForTopicCondition(BrandEnum brand, String topicId) {
		return getOpenOnlineCondition(brand).and(WI.TID.eq(topicId));
	}

	private Condition getRecommonCondition(BrandEnum brand) {
		return getOpenOnlineCondition(brand).and(WI.IS_RECOMM.eq(1));
	}

	/********************************************************************/
	/**************************** works info ****************************/
	/********************************************************************/

	public boolean isExistWorksInfo(String id, BrandEnum brand) {
		return mediaCtx.fetchExists(WI, WI.ID.eq(id).and(WI.BRAND.eq(brand.name())));
	}

	public WorksInfo getWorksInfo(String id, BrandEnum brand) {
		return mediaCtx.selectFrom(WI).where(WI.ID.eq(id).and(WI.BRAND.eq(brand.name()))).fetchAnyInto(WorksInfo.class);
	}

	public WorksInfo getSimpleWorksInfo(String id, BrandEnum brand) {
		return mediaCtx.select(WI.ID, WI.SCHOOL_ID, WI.LESSON_ID, WI.PUID, WI.SUID, WI.IS_OPEN, WI.IS_COLLECTIVE)
				.from(WI).where(WI.ID.eq(id).and(WI.BRAND.eq(brand.name()))).fetchAnyInto(WorksInfo.class);
	}

	public List<WorksInfo> findSimpleWorksInfo(Collection<String> artIds) {
		return mediaCtx.select(WI.ID, WI.PUID, WI.SUID, WI.TITLE, WI.PIC, WI.IS_OPEN, WI.IS_COLLECTIVE).from(WI)
				.where(WI.ID.in(artIds)).fetchInto(WorksInfo.class);
	}

	public void updateOpenInfo(String id, BrandEnum brand, int isOpen, int addLikeCnt) {
		mediaCtx.update(WI).set(WI.IS_OPEN, isOpen).set(WI.LIKE_CNT, WI.LIKE_CNT.add(addLikeCnt))
				.set(WI.LAST_UPDATE, System.currentTimeMillis()).where(WI.ID.eq(id).and(WI.BRAND.eq(brand.name())))
				.execute();
	}

	public int countOpenNewWorks(BrandEnum brand, Collection<String> workIds, long timestamp) {
		return mediaCtx.fetchCount(WI, WI.ID.in(workIds).and(getOpenOnlineCondition(brand)).and(WI.CREATE_TIME.gt(timestamp)));
	}

	public int countRecommonWorks(BrandEnum brand) {
		return mediaCtx.fetchCount(WI, getRecommonCondition(brand));
	}

	public List<String> findRecommonWorksId(BrandEnum brand, int start, int size) {
		return mediaCtx.select(WI.ID).from(WI).where(getRecommonCondition(brand))
				.orderBy(WI.RECOMM_SEQ.desc(), WI.CREATE_TIME.desc()).limit(start, size).fetchInto(String.class);
	}

	public int countOpenWorksForTopic(BrandEnum brand, String topicId) {
		return mediaCtx.fetchCount(WI, getOpenForTopicCondition(brand, topicId));
	}

	public List<String> getOpenWorkIdsForTopic(BrandEnum brand, String tid, boolean isHot, int start, int size) {
		return mediaCtx.select(WI.ID).from(WI).where(getOpenForTopicCondition(brand, tid))
				.orderBy(isHot ? WI.LIKE_CNT.desc() : WI.LAST_UPDATE.desc(), WI.CREATE_TIME.desc()).limit(start, size)
				.fetchInto(String.class);
	}

	public void addLikeCnt(String id, BrandEnum brand, int addLikeCnt) {
		mediaCtx.update(WI).set(WI.LIKE_CNT, WI.LIKE_CNT.add(addLikeCnt))
				.where(WI.ID.eq(id).and(WI.BRAND.eq(brand.name()))).execute();
	}

	private Condition getFilterOpenWorkCondition(BrandEnum brand, Collection<String> wids) {
		return WI.ID.in(wids).and(getOpenOnlineCondition(brand));
	}

	public List<String> filterOpenWorkIds(BrandEnum brand, Collection<String> wids, boolean isHot, int start,
			int size) {
		return mediaCtx.select(WI.ID).from(WI).where(getFilterOpenWorkCondition(brand, wids))
				.orderBy(isHot ? WI.LIKE_CNT.desc() : WI.LAST_UPDATE.desc(), WI.CREATE_TIME.desc()).limit(start, size)
				.fetchInto(String.class);
	}

	public int countFilterOpenWorkIds(BrandEnum brand, Collection<String> wids) {
		return mediaCtx.fetchCount(WI, getFilterOpenWorkCondition(brand, wids));
	}
}
