package com.jz.jar.media.repository;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.jooq.Condition;
import org.jooq.TableOnConditionStep;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Repository;

import com.jz.common.utils.text.StringTools;
import com.jz.jar.media.enums.AuditStatus;
import com.jz.jar.media.enums.BrandEnum;
import com.jz.jooq.media.Tables;
import com.jz.jooq.media.tables.pojos.WorksInfo;
import com.jz.jooq.media.tables.records.WorksInfoRecord;

@Lazy
@Repository
public class WorksInfoRepository extends MediaBaseRepository implements BaseConditionRepository<WorksInfoRecord> {

	private static final com.jz.jooq.media.tables.WorksInfo WI = Tables.WORKS_INFO;
	private static final com.jz.jooq.media.tables.WorksThemeRelation WTR = Tables.WORKS_THEME_RELATION;

	public boolean existWorksInfo(String id, BrandEnum brand) {
		return mediaCtx.fetchExists(WI, WI.ID.eq(id).and(WI.BRAND.eq(brand.name())));
	}

	public WorksInfo getWorksInfo(String id, BrandEnum brand) {
		return mediaCtx.selectFrom(WI).where(WI.ID.eq(id).and(WI.BRAND.eq(brand.name()))).fetchAnyInto(WorksInfo.class);
	}

	public WorksInfo getSimpleWorksInfo(String id, BrandEnum brand) {
		return mediaCtx.select(WI.ID, WI.SCHOOL_ID, WI.LESSON_ID, WI.PUID, WI.SUID, WI.IS_OPEN).from(WI)
				.where(WI.ID.eq(id).and(WI.BRAND.eq(brand.name()))).fetchAnyInto(WorksInfo.class);
	}

	public List<WorksInfo> findSimpleWorksInfo(Collection<String> artIds) {
		return mediaCtx.select(WI.ID, WI.PUID, WI.RESOURCE, WI.IS_OPEN).from(WI).where(WI.ID.in(artIds))
				.fetchInto(WorksInfo.class);
	}

	public void updateOpenInfo(String id, BrandEnum brand, int isOpen) {
		mediaCtx.update(WI).set(WI.IS_OPEN, isOpen).set(WI.LAST_UPDATE, System.currentTimeMillis())
				.where(WI.ID.eq(id).and(WI.BRAND.eq(brand.name()))).execute();
	}

	/** 关注列表 */
	private Condition getCondition(BrandEnum brand, Collection<String> uids) {
		return WI.BRAND.eq(brand.name()).and(WI.PUID.in(uids)).and(WI.IS_OPEN.ge(1))
				.and(WI.STATUS.eq(AuditStatus.online.getCode()));
	}

	public int countOpenNewWorks(BrandEnum brand, Collection<String> uids, long timestamp) {
		return mediaCtx.fetchCount(WI, getCondition(brand, uids).and(WI.CREATE_TIME.gt(timestamp)));
	}

	public int countOpenWorks(BrandEnum brand, Collection<String> uids) {
		return mediaCtx.fetchCount(WI, getCondition(brand, uids));
	}

	public List<String> findOpenWorksId(BrandEnum brand, Collection<String> uids, int start, int size) {
		return mediaCtx.select(WI.ID).from(WI).where(getCondition(brand, uids)).orderBy(WI.CREATE_TIME.desc())
				.limit(start, size).fetchInto(String.class);
	}

	/** 推荐列表 */
	private Condition getRecommonCondition(BrandEnum brand) {
		return WI.BRAND.eq(brand.name()).and(WI.IS_OPEN.ge(1)).and(WI.STATUS.eq(AuditStatus.online.getCode()));
	}

	public int countRecommonWorks(BrandEnum brand) {
		return mediaCtx.fetchCount(WI, getRecommonCondition(brand));
	}

	public List<String> findRecommonWorksId(BrandEnum brand, int start, int size) {
		return mediaCtx.select(WI.ID).from(WI).where(getRecommonCondition(brand)).orderBy(WI.CREATE_TIME.desc())
				.limit(start, size).fetchInto(String.class);
	}

	/** 获取存在作品的学校及课程 */
	public List<WorksInfo> findExistWorksInfo(Map<String, Collection<String>> school2LessonIds, String suid) {
		Condition condition = getOrWhereCondition(WI.SCHOOL_ID, WI.LESSON_ID, school2LessonIds);
		return mediaCtx.select(WI.SCHOOL_ID, WI.LESSON_ID, WI.ID).from(WI).where(condition.and(WI.SUID.eq(suid)))
				.fetchInto(WorksInfo.class);
	}

	/** 学员作品列表 */
	private Condition getStudentCondition(BrandEnum brand, String puid, String suid) {
		Condition condition = WI.BRAND.eq(brand.name()).and(WI.PUID.eq(puid));
		if (StringTools.isNotEmpty(suid))
			return condition.and(WI.SUID.eq(suid));
		return condition;
	}

	public int countStudentWorks(BrandEnum brand, String puid, String suid) {
		return mediaCtx.fetchCount(WI, getStudentCondition(brand, puid, suid));
	}

	public List<String> findStudentWorksId(BrandEnum brand, String puid, String suid, int start, int size) {
		return mediaCtx.select(WI.ID).from(WI).where(getStudentCondition(brand, puid, suid))
				.orderBy(WI.CREATE_TIME.desc()).limit(start, size).fetchInto(String.class);
	}

	public List<String> findAllStudentWorksId(BrandEnum brand, String puid, String suid) {
		return mediaCtx.select(WI.ID).from(WI).where(getStudentCondition(brand, puid, suid)).fetchInto(String.class);
	}

	/********************************************************************/
	/*********************** works theme relation *************************/
	/********************************************************************/

	private TableOnConditionStep getJoinTable() {
		return WTR.join(WI).on(WTR.WORK_ID.eq(WI.ID));
	}

	private Condition getRelationCondition(BrandEnum brand, String themeId) {
		return WTR.THEME_ID.eq(themeId).and(WI.BRAND.eq(brand.name())).and(WI.IS_OPEN.ge(1))
				.and(WI.STATUS.eq(AuditStatus.online.getCode()));
	}

	public int countWorks(BrandEnum brand, String themeId) {
		return mediaCtx.fetchCount(getJoinTable(), getRelationCondition(brand, themeId));
	}

	public List<String> getWorkIds(BrandEnum brand, String themeId, boolean isHot, int start, int size) {
		return mediaCtx.select(WTR.WORK_ID).from(getJoinTable()).where(getRelationCondition(brand, themeId))
				.orderBy(isHot ? WTR.LIKE_CNT.desc() : WTR.UPLOAD_TIME.desc()).limit(start, size)
				.fetchInto(String.class);
	}

	public List<String> getAllThemeIds(String artId) {
		return mediaCtx.select(WTR.THEME_ID).from(WTR).where(WTR.WORK_ID.eq(artId)).fetchInto(String.class);
	}

	public void addLikeCnt(String artId, int cnt) {
		mediaCtx.update(WTR).set(WTR.LIKE_CNT, WTR.LIKE_CNT.add(cnt)).where(WTR.WORK_ID.eq(artId)).execute();
	}
}
