package com.jz.jar.media.proxy;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.jz.common.utils.collection.ArrayMapTools;
import com.jz.jar.media.cache.SpecialPracticeCache;
import com.jz.jar.media.service.PlaylistRelationService;
import com.jz.jar.media.service.SpecialPracticeHistoryService;
import com.jz.jooq.media.tables.pojos.SpecialPracticeHistory;

/**
 * @Title SpecialPracticeProxy
 * @Package com.jz.jar.media.proxy
 * @author tangjunfeng
 * @date 2018年8月13日 下午5:19:17
 * @version V1.0
 */
@Lazy
@Service
public class SpecialPracticeProxy {

	private static final BigDecimal _100 = new BigDecimal(100);

	@Autowired
	private PlaylistRelationService playlistRelationService;
	@Autowired
	private SpecialPracticeHistoryService specialPracticeHistoryService;
	@Autowired
	private SpecialPracticeCache specialPracticeCache;

	/**
	 * 获取专项练习的专辑数量<br/>
	 * 当前获取只能从缓存中获取,具体查看方法 {@link #cutSpecialPracticePlaylist()}<br/>
	 * 所以当前方法必须在 {@link #getSpecialPlaylist()} 之后调用
	 */
	public int countSpecialPlaylist(String suid, String cid, boolean isFinished) {
		return specialPracticeCache.countSpecialPlaylist(suid, cid, isFinished);
	}

	/** 获取专项练习的专辑ID信息 */
	public Map<String, Integer> getSpecialPlaylist(String suid, String lid, String cid, boolean isFinished, int start,
			int size) {
		Map<String, Integer> playlist = specialPracticeCache.getSpecialPlaylist(suid, cid, isFinished, start, size);
		if (null != playlist) // 当前内容缓存中存在
			return playlist;
		// 缓存中不存在则直接从db中获取,然后写入缓存
		Pair<Map<String, Integer>, Map<String, Integer>> pair = this.cutSpecialPracticePlaylist(suid, lid, cid);
		playlist = isFinished ? pair.getRight() : pair.getLeft();
		if (null == playlist)
			return null;
		int fromIndex = start < 0 ? 0 : start;
		int toIndex = playlist.size() >= start + size ? start + size : playlist.size();
		if (fromIndex < toIndex) {
			Map<String, Integer> resultPlaylist = playlist;
			List<String> keys = Lists.newArrayList(playlist.keySet());
			return keys.subList(fromIndex, toIndex).stream()
					.collect(Collectors.toMap(it -> it, it -> resultPlaylist.get(it)));
		}
		return null;
	}

	/** 切分专项练习的专辑为(未完成/已完成) */
	public Pair<Map<String, Integer>, Map<String, Integer>> cutSpecialPracticePlaylist(String suid, String lid,
			String cid) {
		// 获取专项练习的专辑ids
		List<String> pidArray = playlistRelationService.findSpecialPlaylist(lid, cid);
		// 获取当前专项练习的专辑播放记录
		List<SpecialPracticeHistory> histories = specialPracticeHistoryService.getHistoryForPlaylist(suid, pidArray);
		// pid, 播放完成度的百分比值
		Map<String, Integer> unfinished = Maps.newLinkedHashMap(), finished = Maps.newLinkedHashMap();
		if (ArrayMapTools.isEmpty(histories)) { // 一个都没播过
			unfinished.putAll(pidArray.stream().collect(Collectors.toMap(it -> it, it -> 0)));
		} else { // 已经播过几个专辑了
			// 计算专项练习的专辑完成度
			histories.forEach(hy -> {
				if (hy.getPlayCnt() >= hy.getTargetCnt()) {
					finished.put(hy.getPid(), 100); // 已观看完毕, 则放到已经完成的列表中
					return;
				}
				BigDecimal targetCnt = new BigDecimal(hy.getTargetCnt());
				// 当前专辑的播放完成度
				int percent = new BigDecimal(hy.getPlayCnt()).divide(targetCnt, 2, BigDecimal.ROUND_HALF_UP).multiply(_100).intValue();
				unfinished.put(hy.getPid(), percent);
			});
		}
		// 存储未完成的专辑列表
		specialPracticeCache.setSpecialPlaylist(suid, cid, unfinished, false);
		// 存储已完成的专辑列表
		specialPracticeCache.setSpecialPlaylist(suid, cid, finished, true);
		return Pair.of(unfinished, finished);
	}
}
