package com.jz.gymchina.jar.resources.repository;

import java.util.Collection;
import java.util.List;

import org.jooq.Condition;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Repository;

import com.google.common.collect.Lists;
import com.jz.gymchina.jar.resources.enums.ResourceEnum;
import com.jz.jooq.gymchina.resources.Tables;
import com.jz.jooq.gymchina.resources.tables.pojos.GymFile;
import com.jz.jooq.gymchina.resources.tables.records.GymFileRecord;

/**
 * @Title FileRepository
 * @Package org.gymchina.resources.repository
 * @author tangjunfeng
 * @date 2019年3月11日 下午12:08:29
 * @version V1.0
 */
@Lazy
@Repository
public class FileRepository extends ResourceBaseRepository {

	private static com.jz.jooq.gymchina.resources.tables.GymFile FILE = Tables.GYM_FILE;

	public Condition getWhereCondition(String dirId) {
		return FILE.DIR_ID.eq(dirId).and(FILE.STATUS.eq(ResourceEnum.online.name()));
	}

	public int countFilesForDirectory(String dirId) {
		return resourcesCtx.fetchCount(FILE, getWhereCondition(dirId));
	}

	public boolean existFile(String fileId) {
		return resourcesCtx.fetchExists(FILE, FILE.ID.eq(fileId).and(FILE.STATUS.eq(ResourceEnum.online.name())));
	}

	public int countExistFile(Collection<String> fileIds) {
		return resourcesCtx.fetchCount(FILE, FILE.ID.in(fileIds).and(FILE.STATUS.eq(ResourceEnum.online.name())));
	}

	public List<GymFile> getFilesForDirectory(String dirId) {
		return resourcesCtx.select(FILE.ID, FILE.NAME, FILE.TYPE, FILE.OSS_URL, FILE.SIZE, FILE.LAST_UPDATE).from(FILE)
				.where(getWhereCondition(dirId)).orderBy(FILE.NAME.asc()).fetchInto(GymFile.class);
	}

	public List<GymFile> getFilesForDirectory(Collection<String> dirIds) {
		return resourcesCtx.selectFrom(FILE)
				.where(FILE.DIR_ID.in(dirIds).and(FILE.STATUS.eq(ResourceEnum.online.name()))).fetchInto(GymFile.class);
	}

	public List<GymFile> getFiles(Collection<String> fileIds) {
		return resourcesCtx.selectFrom(FILE).where(FILE.ID.in(fileIds).and(FILE.STATUS.eq(ResourceEnum.online.name())))
				.fetchInto(GymFile.class);
	}

	public GymFile getFile(String fileId) {
		return resourcesCtx.selectFrom(FILE).where(FILE.ID.eq(fileId).and(FILE.STATUS.eq(ResourceEnum.online.name())))
				.fetchAnyInto(GymFile.class);
	}

	public String getFileDirectory(String fileId) {
		return resourcesCtx.select(FILE.DIR_ID).from(FILE).where(FILE.ID.eq(fileId)).fetchAnyInto(String.class);
	}

	public void updateFileName(String fileId, String name, String uid) {
		resourcesCtx.update(FILE).set(FILE.NAME, name).set(FILE.UID, uid)
				.set(FILE.LAST_UPDATE, System.currentTimeMillis()).where(FILE.ID.eq(fileId)).execute();
	}

	public void deleteFile(String fileId, String ossUrl, String uid) {
		resourcesCtx.update(FILE).set(FILE.STATUS, ResourceEnum.delete.name()).set(FILE.OSS_URL, ossUrl)
				.set(FILE.UID, uid).set(FILE.LAST_UPDATE, System.currentTimeMillis()).where(FILE.ID.eq(fileId))
				.execute();
	}

	public void addFile(String fileId, String fileName, String dirId, String ext, String ossUrl, long size,
			String uid) {
		long timeMillis = System.currentTimeMillis();
		GymFileRecord file = new GymFileRecord();
		file.setId(fileId);
		file.setName(fileName);
		file.setDirId(dirId);
		file.setType(ext);
		file.setOssUrl(ossUrl);
		file.setSize(size);
		file.setStatus(ResourceEnum.online.name());
		file.setUid(uid);
		file.setCreateTime(timeMillis);
		file.setLastUpdate(timeMillis);
		resourcesCtx.insertInto(FILE).set(file).execute();
	}

	public void batchInsertFiles(Collection<GymFile> files) {
		List<GymFileRecord> records = Lists.newArrayList();
		for (GymFile file : files) {
			records.add(resourcesCtx.newRecord(FILE, file));
		}
		resourcesCtx.batchInsert(records).execute();
	}
}
