package com.jz.jar.franchise.repository;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.jooq.Condition;
import org.jooq.impl.DSL;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Repository;

import com.jz.common.utils.text.StringTools;
import com.jz.jooq.franchise.Tables;
import com.jz.jooq.franchise.tables.pojos.School;

@Lazy
@Repository
public class SchoolRepository extends FranchiseBaseRepository {

	private static final com.jz.jooq.franchise.tables.School SC = Tables.SCHOOL;

	public School getSchool(String id) {
		return franchiseCtx.selectFrom(SC).where(SC.ID.eq(id)).fetchAnyInto(School.class);
	}

	public School getSimpleSchool(String id) {
		return franchiseCtx.select(SC.ID, SC.CENTER_NAME, SC.CENTER_ADDRESS, SC.PHONE).from(SC).where(SC.ID.eq(id))
				.fetchAnyInto(School.class);
	}

	public List<School> getSimpleSchool(Collection<String> ids) {
		return franchiseCtx.select(SC.ID, SC.CENTER_NAME, SC.CENTER_ADDRESS, SC.PHONE).from(SC).where(SC.ID.in(ids))
				.fetchInto(School.class);
	}

	private Condition getCondition(String brand) {
		// 预约的学校只展示 code >= 100 的学校
		return SC.BRAND_ID.eq(brand).and((SC.STATUS.eq(1).and(SC.LAST_CONTRACT.ne(1))).or(SC.STATUS.eq(4)))
				.and(SC.CODE.ge(100));
	}

	private Condition getSchoolCondition(String brand, String prov, String city, boolean isReservation) {
		Condition condition = getCondition(brand);
		if (StringTools.isNotEmpty(prov))
			condition = condition.and(SC.PROV.eq(prov));
		if (StringTools.isNotEmpty(city))
			condition = condition.and(SC.CITY.eq(city));
		if (isReservation) // 非预约试听可展示所有的学校
			condition = condition.and(SC.CENTER_NAME.isNotNull()).and(SC.CENTER_NAME.ne(""));
		return condition;
	}

	/** city_level字段用作表示学校数量（不想添加字段） */
	public List<School> getSchoolAreas(String brand, boolean isReservation) {
		return franchiseCtx
				.select(SC.PROV, SC.CITY, DSL.count(DSL.one()).as(SC.CITY_LEVEL))
				.from(SC)
				.where(getSchoolCondition(brand, null, null, isReservation))
				.groupBy(SC.PROV, SC.CITY)
				.orderBy(
						DSL.choose(SC.PROV).when("北京", 9999).when("上海", 8888).when("广东", 7777)
								.otherwise(DSL.count(DSL.one())).desc()).fetchInto(School.class);
	}

	public List<School> getSchools(String brand, String prov, String city, boolean isReservation) {
		return franchiseCtx
				.select(SC.ID,
						DSL.choose().when(SC.STATUS.eq(1), SC.CENTER_NAME)
								.otherwise(DSL.concat(SC.CENTER_NAME, "(店铺升级中)")).as(SC.CENTER_NAME),
						SC.CENTER_ADDRESS, SC.PHONE, SC.LAT, SC.LNG, SC.JOIN_TIME, SC.CONTRACT_END_TIME)
				.from(SC)
				.where(getSchoolCondition(brand, prov, city, isReservation))
				.orderBy(
						DSL.choose()
								.when(SC.OPEN_TIME.isNotNull().and(SC.OPEN_TIME.lt(System.currentTimeMillis()))
										.and(SC.STATUS.eq(1)), 1).otherwise(2).asc(), SC.CODE.desc())
				.fetchInto(School.class);
	}

	public List<School> getSchoolForName(String brand, String schoolName) {
		return franchiseCtx
				.select(SC.ID,
						DSL.choose().when(SC.STATUS.eq(1), SC.NAME).otherwise(DSL.concat(SC.NAME, "(店铺升级中)"))
								.as(SC.CENTER_NAME), SC.CENTER_ADDRESS, SC.PHONE, SC.LAT, SC.LNG, SC.JOIN_TIME,
						SC.CONTRACT_END_TIME).from(SC)
				.where(getCondition(brand).and(SC.NAME.like("%" + schoolName + "%"))).orderBy(SC.CODE.desc())
				.fetchInto(School.class);
	}

	public String getSchoolName(String id) {
		return franchiseCtx.select(SC.NAME).from(SC).where(SC.ID.eq(id).and(SC.STATUS.eq(1)))
				.fetchAnyInto(String.class);
	}

	public List<String> getSchoolIdForBrand(String brand) {
		return franchiseCtx.select(SC.ID).from(SC).where(SC.BRAND_ID.eq(brand).and(SC.STATUS.eq(1)))
				.fetchInto(String.class);
	}

	public Map<String, String> mutiGetId2NameMap(Collection<String> schoolIds) {
		return franchiseCtx.select(SC.ID, SC.NAME).from(SC).where(SC.ID.in(schoolIds)).fetchMap(SC.ID, SC.NAME);
	}

	public List<String> filterSchoolIdByBrand(Collection<String> schoolIds, String brandId) {
		return franchiseCtx.select(SC.ID).from(SC).where(SC.ID.in(schoolIds).and(SC.BRAND_ID.eq(brandId)))
				.fetchInto(String.class);
	}

	public List<String> filterDirectSchoolIds(Collection<String> schoolIds) {
		return franchiseCtx.select(SC.ID).from(SC).where(SC.ID.in(schoolIds).and(SC.TYPE.eq(1)))
				.fetchInto(String.class);
	}

}
