package com.jz.jar.franchise.cache;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import com.google.common.collect.Lists;
import com.jz.common.utils.collection.ArrayMapTools;
import com.jz.common.utils.json.GsonTools;
import com.jz.common.utils.text.StringTools;
import com.jz.jooq.franchise.tables.pojos.Lesson;

import redis.clients.jedis.JedisCluster;

/**
 * @Title LessonCache
 * @Package com.jz.jar.franchise.cache
 * @author tangjunfeng
 * @date 2018年8月30日 下午5:56:06
 * @version V1.0
 */
@Lazy
@Component
public class LessonCache {

	private static final Logger logger = LoggerFactory.getLogger(LessonCache.class);

	/** 过期时间:10m */
	private static final int seconds = 600;

	@Autowired
	private JedisCluster jedisCluster;

	private String key(Lesson lesson) {
		return key(lesson.getSchoolId(), lesson.getLessonId());
	}

	private String key(String school, String lesson) {
		return new StringBuffer().append("school2Lesson:").append(school).append(":").append(lesson).toString();
	}

	public void setLesson(Lesson lesson) {
		if (null == lesson)
			return;
		try {
			jedisCluster.set(key(lesson), GsonTools.gson.toJson(lesson), "nx", "ex", seconds);
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		}
	}

	public void setLesson(List<Lesson> lessons) {
		if (ArrayMapTools.isEmpty(lessons))
			return;
		try {
			lessons.forEach(it -> {
				jedisCluster.set(key(it), GsonTools.gson.toJson(it), "nx", "ex", seconds);
			});
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		}
	}

	public Lesson getLesson(String school, String lesson) {
		if (!StringTools.allNotEmptyAndBlank(school, lesson))
			return null;
		try {
			String gson = jedisCluster.get(key(school, lesson));
			if (StringTools.isEmptyAndBlank(gson))
				return null;
			return GsonTools.gson.fromJson(gson, Lesson.class);
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
			return null;
		}
	}

	public List<Lesson> getLesson(Map<String, Collection<String>> school2Lessons) {
		if (ArrayMapTools.isEmpty(school2Lessons))
			return null;
		try {
			List<Lesson> result = Lists.newArrayList();
			school2Lessons.forEach((schoolId, lessons) -> {
				lessons.forEach(lid -> {
					String gson = jedisCluster.get(key(schoolId, lid));
					if (StringTools.isNotEmptyAndBlank(gson)) {
						result.add(GsonTools.gson.fromJson(gson, Lesson.class));
					}
				});
			});
			return result;
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
			return null;
		}
	}

}
