package com.jz.jar.franchise.service;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.jz.common.utils.collection.ArrayMapTools;
import com.jz.jar.franchise.enums.ContractParentSignStatus;
import com.jz.jar.franchise.repository.ContractPostponeRepository;
import com.jz.jar.franchise.repository.ContractRepository;
import com.jz.jar.franchise.repository.ContractTermsRepository;
import com.jz.jar.franchise.repository.StudentInfoRepository;
import com.jz.jar.franchise.repository.StudentSchoolContractRepository;
import com.jz.jar.franchise.wrapper.ContractPostponeWrapper;
import com.jz.jar.franchise.wrapper.ContractStudentWrapper;
import com.jz.jar.franchise.wrapper.ContractWrapper;
import com.jz.jar.franchise.wrapper.CourseWrapper;
import com.jz.jar.franchise.wrapper.SchoolWrapper;
import com.jz.jar.franchise.wrapper.UserSimpleWrapper;
import com.jz.jooq.franchise.tables.pojos.Contract;
import com.jz.jooq.franchise.tables.pojos.ContractCompany;
import com.jz.jooq.franchise.tables.pojos.ContractDetail;
import com.jz.jooq.franchise.tables.pojos.ContractPostpone;
import com.jz.jooq.franchise.tables.pojos.School;
import com.jz.jooq.franchise.tables.pojos.StudentInfo;
import com.jz.jooq.franchise.tables.pojos.StudentSchoolContract;
import com.jz.jooq.franchise.tables.pojos.StudentSchoolExtra;

@Lazy
@Service
public class ContractService {

	private Logger logger = LoggerFactory.getLogger(ContractService.class);

	@Autowired
	private ContractRepository contractRepository;
	@Autowired
	private StudentSchoolContractRepository studentSchoolContractRepository;
	@Autowired
	private FUserInfoService fUserInfoService;
	@Autowired
	private CourseService courseService;
	@Autowired
	private CoursePackService coursePackService;
	@Autowired
	private ParentInfoService parentInfoService;
	@Autowired
	private StudentInfoRepository studentInfoRepository;
	@Autowired
	private ContractPostponeRepository contractPostponeRepository;
	@Autowired
	private ContractTermsRepository contractTermsRepository;
	@Autowired
	private ContractCompanyService contractCompanyService;

	public Contract getContract(String brandId, String contractId) {
		return contractRepository.getContractForBranch(brandId, contractId);
	}

	public Pair<ContractWrapper, SchoolWrapper> getContractDetailWrapper(String contractId, String schoolId,
			School school) {
		Contract contract = contractRepository.getContract(contractId, schoolId);
		ContractDetail contractDetail = contractRepository.getContractDetail(contractId, schoolId);
		if (contract == null) {
			logger.error("获取不到合同信息");
			return null;
		}
		if (contractDetail == null) {
			logger.error("合同信息缺失，请联系管理员");
			return null;
		}

		ContractWrapper contractWrapper = ContractWrapper.of(contract);
		contractWrapper.setContractDetail(contractDetail);

		// 封装学员信息
		List<StudentSchoolContract> sscs = studentSchoolContractRepository.getSSCByContractId(schoolId, contractId);
		if (ArrayMapTools.isEmpty(sscs)) {
			logger.error("合同学生信息缺失，请联系管理员");
			return null;
		}
		List<ContractStudentWrapper> students = Lists.newArrayList();
		for (StudentSchoolContract ssc : sscs) {
			StudentInfo si = studentInfoRepository.getStudent(ssc.getSuid());
			if (si == null) {
				logger.warn("合同学员信息不存在:" + ssc.getSuid());
				return null;
			}
			students.add(ContractStudentWrapper.ofStudentInfo(si));
		}
		contractWrapper.setStudents(students);

		// 家长信息
		contractWrapper.setParent(parentInfoService.getParentWrapper(contract.getPuid()));
		// 课包信息
		contractWrapper.setCoursePack(coursePackService.getCoursePackWrapper(school.getBrandId(), school.getProv(),
				school.getCity(), schoolId, contract.getCoursePackId()));

		List<Integer> courseIds = coursePackService.getCoursesByPackId(school.getBrandId(), school.getProv(),
				school.getCity(), schoolId, contract.getCoursePackId());
		if (ArrayMapTools.isNotEmpty(courseIds)) {
			Map<Integer, CourseWrapper> courseMap = courseService.mutiGetCourseInfoMap(school.getBrandId(), schoolId,
					courseIds, false);
			if (ArrayMapTools.isNotEmpty(courseIds)) {
				contractWrapper.setCourses(courseMap.values());
			}
		}

		Set<String> fuids = Sets.newHashSet();
		if (StringUtils.isNotEmpty(contract.getAdviser())) {
			fuids.add(contract.getAdviser());
		}
		if (StringUtils.isNotEmpty(contract.getAdviser2())) {
			fuids.add(contract.getAdviser2());
		}
		if (StringUtils.isNotEmpty(contract.getTeacher())) {
			fuids.add(contract.getTeacher());
		}
		if (StringUtils.isNotEmpty(contract.getTeacher2())) {
			fuids.add(contract.getTeacher2());
		}
		if (StringUtils.isNotEmpty(contract.getMarket())) {
			fuids.add(contract.getMarket());
		}
		// 加盟商用户信息，顾问老师市场等
		Map<String, UserSimpleWrapper> userMap = null;
		if (ArrayMapTools.isNotEmpty(fuids)) {
			userMap = fUserInfoService.mutiGetUserWrappers(fuids);
		}
		if (StringUtils.isNotEmpty(contract.getAdviser())) {
			contractWrapper.setAdviser(userMap.get(contract.getAdviser()));
		}
		if (StringUtils.isNotEmpty(contract.getAdviser2())) {
			contractWrapper.setAdviser2(userMap.get(contract.getAdviser2()));
			contractWrapper.setAdviser2Rat(contract.getAdviser2Rat());
		}
		if (StringUtils.isNotEmpty(contract.getTeacher())) {
			contractWrapper.setTeacher(userMap.get(contract.getTeacher()));
		}
		if (StringUtils.isNotEmpty(contract.getTeacher2())) {
			contractWrapper.setTeacher2(userMap.get(contract.getTeacher2()));
		}
		if (StringUtils.isNotEmpty(contract.getMarket())) {
			contractWrapper.setMarket(userMap.get(contract.getMarket()));
		}

		// 增加合同延期信息
		List<ContractPostpone> postpones = contractPostponeRepository.getPayedPostpones(schoolId, contractId);
		if (ArrayMapTools.isNotEmpty(postpones)) {
			List<ContractPostponeWrapper> pps = Lists.newArrayList();
			for (ContractPostpone cp : postpones) {
				pps.add(ContractPostponeWrapper.of(cp));
			}
			contractWrapper.setPostpones(pps);
		}

		contractWrapper.setSerialNum(school.getCode() + "" + contract.getPayTime());

		// 电子签信息
		contractWrapper.setParentSignContent(contractRepository.getParentSignContent(contractId));

		if (StringUtils.isNotEmpty(contractDetail.getTermId())) {
			contractWrapper.setTermsDetails(contractTermsRepository.getDetails(schoolId, contractDetail.getTermId()));
			contractWrapper.setTermsRemark(contractTermsRepository.getRemark(schoolId, contractDetail.getTermId()));
		}

		SchoolWrapper schoolWrapper = SchoolWrapper.ofContract(school);
		if (contractDetail.getCompanyId() != null) {
			ContractCompany cc = contractCompanyService.getContractCompany(schoolId, contractDetail.getCompanyId());
			if (cc != null) {
				schoolWrapper.setCompany(cc.getCompany()).setStamp(cc.getStamp());
			}
		}

		return Pair.of(contractWrapper, schoolWrapper);
	}

	public List<ContractWrapper> getParentContractList(String brandId, String puid, boolean needNoPay) {
		List<Contract> contracts = needNoPay ? contractRepository.getUserContractList(brandId, puid)
				: contractRepository.getPayedContractList(brandId, puid);
		List<ContractWrapper> list = Lists.newArrayList();
		if (ArrayMapTools.isEmpty(contracts)) {
			return list;
		}
		List<String> contractIds = contracts.stream().map(it -> it.getContractId()).collect(Collectors.toList());
		List<Map<String, Object>> consumeList = studentSchoolContractRepository.mutiCalContractConsumes(contractIds);
		Map<String, Integer> consumeOfficalLessonMap = Maps.newHashMap();
		Map<String, Integer> noScheduleOfficalLessonMap = Maps.newHashMap();
		for (Map<String, Object> map : consumeList) {
			String contractId = MapUtils.getString(map, "contractId");
			consumeOfficalLessonMap.put(contractId, MapUtils.getIntValue(map, "consumeOfficalLesson"));
			noScheduleOfficalLessonMap.put(contractId, MapUtils.getIntValue(map, "noScheduleOfficalLesson"));
		}
		Map<String, Integer> leftExtraLessonMap = Maps.newHashMap(), consumeMap = Maps.newHashMap();
		List<StudentSchoolExtra> extras = studentSchoolContractRepository.mutiGetContractExtras(contractIds);
		if (ArrayMapTools.isNotEmpty(extras)) {
			for (StudentSchoolExtra extra : extras) {
				if (extra.getTotal() > extra.getConsume()) {
					leftExtraLessonMap.put(
							extra.getContractId(),
							MapUtils.getIntValue(leftExtraLessonMap, extra.getContractId()) + extra.getTotal()
									- extra.getConsume());
					consumeMap.put(extra.getContractId(), MapUtils.getIntValue(consumeMap, extra.getContractId())
							+ extra.getConsume());
				}
			}
		}
		for (Contract c : contracts) {
			ContractWrapper wrapper = ContractWrapper.of(c);
			int consumeOfficalLesson = MapUtils.getIntValue(consumeOfficalLessonMap, c.getContractId());
			wrapper.setConsumeOfficalLesson(consumeOfficalLesson);
			wrapper.setNoScheduleOfficalLesson(MapUtils.getIntValue(noScheduleOfficalLessonMap, c.getContractId()));

			int extraLeftLesson = MapUtils.getIntValue(leftExtraLessonMap, c.getContractId());
			wrapper.setExtraLeftLesson(extraLeftLesson);
			int totalLeftLesson = c.getTotalOfficalLesson() - consumeOfficalLesson + extraLeftLesson;
			wrapper.setTotalLeftLesson(totalLeftLesson);
			wrapper.setConsumeExtraLesson(MapUtils.getIntValue(consumeMap, c.getContractId()));
			wrapper.setIsFinished(totalLeftLesson <= 0);

			// 合同状态0未交费 1已缴费 2已交订金 3已退费 4已退订金5已转出-1废弃
			// 10等待校长审批 11校长驳回 12等待投资人审批 13投资人驳回 14等待财务确认15 财务驳回16等待总部确认17总部驳回
			// 20延期等待校长审批 21延期校长驳回 22延期等待投资人审批 23延期投资人驳回 24延期等待财务确认25延期 财务驳回
			// 30退费等待校长审批 31退费校长驳回 32退费等待投资人审批 33退费投资人驳回 34退费等待财务确认35退费财务驳回
			// 40退订金等待校长审批 41退订金校长驳回 42退订金等待投资人审批 43退订金投资人驳回
			// 44退订金等待财务确认45退订金财务驳回
			// 50转出中

			// 合同状态 100待签署 101审核中 102在读 103完结 104已退费 105已转出 106订金
			if (c.getStatus() == 3 || c.getStatus() == 4) {
				wrapper.setStatus(104);
			} else if (c.getStatus() == 5) {
				wrapper.setStatus(105);
			} else if (c.getStatus() == 2 || (c.getStatus() >= 40 && c.getStatus() <= 49)) {
				wrapper.setStatus(106);
			} else {
				if (c.getStatus() >= 10 && c.getStatus() < 20) { // 审核中
					if (c.getParentSignStatus() == ContractParentSignStatus.waitSignAffect.code
							&& (c.getStatus() == 10 || c.getStatus() == 12 || c.getStatus() == 14
									|| c.getStatus() == 16 || c.getStatus() == 18)) {
						wrapper.setStatus(100);
					} else {
						wrapper.setStatus(101);
					}
				} else {
					wrapper.setStatus(totalLeftLesson <= 0 ? 103 : 102);
				}
			}
			list.add(wrapper);
		}
		return list;
	}

	public boolean isHaveContract(String brandId, String puid) {
		return contractRepository.isHaveContract(brandId, puid);
	}

	public boolean isOwnedContract(String branchId, String puid) {
		return contractRepository.isOwnedContract(branchId, puid);
	}

	public List<String> getContractedSchoolIds(String brandId, String puid) {
		return contractRepository.getContractedSchoolIds(brandId, puid);
	}

	public Contract getSimpleContract(String contractId) {
		return contractRepository.getSimpleContract(contractId);
	}

	public int cntWaitSignForAppOpen(String brandId, String puid) {
		return contractRepository.cntWaitParentSignForAppOpen(brandId, puid);
	}

	public List<String> getSchoolIds(String brandId, String puid) {
		return contractRepository.getSchoolIds(brandId, puid);
	}

	public boolean checkIsHaveContractAllBrand(String puid) {
		return contractRepository.checkIsHaveContractAllBrand(puid);
	}

	public String getContractedSchoolId(String brandId, String puid) {
		return contractRepository.getContractedSchoolId(brandId, puid);
	}
}
