package com.jz.common.utils.datetime;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

import com.jz.common.utils.text.StringTools;

public class TimeTools {

	/** 取得当前时间戳与modulus的模数 */
	public static long getTimestampModulus(int modulus) {
		return getCurrentTime() % modulus;
	}

	/** 让线程休眠一段时间 */
	public static void sleep(int seconds) {
		sleepMillis(seconds * 1000);
	}

	/** 休眠N毫秒 */
	public static void sleepMillis(long millis) {
		try {
			Thread.sleep(millis);
		} catch (Exception ex) {
		}
	}

	/** 根据一个Unix时间截取得一个日期字符串（yyyy-MM-dd HH:mm:ss） */
	public static String getDateTime(long timestamp) {
		return new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(new Date(timestamp));
	}

	public static String getDateTime(long timestamp, String format) {
		return new SimpleDateFormat(format).format(new Date(timestamp));
	}

	public static String getDateTime(long timestamp, SimpleDateFormat format) {
		return format.format(new Date(timestamp));
	}

	public static String getCurrentDateTime(String format) {
		return getDateTime(Calendar.getInstance().getTimeInMillis(), format);
	}

	/**
	 * 根据一个日期字符串取得一个Unix时间截
	 * 
	 * @param dateStr 如:2009-05-15,或2009-05-15 00:00:01
	 */
	public static long getUnixTime(String dateStr) {
		Date date = null;
		try {
			date = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").parse(dateStr.trim());
		} catch (ParseException ex) {
			try {
				date = new SimpleDateFormat("yyyy-MM-dd").parse(dateStr.trim());
			} catch (ParseException ex2) {
			}
		}

		return date != null ? date.getTime() : 0L;
	}

	/** 取得当前的时间戳 */
	public static long getCurrentTime() {
		return System.currentTimeMillis();
	}

	/** 取得当前的时间戳（秒） */
	public static long getCurrentTimeSeconds() {
		return System.currentTimeMillis() / 1000;
	}

	/** 取得当前的日期 */
	public static String getCurrentDate() {
		return getDateTime(System.currentTimeMillis());
	}

	/** 将时间秒数格式化hh:mm:ss的格式 */
	public static String format(long seconds) {
		int h = (int) (seconds / 3600);
		int m = (int) (seconds - h * 3600) / 60;
		int s = (int) (seconds - h * 3600 - m * 60);
		return String.format("%s:%s:%s", h, m, s);
	}

	private static String transferDouble(int in) {
		return in > 9 ? in + "" : "0" + in;
	}

	/** 将时间秒数格式化hh:mm:ss的格式 */
	public static String doubleFormat(long seconds) {
		int h = (int) (seconds / 3600);
		int m = (int) (seconds - h * 3600) / 60;
		int s = (int) (seconds - h * 3600 - m * 60);
		return String.format("%s:%s:%s", transferDouble(h), transferDouble(m), transferDouble(s));
	}

	/** 获取当前周一的开始时间 */
	public static long getWeekStartTime() {
		Calendar c = Calendar.getInstance();
		c.set(c.get(Calendar.YEAR), c.get(Calendar.MONTH), c.get(Calendar.DAY_OF_MONTH), 0, 0, 0);
		c.set(Calendar.MILLISECOND, 0);
		int day_of_week = c.get(Calendar.DAY_OF_WEEK) - 1;
		if (day_of_week == 0) // 星期天
			day_of_week = 7;
		c.add(Calendar.DATE, -day_of_week + 1);
		return c.getTimeInMillis();
	}

	/** 获取当前天是星期几 */
	public static int getCurrentDayInWeek() {
		Calendar c = Calendar.getInstance();
		int day_of_week = c.get(Calendar.DAY_OF_WEEK) - 1;
		return day_of_week == 0 ? 7 : day_of_week; // 星期天
	}

	/** 获取当天的开始时间 */
	public static long getCurrentDayFirstTime() {
		return getAnyDayFirstTime(null);
	}

	/** 获取当天的结束时间 */
	public static long getCurrentDayLastTime() {
		return getAnyDayLastTime(null);
	}

	/** 获取某天的开始时间 */
	public static long getAnyDayFirstTime(Long timeMillis) {
		Calendar c = Calendar.getInstance();
		if (null != timeMillis)
			c.setTimeInMillis(timeMillis);
		c.set(c.get(Calendar.YEAR), c.get(Calendar.MONTH), c.get(Calendar.DAY_OF_MONTH), 0, 0, 0);
		c.set(Calendar.MILLISECOND, 0);
		return c.getTimeInMillis();
	}

	/** 获取某天的结束时间 */
	public static long getAnyDayLastTime(Long timeMillis) {
		Calendar c = Calendar.getInstance();
		if (null != timeMillis)
			c.setTimeInMillis(timeMillis);
		c.set(c.get(Calendar.YEAR), c.get(Calendar.MONTH), c.get(Calendar.DAY_OF_MONTH) + 1, 0, 0, 0);
		c.set(Calendar.MILLISECOND, -1);
		return c.getTimeInMillis();
	}

	/** 获取当前月任意一天的开始时间 */
	public static long getCurrentMonthAnyDayFirstTime(int anyDay) {
		if (anyDay < 1 || anyDay > 31)
			throw new RuntimeException("anyDay values [1~31] error anyDay value : " + anyDay);
		Calendar c = Calendar.getInstance();
		c.set(c.get(Calendar.YEAR), c.get(Calendar.MONTH), anyDay, 0, 0, 0);
		c.set(Calendar.MILLISECOND, 0);
		return c.getTimeInMillis();
	}

	/** 获取当前月任意一天的结束时间 */
	public static long getCurrentMonthAnyDayLastTime(int anyDay) {
		if (anyDay < 1 || anyDay > 31)
			throw new RuntimeException("anyDay values [1~31] error anyDay value : " + anyDay);
		Calendar c = Calendar.getInstance();
		c.set(c.get(Calendar.YEAR), c.get(Calendar.MONTH), anyDay + 1, 0, 0, 0);
		c.set(Calendar.MILLISECOND, -1);
		return c.getTimeInMillis();
	}

	/** 获取当前月开始的时间 */
	public static long getCurrentMonthFirstTime() {
		Calendar c = Calendar.getInstance();
		c.set(c.get(Calendar.YEAR), c.get(Calendar.MONTH), 1, 0, 0, 0);
		c.set(Calendar.MILLISECOND, 0);
		return c.getTimeInMillis();
	}

	/** 获取当前月最后一刻的时间 */
	public static long getCurrentMonthLastTime() {
		Calendar c = Calendar.getInstance();
		c.set(c.get(Calendar.YEAR), c.get(Calendar.MONTH) + 1, 1, 0, 0, 0);
		c.set(Calendar.MILLISECOND, -1);
		return c.getTimeInMillis();
	}

	/** 获取当前周的开始时间 */
	public static long getCurrentWeekStartTime() {
		return getCurrentWeekAnyDayStartTime(1);
	}

	/** 获取当前周任意一天的开始时间 */
	public static long getCurrentWeekAnyDayStartTime(int anyDay) {
		if (anyDay < 1 || anyDay > 7)
			throw new RuntimeException("anyDay values [1~7] error anyDay value : " + anyDay);
		Calendar c = Calendar.getInstance();
		c.set(c.get(Calendar.YEAR), c.get(Calendar.MONTH), c.get(Calendar.DAY_OF_MONTH), 0, 0, 0);
		c.set(Calendar.MILLISECOND, 0);
		// 一周的开始时间是周一
		c.setFirstDayOfWeek(Calendar.MONDAY);
		c.set(Calendar.DAY_OF_WEEK, c.getFirstDayOfWeek() + (anyDay - 1));
		return c.getTimeInMillis();
	}

	/** 获取当前周的结束时间 */
	public static long getCurrentWeekLastTime() {
		return getCurrentWeekAnyDayLastTime(7);
	}

	/** 获取当前周任意一天的结束时间 */
	public static long getCurrentWeekAnyDayLastTime(int anyDay) {
		if (anyDay < 1 || anyDay > 7)
			throw new RuntimeException("anyDay values [1~7] error anyDay value : " + anyDay);
		Calendar c = Calendar.getInstance();
		// 获取当前周的周日的开始时间
		c.setTimeInMillis(getCurrentWeekAnyDayStartTime(anyDay));
		c.add(Calendar.DAY_OF_MONTH, 1);
		c.set(Calendar.MILLISECOND, -1);
		return c.getTimeInMillis();
	}

	/**
	 * 获取今天所属全年第几周<br/>
	 * 如：201701
	 */
	public static int getCurrentWeekOfYear() {
		return getCurrentWeekOfYear(null);
	}

	/**
	 * 获取今天所属全年第几周<br/>
	 * 如：201701
	 */
	public static int getCurrentWeekOfYear(Long timeMillis) {
		Calendar calendar = Calendar.getInstance();
		if (null != timeMillis) {
			calendar.setTimeInMillis(timeMillis);
		}
		int week = calendar.get(Calendar.WEEK_OF_YEAR);
		if (calendar.get(Calendar.DAY_OF_WEEK) - 1 == 0) // 星期天
			week--;
		if (week == 0) { // 新年
			calendar.add(Calendar.DAY_OF_MONTH, -1);
			week = calendar.get(Calendar.WEEK_OF_YEAR);
		}
		// 当前周属于新的一年
		if (week == 1 && calendar.get(Calendar.MONTH) == 11) {
			return Integer.parseInt(calendar.get(Calendar.YEAR) + 1 + StringTools.twoDigits(week));
		}
		return Integer.parseInt(calendar.get(Calendar.YEAR) + StringTools.twoDigits(week));
	}

	/**
	 * 获取上周所属全年第几周<br/>
	 * 如：201701
	 */
	public static int getLastWeekOfYear() {
		Calendar calendar = Calendar.getInstance();
		calendar.add(Calendar.WEEK_OF_YEAR, -1);
		return getCurrentWeekOfYear(calendar.getTimeInMillis());
	}

	/** 获取下周的开始时间 */
	public static long getNextWeekStartTime() {
		return getNextWeekAnyDayStartTime(1);
	}

	/** 获取下周任意一天的开始时间 */
	public static long getNextWeekAnyDayStartTime(int anyDay) {
		if (anyDay < 1 || anyDay > 7)
			throw new RuntimeException("anyDay values [1~7] error anyDay value : " + anyDay);
		Calendar c = Calendar.getInstance();
		c.set(c.get(Calendar.YEAR), c.get(Calendar.MONTH), c.get(Calendar.DAY_OF_MONTH), 0, 0, 0);
		// 下周
		c.add(Calendar.WEEK_OF_YEAR, 1);
		int day_of_week = c.get(Calendar.DAY_OF_WEEK) - 1;
		if (day_of_week == 0) // 星期天
			day_of_week = 7;
		c.add(Calendar.DAY_OF_MONTH, -day_of_week + anyDay);
		return c.getTimeInMillis();
	}

	/** 获取下周的结束时间 */
	public static long getNextWeekLastTime() {
		return getNextWeekAnyDayLastTime(7);
	}

	/** 获取下周任意一天的结束时间 */
	public static long getNextWeekAnyDayLastTime(int anyDay) {
		if (anyDay < 1 || anyDay > 7)
			throw new RuntimeException("anyDay values [1~7] error anyDay value : " + anyDay);
		Calendar c = Calendar.getInstance();
		c.set(c.get(Calendar.YEAR), c.get(Calendar.MONTH), c.get(Calendar.DAY_OF_MONTH) + 1, 0, 0, 0);
		c.set(Calendar.MILLISECOND, -1);
		// 下周
		c.add(Calendar.WEEK_OF_YEAR, 1);
		int day_of_week = c.get(Calendar.DAY_OF_WEEK) - 1;
		if (day_of_week == 0) // 星期天
			day_of_week = 7;
		c.add(Calendar.DAY_OF_MONTH, -day_of_week + anyDay);
		return c.getTimeInMillis();
	}

	/**
	 * 根据年的第几周获取该周的开始时间<br/>
	 * 如：201837，返回2018-09-10 00:00:00的时间戳
	 */
	public static long getWeekOfYearFirstTime(int weekOfYear) {
		int year = Integer.parseInt((weekOfYear + "").substring(0, 4));
		Calendar c = Calendar.getInstance();
		c.set(year, c.get(Calendar.MONTH), c.get(Calendar.DAY_OF_MONTH), 0, 0, 0);
		c.set(Calendar.MILLISECOND, 0);
		int yearOfFirstWeek = Integer.parseInt(year + StringTools.twoDigits(0));
		int weekNum = weekOfYear - yearOfFirstWeek;
		// 设置周一是一周的第一天
		c.setFirstDayOfWeek(Calendar.MONDAY);
		c.set(Calendar.WEEK_OF_YEAR, weekNum);
		c.set(Calendar.DAY_OF_WEEK, c.getFirstDayOfWeek());
		return c.getTimeInMillis();
	}

	/**
	 * 根据年的第几周获取该周的结束时间<br/>
	 * 如：201837，返回2018-09-16 23:59:59的时间戳
	 */
	public static long getWeekOfYearLastTime(int weekOfYear) {
		Calendar c = Calendar.getInstance();
		c.setTimeInMillis(getWeekOfYearFirstTime(weekOfYear));
		c.add(Calendar.DAY_OF_WEEK, 7);
		c.add(Calendar.MILLISECOND, -1);
		return c.getTimeInMillis();
	}

	/** 获取周一的开始时间(2018-09-10) */
	public static int getWeekStartTime(Long timeMillis) {
		Calendar c = Calendar.getInstance();
		if (null != timeMillis) {
			c.setTimeInMillis(timeMillis);
		}
		// 一周开始是周一
		c.setFirstDayOfWeek(Calendar.MONDAY);
		c.set(Calendar.DAY_OF_WEEK, c.getFirstDayOfWeek());
		return Integer.parseInt(c.get(Calendar.YEAR) + StringTools.twoDigits(c.get(Calendar.MONTH) + 1)
				+ StringTools.twoDigits(c.get(Calendar.DAY_OF_MONTH)));
	}

	/** 获取前一周周一的开始时间(2018-09-03) */
	public static int getLastWeekStartTime(Long timeMillis) {
		Calendar c = Calendar.getInstance();
		if (null != timeMillis) {
			c.setTimeInMillis(timeMillis);
		}
		// 一周开始是周一
		c.setFirstDayOfWeek(Calendar.MONDAY);
		c.set(Calendar.DAY_OF_WEEK, c.getFirstDayOfWeek());
		c.add(Calendar.WEEK_OF_YEAR, -1);
		return Integer.parseInt(c.get(Calendar.YEAR) + StringTools.twoDigits(c.get(Calendar.MONTH) + 1)
				+ StringTools.twoDigits(c.get(Calendar.DAY_OF_MONTH)));
	}

}