package com.jz.jar.business.proxy.common;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.jz.common.utils.collection.ArrayMapTools;
import com.jz.common.utils.collection.OrderTool;
import com.jz.jar.business.card.ItemCard;
import com.jz.jar.business.request.VideoRequest;
import com.jz.jar.business.tool.VideoTool;
import com.jz.jar.business.wrapper.VideoWrapper;
import com.jz.jar.media.enums.BrandEnum;
import com.jz.jar.media.enums.SourceStatus;
import com.jz.jar.media.service.UploadVideoScoreService;
import com.jz.jar.media.service.UploadVideoService;
import com.jz.jooq.media.tables.pojos.UploadVideo;
import com.jz.jooq.media.tables.pojos.UploadVideoScore;

/**
 * @Title UploadVideoProxy
 * @Package com.jz.jar.media.proxy
 * @author tangjunfeng
 * @date 2018年10月24日 上午11:49:03
 * @version V1.0
 */
@Lazy
@Service
public class UploadVideoProxy {

	@Autowired
	private UploadVideoService uploadVideoService;
	@Autowired
	private UploadVideoScoreService uploadVideoScoreService;

	/** 是否学员上传的视频 */
	public boolean isStudentUploadVideo(String wid) {
		// oss.oss_identity_config.prefix
		return wid.startsWith("sv_");
	}

	/** 获取当前用户上传的视频总数 */
	public int countVideoForUid(String uid, BrandEnum brand, List<SourceStatus> status) {
		return uploadVideoService.countVideos(uid, status, brand);
	}

	/** 获取当前用户上传的视频信息 */
	public List<ItemCard> getVideosForUid(String uid, BrandEnum brand, List<SourceStatus> status, int start, int size) {
		List<String> wids = uploadVideoService.getVideos(uid, status, brand, start, size);
		if (ArrayMapTools.isEmpty(wids))
			return null;
		return packageVideoCard(VideoRequest.of(uid, wids));
	}

	public VideoWrapper packageVideoWrapper(UploadVideo video) {
		if (null == video)
			return null;
		VideoWrapper vw = VideoWrapper.of(video);
		// 只有已上线的视频才可以播放
		if (SourceStatus.online.getCode() == video.getStatus()) {
			// 视频播放地址
			VideoTool.getVideoUrl(video, vw);
			// 图片
			VideoTool.getSnapshotUrl(video.getUid(), video, vw);
			return vw;
		}
		VideoTool.getUploadVideoDefault(vw);
		return vw;
	}

	public VideoWrapper packageVideoWrapper(String uid, UploadVideo video) {
		VideoWrapper vw = VideoWrapper.of(video);
		// 只有已上线的视频才可以播放
		if (SourceStatus.online.getCode() == video.getStatus()) {
			// 视频播放地址
			VideoTool.getVideoUrl(video, vw);
			// 图片
			VideoTool.getSnapshotUrl(uid, video, vw);
			return vw;
		}
		VideoTool.getUploadVideoDefault(vw);
		return vw;
	}

	/** 将当前视频信息包装成卡片信息 */
	public List<ItemCard> packageVideoCard(VideoRequest request) {
		if (ArrayMapTools.isEmpty(request.getWids()))
			return null;
		List<UploadVideo> videos = uploadVideoService.getVideoInfo(request.getWids());
		if (ArrayMapTools.isEmpty(videos))
			return null;
		return packageVideoCard(request.getUid(), OrderTool.keepOrder(request.getWids(), videos, UploadVideo::getWid));
	}

	/** 将当前视频信息包装成卡片信息 */
	public List<ItemCard> packageVideoCard(String uid, List<UploadVideo> videos) {
		if (ArrayMapTools.isEmpty(videos))
			return null;
		List<String> wids = videos.stream().map(it -> it.getWid()).collect(Collectors.toList());
		List<UploadVideoScore> scoreList = uploadVideoScoreService.mutiGetVideoScores(wids);
		Map<String, UploadVideoScore> scoreMap = Maps.newHashMap();
		if (ArrayMapTools.isNotEmpty(scoreList)) {
			scoreList.forEach(it -> {
				scoreMap.put(it.getWid(), it);
			});
		}

		List<ItemCard> rsList = Lists.newArrayList();
		for (UploadVideo uv : videos) {
			VideoWrapper wrapper = packageVideoWrapper(uid, uv);
			if (scoreMap.containsKey(uv.getWid())) {
				UploadVideoScore score = scoreMap.get(uv.getWid());
				wrapper.setScore(score.getScore());
				wrapper.setRemark(score.getRemark());
			}
			rsList.add(ItemCard.of(wrapper));
		}
		return rsList;
	}
}
