package com.jz.jar.business.proxy;

import java.util.List;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.jz.common.utils.collection.ArrayMapTools;
import com.jz.common.utils.jz.IdTools;
import com.jz.common.utils.text.StringTools;
import com.jz.jar.business.request.ReservationRequest;
import com.jz.jar.business.response.ReservationResponse;
import com.jz.jar.franchise.common.GlobalChannel;
import com.jz.jar.franchise.service.FUserFlowNoticeService;
import com.jz.jar.franchise.service.FUserRoleService;
import com.jz.jar.franchise.service.MarketCasePoolService;
import com.jz.jar.franchise.service.ParentInfoService;
import com.jz.jar.franchise.service.SchoolService;
import com.jz.jar.franchise.service.StudentInfoService;
import com.jz.jar.franchise.wrapper.CaseChildWrapper;
import com.jz.jar.media.enums.BrandEnum;
import com.jz.jar.media.service.ReservationService;
import com.jz.jooq.franchise.tables.pojos.MarketCasePool;
import com.jz.jooq.franchise.tables.pojos.ParentInfo;
import com.jz.jooq.franchise.tables.pojos.School;
import com.jz.jooq.franchise.tables.pojos.StudentInfo;
import com.jz.jooq.franchise.tables.records.FuserFlowNoticeRecord;
import com.jz.jooq.media.tables.records.ReservationRecord;

/**
 * @Title ReservationBusinessProxy
 * @Package com.jz.business.proxy
 * @author tangjunfeng
 * @date 2018年11月19日 下午6:52:47
 * @version V1.0
 */
@Lazy
@Service
public class ReservationBusinessProxy {

	@Autowired
	private SchoolService schoolService;
	@Autowired
	private ParentInfoService parentInfoService;
	@Autowired
	private ReservationService reservationService;
	@Autowired
	private MarketCasePoolService marketCasePoolService;
	@Autowired
	private FUserFlowNoticeService fUserFlowNoticeService;
	@Autowired
	private FUserRoleService fUserRoleService;
	@Autowired
	private StudentInfoService studentInfoService;

	@Autowired
	private ParentInfoBusinessProxy parentInfoBusinessProxy;
	
	/** 申请预约试听 */
	@Transactional
	public ReservationResponse apply(ReservationRequest request) {
		School school = schoolService.getSchool(request.getSchoolId());
		if (null == school || school.getCode() < 100 || !school.getBrandId().equals(request.getBrand().name()))
			return ReservationResponse.failed("无效的中心");

		ParentInfo parentInfo = null;
		if (StringTools.isNotEmptyAndBlank(request.getUid())) {
			parentInfo = parentInfoService.getParentInfoForId(request.getUid());
		} else if (StringTools.isNotEmptyAndBlank(request.getPhone())) {
			// 根据手机号获取用户信息，如果用户不存在则创建
			Pair<ParentInfo, Boolean> pair = parentInfoBusinessProxy.getOrCreateParentInfo(request.getPhone(), false);
			parentInfo = pair.getLeft();
		}
		if (null == parentInfo)
			return ReservationResponse.failed("无效的用户");
		// 一个用户一个品牌最多预约三个
		if (reservationService.countValidReservation(request.getBrand(), parentInfo.getPuid()) >= 3)
			return ReservationResponse.failed("已达到预约上限");
		// 当前中心已经预约过
		if (reservationService.existValidReservation(parentInfo.getPuid(), school.getId()))
			return ReservationResponse.exist();
		
		// 保存预约信息
		this.saveReservation(request.getBrand(), parentInfo, school);
		
		Set<String> fuids = Sets.newHashSet(
				fUserRoleService.getEnableUidsByRole(school.getId(), Lists.newArrayList("master", "fu_master")));
		MarketCasePool mcp = marketCasePoolService.getMarketCaseByPhone(school.getId(), parentInfo.getPhone());
		String caseId = null;
		if (mcp == null) { // 创建例子信息
			caseId = this.createMarketCase(request, school, parentInfo);
		} else {
			caseId = mcp.getId();
			if (StringUtils.isNotEmpty(mcp.getAdviser())) {
				fuids.add(mcp.getAdviser());
			}
		}
		// 这里要发送消息给对应的人
		if (ArrayMapTools.isNotEmpty(fuids)) {
			List<FuserFlowNoticeRecord> notices = Lists.newArrayList();
			String content = "手机号为" + parentInfo.getPhone() + "的用户预约的您的试听，请尽快联系!";
			for (String fuid : fuids) {
				notices.add(new FuserFlowNoticeRecord(null, fuid, school.getId(), "marketCase", content, caseId,
						System.currentTimeMillis(), 0));
			}
			fUserFlowNoticeService.mutiCreateNotice(notices);
		}
		return ReservationResponse.ok();
	}

	/** 根据孩子信息创建例子 */
	private String createMarketCase(ReservationRequest request, School school, ParentInfo parentInfo) {
		// 这里要create
		List<CaseChildWrapper> childs = Lists.newArrayList();
		List<StudentInfo> sis = null;
		// 未使用当前选择的学员才查询
		if (null == request.getSuid())
			sis = studentInfoService.getStudents(parentInfo.getPuid());
		// 将当前用户信息加入到例子中
		if (request.getIsExistStudent()) {
			if (null != request.getSuid()) { // 使用当前选择的学员
				childs.add(CaseChildWrapper.of(studentInfoService.getStudent(request.getSuid())));
			} else { // 手动填的学员信息
				childs.add(CaseChildWrapper.of(request.getName(), request.getSex(), request.getBirthday()));
				// 无名字和性别相同的孩子则创建
				if (!searchStudent(sis, request)) {
					studentInfoService.refreshStudent(parentInfo.getPuid(), IdTools.encode(parentInfo.getPuid()),
							request.getName(), null, request.getSex(), request.getBirthday(), null);
				}
			}
		} else if (ArrayMapTools.isNotEmpty(sis)) {
			// 配置例子中的学员信息
			for (StudentInfo si : sis) {
				childs.add(CaseChildWrapper.of(si));
			}
		}
		// 创建例子
		String caseId = IdTools.encode(parentInfo.getPuid());
		marketCasePoolService.createCase(caseId, "system", school.getId(), GlobalChannel.appYuyueChannelId,
				parentInfo.getPhone(), "", "", childs, parentInfo.getName(), parentInfo.getRelation(),
				parentInfo.getAddress(), parentInfo.getDistrict(), "A", 0l);
		return caseId;
	}

	/** 创建预约试听信息 */
	private void saveReservation(BrandEnum brand, ParentInfo parentInfo, School school) {
		ReservationRecord record = new ReservationRecord();
		record.setBrand(brand.name());
		record.setUid(parentInfo.getPuid());
		record.setPhone(parentInfo.getPhone());
		record.setName(parentInfo.getName());
		record.setSchoolId(school.getId());
		record.setProv(school.getProv());
		record.setCenterName(school.getCenterName());
		record.setCenterAddress(school.getCenterAddress());
		// 1直营 2加盟
		record.setCenterType(getCenterType(school.getType()));
		record.setStatus(1);
		record.setCreateTime(System.currentTimeMillis());
		reservationService.saveRecord(record);
	}

	private String getCenterType(int type) {
		if (1 == type)
			return "直营";
		if (2 == type)
			return "加盟";
		return "其它";
	}
	
	private boolean searchStudent(List<StudentInfo> studentInfos, ReservationRequest request) {
		if (ArrayMapTools.isEmpty(studentInfos) || !request.getIsExistStudent())
			return false;
		if (StringTools.isEmpty(request.getName()) || StringTools.isEmpty(request.getSex()))
			return false;
		for (StudentInfo si : studentInfos) {
			if (StringTools.allEmpty(si.getName(), si.getEngName()))
				continue;
			if (StringTools.contains(request.getName(), si.getName(), si.getEngName())
					&& si.getSex().equals(request.getSex())) {
				return true;
			}
		}
		return false;
	}
}
