package com.jz.jar.search.service;

import java.lang.reflect.Type;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import javax.annotation.PostConstruct;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import com.aliyun.opensearch.DocumentClient;
import com.aliyun.opensearch.OpenSearchClient;
import com.aliyun.opensearch.SearcherClient;
import com.aliyun.opensearch.sdk.generated.OpenSearch;
import com.aliyun.opensearch.sdk.generated.commons.OpenSearchClientException;
import com.aliyun.opensearch.sdk.generated.commons.OpenSearchException;
import com.aliyun.opensearch.sdk.generated.commons.OpenSearchResult;
import com.aliyun.opensearch.sdk.generated.document.Command;
import com.aliyun.opensearch.sdk.generated.search.Config;
import com.aliyun.opensearch.sdk.generated.search.Order;
import com.aliyun.opensearch.sdk.generated.search.SearchFormat;
import com.aliyun.opensearch.sdk.generated.search.SearchParams;
import com.aliyun.opensearch.sdk.generated.search.Sort;
import com.aliyun.opensearch.sdk.generated.search.SortField;
import com.aliyun.opensearch.sdk.generated.search.general.SearchResult;
import com.google.common.collect.Lists;
import com.google.gson.reflect.TypeToken;
import com.jz.common.utils.collection.ArrayMapTools;
import com.jz.common.utils.json.GsonTools;
import com.jz.common.utils.text.StringTools;
import com.jz.jar.media.enums.PlaylistType;
import com.jz.jar.search.entity.business.tomato.ETomatoSearchBean;
import com.jz.jar.search.entity.business.tomato.TomatoSearchType;
import com.jz.jar.search.entity.push.AliOpenSearchPushTemplate;
import com.jz.jar.search.entity.result.AliOpenSearchBaseResult;
import com.jz.jar.search.entity.result.AliOpenSearchResult;

@Lazy
@Service
public class TomatoSearchService extends AbstractOpenSearchService {

	private static Logger logger = LoggerFactory.getLogger(TomatoSearchService.class);
	
	@Value("${aliyun.open.search.tomato.app.name}")
	private String appName;
	@Value("${aliyun.open.search.tomato.table.name}")
	private String tableName;

	private static Config config;
	private static final Type type = new TypeToken<AliOpenSearchBaseResult<ETomatoSearchBean>>() {}.getType();

	@PostConstruct
	public void init() {
		// 定义Config对象，用于设定config子句参数，分页或数据返回格式，指定应用名等等
		config = new Config(Lists.newArrayList(appName));
		// 设置返回格式为json,目前只支持返回xml和json格式，暂不支持返回fulljson类型
		config.setSearchFormat(SearchFormat.JSON);
		// 设置搜索结果返回应用中哪些字段
		// 返回的attrs : "xx\ta\tb" 是类似这种结果，实际值是 [xx, a, b]
		config.setFetchFields(Lists.newArrayList("id", "type"));
	}

	public AliOpenSearchResult<ETomatoSearchBean> search(String name, PlaylistType filterType, Collection<String> attrs,
			int start, int size) throws OpenSearchException, OpenSearchClientException {
		// 创建SearcherClient对象，并以OpenSearchClient对象作为构造参数
		SearcherClient searcherClient = new SearcherClient(openSearchClient);
		config.setStart(start).setHits(size);
		// 创建参数对象
		SearchParams searchParams = new SearchParams(config);
		// 设置查询子句，若需多个索引组合查询，需要setQuery处合并，否则若设置多个setQuery后面的会替换前面查询
		if (StringTools.isNotEmptyAndBlank(name)) {
			searchParams.setQuery("name:'" + name + "'");
		}
		StringBuffer filter = new StringBuffer();
		// 类型筛选
		if (null != filterType) {
			filter.append("in(type, \"");
			if (PlaylistType.video == filterType) {
				filter.append(TomatoSearchType.getVideoCode("|"));
			} else {
				filter.append(TomatoSearchType.getImageTxtCode("|"));
			}
			filter.append("\")");
		}
		// 属性筛选
		if (ArrayMapTools.isNotEmpty(attrs)) {
			if (filter.length() > 0)
				filter.append(" AND ");
			for (String attrId : attrs) {
				filter.append("attrs=\"").append(attrId).append("\"").append(" AND ");
			}
			filter.delete(filter.length() - 5, filter.length());
		}
		if (filter.length() > 0)
			searchParams.setFilter(filter.toString());
		// 设置排序
		Sort sorter = new Sort();
		// 根据类型排序
		sorter.addToSortFields(new SortField("type", Order.INCREASE));
		// 更新时间
		sorter.addToSortFields(new SortField("timer", Order.DECREASE));
		// 根据搜索词相关度高在前
		sorter.addToSortFields(new SortField("RANK", Order.DECREASE));
		searchParams.setSort(sorter);
		SearchResult searchResult = searcherClient.execute(searchParams);
		AliOpenSearchBaseResult<ETomatoSearchBean> gsonResult = GsonTools.gson.fromJson(searchResult.getResult(), type);
		return gsonResult.getResult();
	}

	public boolean searchPushCommand(List<AliOpenSearchPushTemplate<ETomatoSearchBean>> pojos) {
		if (ArrayMapTools.isEmpty(pojos))
			return true;
		return searchPushCommand(GsonTools.gson.toJson(pojos));
	}

	@Override
	public boolean searchPushCommand(String docs) {
		if (StringTools.isEmpty(docs))
			return true;
		// 定义DocumentClient对象添加json格式doc数据批量提交
		DocumentClient documentClient = new DocumentClient(openSearchClient);
		try {
			// 执行推送操作
			OpenSearchResult osr = documentClient.push(docs, appName, tableName);
			// 判断数据是否推送成功，主要通过判断2处，第一处判断用户方推送是否成功，第二处是应用控制台中有无报错日志
			// 用户方推送成功后，也有可能在应用端执行失败，此错误会直接在应用控制台错误日志中生成，比如字段内容转换失败
			if (osr.getResult().equalsIgnoreCase("true"))
				return true;
			logger.error("add doc failed, infos : {}, result : {}", docs, osr.getResult());
		} catch (OpenSearchException | OpenSearchClientException e) {
			logger.error(e.getMessage(), e);
		}
		// 执行失败抛出异常
		logger.error("更新番茄田搜索内容失败：\n" + docs);
		throw new RuntimeException("更新番茄田搜索内容失败");
	}

	public static void main(String[] args) throws OpenSearchException, OpenSearchClientException {
		TomatoSearchService ts = new TomatoSearchService();
		ts.appName = "tomato_app_test";
		ts.tableName = "tomato_context_resources";
		ts.init();

		String accessKeyId = "LTAIXIOV1fsWtqzM";
		String accessKeySecret = "aEnYKzdq9wZS5MLrJw6Dj8amkW3XK9";
		String host = "http://opensearch-cn-beijing.aliyuncs.com";
		OpenSearch openSearch = new OpenSearch(accessKeyId, accessKeySecret, host);
		ts.openSearchClient = new OpenSearchClient(openSearch);
		
		List<AliOpenSearchPushTemplate<ETomatoSearchBean>> pojos = Lists.newArrayList();
		pojos.add(AliOpenSearchPushTemplate.of(Command.DELETE, new ETomatoSearchBean().setId("ystllvtx4yn0ekm6").setName("千岛之国的神秘音画").setTimer(1560218508337L).setAttrs(Arrays.asList("aa", "hh", "xx", "T1", "T2")).setType(TomatoSearchType.video_playlist)));
		pojos.add(AliOpenSearchPushTemplate.of(Command.DELETE, new ETomatoSearchBean().setId("xx_imt").setName("饿死我算啦").setTimer(0L).setAttrs(Arrays.asList("aa", "hh", "T1", "T2")).setType(TomatoSearchType.img_txt_playlist)));
		pojos.add(AliOpenSearchPushTemplate.of(Command.DELETE, new ETomatoSearchBean().setId("uy56unuxsissgqql").setName("金刚葫芦娃").setTimer(1560218528337L).setAttrs(Arrays.asList("hh", "xx", "T1", "T2")).setType(TomatoSearchType.video_playlist)));
		pojos.add(AliOpenSearchPushTemplate.of(Command.DELETE, new ETomatoSearchBean().setId("xha4bu1wl1vd03es").setName("福禄小金刚").setTimer(1560218838337L).setAttrs(Arrays.asList("xx", "T1", "T2")).setType(TomatoSearchType.video_playlist)));
		pojos.add(AliOpenSearchPushTemplate.of(Command.DELETE, new ETomatoSearchBean().setId("1").setName("变态小辣椒").setTimer(1560218508737L).setAttrs(Arrays.asList("aa", "T1", "T2")).setType(TomatoSearchType.img_txt)));
		pojos.add(AliOpenSearchPushTemplate.of(Command.DELETE, new ETomatoSearchBean().setId("2").setName("辣椒小变态").setTimer(1560218505337L).setAttrs(Arrays.asList("hh", "aa", "T1", "T2")).setType(TomatoSearchType.img_txt)));
		pojos.add(AliOpenSearchPushTemplate.of(Command.DELETE, new ETomatoSearchBean().setId("99e196f0c26fc1a6cf").setName("小小辣椒").setTimer(1560218505337L).setAttrs(Arrays.asList("aa", "xx", "T1", "T2")).setType(TomatoSearchType.video)));
		pojos.add(AliOpenSearchPushTemplate.of(Command.DELETE, new ETomatoSearchBean().setId("99ef6a664f8d520138").setName("小雍和宫大和尚").setTimer(1560218505337L).setAttrs(Arrays.asList("aa", "xx", "T1", "T2")).setType(TomatoSearchType.video)));
		pojos.add(AliOpenSearchPushTemplate.of(Command.DELETE, new ETomatoSearchBean().setId("99f4b53a68d9005634").setName("一不小心不见啦").setTimer(1560218505337L).setAttrs(Arrays.asList("hh", "xx", "T1", "T2")).setType(TomatoSearchType.video)));
		System.out.println("result : " + ts.searchPushCommand(pojos));
		
		System.out.println("result : " + GsonTools.gson.toJson(ts.search(null, PlaylistType.img_txt, null, 0, 5)));
	}
}