package com.jz.jar.business.proxy.common;

import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.google.common.collect.Maps;
import com.jz.common.utils.collection.ArrayMapTools;
import com.jz.common.utils.text.StringTools;
import com.jz.jar.business.request.NextPlayVideoRequest;
import com.jz.jar.business.request.VideoRequest;
import com.jz.jar.business.wrapper.VideoWrapper;
import com.jz.jar.media.enums.BrandEnum;
import com.jz.jar.media.enums.DbField;
import com.jz.jar.media.service.MediaVideoService;
import com.jz.jar.media.service.PlaylistRelationService;
import com.jz.jar.media.service.PlaylistService;
import com.jz.jar.media.service.PlaylistVideoService;
import com.jz.jar.media.service.TaskPlayHistoryService;
import com.jz.jar.media.service.UserPlayHistoryService;
import com.jz.jooq.media.tables.pojos.TaskPlayHistory;
import com.jz.jooq.media.tables.pojos.UserPlayHistory;

/**
 * @Title PlaylistVideoProxy
 * @Package com.jz.jar.media.proxy
 * @author tangjunfeng
 * @date 2018年8月8日 下午6:26:16
 * @version V1.0
 */
@Lazy
@Service
public class PlaylistVideoProxy {

	private static final Random random = new Random();

	@Autowired
	private MediaVideoService mediaVideoService;
	@Autowired
	private PlaylistService playlistService;
	@Autowired
	private UserPlayHistoryService userPlayHistoryService;
	@Autowired
	private PlaylistVideoService playlistVideoService;
	@Autowired
	private PlaylistRelationService playlistRelationService;
	@Autowired
	private TaskPlayHistoryService taskPlayHistoryService;
	@Autowired
	private VideoProxy videoProxy;

	public String getNextPlayVideo(String pid, String wid) {
		List<String> wids = playlistVideoService.findWids(pid);
		// 当前专辑是空的 || 当前专辑内无当前视频
		if (ArrayMapTools.isEmpty(wids) || !wids.contains(wid))
			return wid;
		// 当前视频为最后一集
		String lastWid = wids.get(wids.size() - 1);
		if (lastWid.equals(wid))
			return wids.get(0);
		// 读取当前视频的下一集
		for (int i = 0; i < wids.size(); i++) {
			if (wid.equals(wids.get(i))) {
				return wids.get(i + 1);
			}
		}
		return wid;
	}

	/** 获取专辑下一个需要播放的视频id */
	public String getNextPlayVideoForHistory(String uid, String pid, BrandEnum brand) {
		if (StringTools.isNotEmpty(uid)) {
			// 获取当前专辑的播放历史
			UserPlayHistory history = userPlayHistoryService.getHistory(uid, pid, brand);
			if (null != history) {
				return history.getNextWid();
			}
		}
		return playlistService.getWidForPlaylist(pid);
	}

	/** 获取专辑下一个需要播放的视频 */
	public VideoWrapper getNextPlayVideoForHistory(String uid, String pid, String firstWid, BrandEnum brand,
			boolean isWithUrl) {
		int playlength = 0;
		String nextWid = firstWid, historyWid = null;
		if (null != uid) {
			// 获取当前专辑的播放历史
			UserPlayHistory history = userPlayHistoryService.getHistory(uid, pid, brand);
			if (null != history) {
				playlength = history.getPlayLength();
				historyWid = history.getWid();
				nextWid = history.getNextWid();
			}
		}
		if (null == nextWid) {
			nextWid = playlistService.getWidForPlaylist(pid);
		}
		VideoWrapper videoInfo = videoProxy.getVideoInfo(VideoRequest.of(uid, pid, nextWid).setWithUrl(isWithUrl),
				brand);
		if (null != videoInfo) {
			videoInfo.setPlayLength(nextWid.equals(historyWid) ? playlength : 0);
		}
		return videoInfo;
	}

	/** 获取专辑下一个需要播放的视频 */
	public Map<String, VideoWrapper> getNextPlayVideoForHistory(BrandEnum brand, NextPlayVideoRequest request) {
		Map<String, String> pid2Wid = Maps.newHashMap();
		// 获取所有的专辑id
		Set<String> pids = request.getPlaylist().stream().map(it -> it.getPid()).collect(Collectors.toSet());
		// 存在学员id则检查是否需要检查指定任务
		if (null != request.getSuid()) {
			DbField field = null;
			if (request.getIsSpecial()) { // 专项练习
				field = DbField.special;
			} else if (request.getIsEverday()) { // 今日任务
				field = DbField.everday;
			}
			if (null != field) { // 存在指定任务
				String tk = taskPlayHistoryService.getTk(field);
				List<TaskPlayHistory> playHistories = taskPlayHistoryService.findPlayHistories(tk, request.getSuid(),
						pids);
				if (ArrayMapTools.isNotEmpty(playHistories)) {
					playHistories.forEach(it -> pid2Wid.put(it.getPid(), it.getWid()));
					pids.removeAll(pid2Wid.keySet());
				}
			}
		}

		Map<String, Integer> wid2Playlength = Maps.newHashMap();
		if (!pids.isEmpty() && null != request.getUid()) {
			// 获取当前专辑的播放历史
			List<UserPlayHistory> histories = userPlayHistoryService.getHistories(request.getUid(), pids);
			if (ArrayMapTools.isNotEmpty(histories)) {
				histories.forEach(it -> {
					pid2Wid.put(it.getPid(), it.getNextWid());
					wid2Playlength.put(it.getNextWid(), it.getPlayLength());
				});
				pids.removeAll(pid2Wid.keySet());
			}
		}

		// 指定任务和历史都不存在,则播放当前专辑的指定视频
		if (!pids.isEmpty()) {
			request.getPlaylist().forEach(it -> {
				if (pids.contains(it.getPid())) {
					pid2Wid.put(it.getPid(), it.getWid());
					pids.remove(it.getPid());
				}
			});
		}
		
		// 过滤已下线的视频
		List<String> wids = mediaVideoService.filterOnlineVideo(pid2Wid.values(), brand);
		if (ArrayMapTools.isEmpty(wids)) {
			request.getPlaylist().forEach(it -> {
				pid2Wid.put(it.getPid(), it.getWid());
			});
		} else if (wids.size() != pid2Wid.size()) {
			request.getPlaylist().forEach(it -> {
				if (!wids.contains(pid2Wid.get(it.getPid()))) {
					pid2Wid.put(it.getPid(), it.getWid());
				}
			});
		}

		Map<String, VideoWrapper> pid2VideoCard = Maps.newHashMap();
		List<VideoWrapper> videoInfo = videoProxy
				.getVideoInfoNotPid(VideoRequest.of(request.getUid(), pid2Wid.values()));
		if (ArrayMapTools.isEmpty(videoInfo))
			return pid2VideoCard;
		// 补充视频的播放时长
		Map<String, VideoWrapper> wid2Video = Maps.newHashMap();
		videoInfo.forEach(it -> {
			wid2Video.put(it.getWid(), it);
			if (wid2Playlength.containsKey(it.getWid())) {
				it.setPlayLength(wid2Playlength.get(it.getWid()));
			} else {
				it.setPlayLength(0);
			}
		});
		// 补充pid
		pid2Wid.forEach((tp, tw) -> {
			VideoWrapper tvw = wid2Video.get(tw);
			if (null != tvw)
				tvw = tvw.clone().setPid(tp);
			pid2VideoCard.put(tp, tvw);
		});
		return pid2VideoCard;
	}

	@Transactional
	public void addWatchCnt(String wid, String pid, BrandEnum brand) {
		// 随机生成200以内的一个播放数
		int randomWatchCnt = random.nextInt(200);
		// 添加视频播放数
		mediaVideoService.addWatchCnt(wid, brand, randomWatchCnt);
		// 添加专辑播放数
		playlistService.addWatchCnt(pid, brand, randomWatchCnt);
		// 专辑关系添加播放数
		playlistRelationService.addWatchCnt(pid, randomWatchCnt);
	}
}
