package com.jz.jar.business.proxy.common;

import java.lang.reflect.Type;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.gson.reflect.TypeToken;
import com.jz.common.utils.collection.ArrayMapTools;
import com.jz.common.utils.collection.OrderTool;
import com.jz.common.utils.json.GsonTools;
import com.jz.common.utils.text.StringTools;
import com.jz.jar.business.bean.AliyunBean;
import com.jz.jar.business.request.UserRequest;
import com.jz.jar.business.request.WorksRequest;
import com.jz.jar.business.wrapper.StudentWrapper;
import com.jz.jar.business.wrapper.TeacherWrapper;
import com.jz.jar.business.wrapper.UserWrapper;
import com.jz.jar.business.wrapper.WWorkTopicTheme;
import com.jz.jar.business.wrapper.WorksInfoWrapper;
import com.jz.jar.business.wrapper.WorksTreasureWrapper;
import com.jz.jar.franchise.service.CourseTomatoTopicService;
import com.jz.jar.franchise.service.FUserInfoService;
import com.jz.jar.franchise.service.LessonService;
import com.jz.jar.franchise.service.SchoolService;
import com.jz.jar.franchise.service.StudentSchoolService;
import com.jz.jar.media.enums.BrandEnum;
import com.jz.jar.media.enums.DbField;
import com.jz.jar.media.service.CommentInfoService;
import com.jz.jar.media.service.WorksAuthorService;
import com.jz.jar.media.service.WorksInfoService;
import com.jz.jar.media.service.WorksLikeService;
import com.jz.jar.media.service.WorksTreasureService;
import com.jz.jooq.franchise.tables.pojos.CourseTomatoTopic;
import com.jz.jooq.franchise.tables.pojos.FuserInfo;
import com.jz.jooq.media.tables.pojos.WorksAuthor;
import com.jz.jooq.media.tables.pojos.WorksInfo;
import com.jz.jooq.media.tables.pojos.WorksTreasure;

@Lazy
@Service
public class WorksInfoProxy {

	private static final Type pictureType = new TypeToken<List<String>>() {
	}.getType();

	@Autowired
	private CourseTomatoTopicService courseTomatoTopicService;
	@Autowired
	private WorksInfoService worksInfoService;
	@Autowired
	private WorksLikeService worksLikeService;
	@Autowired
	private UserProxy userProxy;
	@Autowired
	private CommentInfoService commentInfoService;
	@Autowired
	private WorksAuthorService worksAuthorService;
	@Autowired
	private SchoolService schoolService;
	@Autowired
	private LessonService lessonService;
	@Autowired
	private FUserInfoService teacherService;
	@Autowired
	private WorksTreasureService worksTreasureService;
	@Autowired
	private StudentSchoolService studentSchoolService;

	public List<WorksInfoWrapper> getSimpleWorksInfo(BrandEnum brand, WorksRequest request) {
		if (ArrayMapTools.isEmpty(request.getWorksId()))
			return null;
		List<WorksInfo> worksInfo = worksInfoService.findSimpleWorksInfo(request.getWorksId());
		if (ArrayMapTools.isEmpty(worksInfo))
			return null;
		Map<String, UserWrapper> uid2User = null;
		Map<String, String> schoolId2Name = null;
		Map<String, StudentWrapper> suid2Student = null;
		Map<String, Integer> likeCount = null;
		List<String> likes = null;
		Map<String, Pair<String, String>> wkid2Puid2Suid = Maps.newHashMap();
		Map<String, List<String>> workId2Suids = Maps.newHashMap();
		Map<String, FuserInfo> lessonId2Teacher = Maps.newHashMap();
		if (request.getWithExtended()) {
			// 获取的作品id
			List<String> wkids = Lists.newArrayList();
			Map<String, Collection<String>> school2Lessons = Maps.newHashMap();
			worksInfo.forEach(it -> {
				wkids.add(it.getId());
				if (school2Lessons.containsKey(it.getSchoolId())) {
					school2Lessons.get(it.getSchoolId()).add(it.getLessonId());
				} else {
					school2Lessons.put(it.getSchoolId(), Lists.newArrayList(it.getLessonId()));
				}
			});
			List<WorksAuthor> worksAuthors = worksAuthorService.findWorkAuthor(brand, wkids);
			Set<String> puids = Sets.newHashSet(), suids = Sets.newHashSet(), schoolIds = Sets.newHashSet();
			worksAuthors.forEach(v -> {
				wkid2Puid2Suid.put(v.getWorkId(), Pair.of(v.getPuid(), v.getSuid()));
				puids.add(v.getPuid());
				suids.add(v.getSuid());
				schoolIds.add(v.getSchoolId());
				if (workId2Suids.containsKey(v.getWorkId())) {
					workId2Suids.get(v.getWorkId()).add(v.getSuid());
				} else {
					workId2Suids.put(v.getWorkId(), Lists.newArrayList(v.getSuid()));
				}
			});
			schoolId2Name = schoolService.mutiGetId2NameMap(schoolIds);
			// get user info
			uid2User = userProxy.getSimpleParentInfoHas(brand, UserRequest.of(puids));
			suid2Student = userProxy.getStudentSimpleInfoHasForSuid(suids, brand, request.getUid());
			// get art like cnt
			likeCount = worksLikeService.getLikeCount(request.getWorksId());
			// get like art ids
			if (null != request.getUid())
				likes = worksLikeService.screenLike(request.getWorksId(), request.getUid());

			if (ArrayMapTools.isNotEmpty(school2Lessons)) {
				Map<String, String> lessonId2TeacherId = lessonService.getLessonId2Teacher(school2Lessons);
				Map<String, FuserInfo> uMap = teacherService.getTeachers(lessonId2TeacherId.values()).stream()
						.collect(Collectors.toMap(m -> m.getUid(), m -> m));
				for (Map.Entry<String, String> m : lessonId2TeacherId.entrySet()) {
					lessonId2Teacher.put(m.getKey(), uMap.get(m.getValue()));
				}
			}
		}
		Map<String, WorksTreasureWrapper> teasureMap = null;
		if (request.getWithTreasure()) {
			List<WorksTreasure> wts = worksTreasureService.mutiGetPayedTreasureByWorkIds(request.getWorksId());
			if (ArrayMapTools.isNotEmpty(wts)) {
				teasureMap = Maps.newHashMap();
				for (WorksTreasure wt : wts) {
					WorksTreasureWrapper wtw = WorksTreasureWrapper.of(wt);
					if (StringUtils.equals(wt.getPuid(), request.getUid())) {
						wtw.setCertPic(wt.getCertPic());
					}
					teasureMap.put(wt.getWorkId(), wtw);
				}
			}
		}
		// package wrapper
		List<WorksInfoWrapper> wrappers = Lists.newArrayList();
		for (WorksInfo workInfo : worksInfo) {
			WorksInfoWrapper wrapper = WorksInfoWrapper.of(workInfo);
			// old data
			if (null != workInfo.getPic() && workInfo.getPic().startsWith("http://")) {
				wrapper.setPic(workInfo.getPic());
			} else {
				wrapper.setPic(AliyunBean.getImagesUrl(workInfo.getPic()));
			}
			if (request.getWithPictures()) {
				if (StringTools.isNotEmptyAndBlank(workInfo.getPictures())) {
					List<String> pics = GsonTools.gson.fromJson(workInfo.getPictures(), pictureType);
					if (null != workInfo.getPic() && workInfo.getPic().startsWith("http://")) {
						wrapper.setPictures(pics);
					}
					// 可展示图片
					List<String> pictures = Lists.newArrayList();
					pics.forEach(it -> pictures.add(AliyunBean.getImagesUrl(it)));
					wrapper.setPictures(pictures);
				}
			}
			if (request.getWithExtended()) {
				Pair<String, String> pair = wkid2Puid2Suid.get(workInfo.getId());
				if (pair != null) {
					// set user info
					wrapper.setUser(uid2User.get(pair.getKey()));
					wrapper.setStudent(suid2Student.get(pair.getValue()));
					// set like cnt
					wrapper.setLikeCnt(MapUtils.getInteger(likeCount, workInfo.getId(), 0));
					// set is like
					wrapper.setIsLike(null != likes && likes.contains(workInfo.getId()));
				}
				if (schoolId2Name.containsKey(workInfo.getSchoolId())) {
					wrapper.setSchoolName(schoolId2Name.get(workInfo.getSchoolId()));
				}
				if (lessonId2Teacher.containsKey(workInfo.getLessonId())) {
					wrapper.setTeacher(TeacherWrapper.of(lessonId2Teacher.get(workInfo.getLessonId())));
				}
			}
			if (request.getWithStudent()) {
				if (workId2Suids.containsKey(workInfo.getId())) {
					List<String> suids = workId2Suids.get(workInfo.getId());
					List<StudentWrapper> authors = Lists.newArrayList();
					for (String tmpSuid : suids) {
						authors.add(suid2Student.get(tmpSuid));
					}
					wrapper.setAuthors(authors);
				}
			}
			if (request.getWithTreasure() && teasureMap != null) {
				WorksTreasureWrapper treasure = teasureMap.get(workInfo.getId());
				if (treasure != null) {
					wrapper.setTreasure(treasure);
					wrapper.setTitle(treasure.getTitle());
				}
			}
			wrappers.add(wrapper);
		}
		return OrderTool.keepOrder(request.getWorksId(), wrappers, WorksInfoWrapper::getId);
	}

	public Map<String, WorksInfoWrapper> getSimpleWorksInfoHas(BrandEnum brand, WorksRequest request) {
		List<WorksInfoWrapper> artInfo = getSimpleWorksInfo(brand, request);
		if (ArrayMapTools.isEmpty(artInfo))
			return Maps.newHashMap();
		Map<String, WorksInfoWrapper> id2Art = Maps.newHashMap();
		artInfo.forEach(it -> id2Art.put(it.getId(), it));
		return id2Art;
	}

	public WorksInfoWrapper getWorkInfo(BrandEnum brand, WorksRequest request) {
		return getWorkInfoForRequest(brand, request.setWithExtended(true));
	}

	public WorksInfoWrapper getWorkInfoForRequest(BrandEnum brand, WorksRequest request) {
		WorksInfo workInfo = worksInfoService.getWorksInfo(request.getWorkId(), brand);
		if (null == workInfo)
			return null;
		// 非集创
		if (workInfo.getIsCollective() <= 0) {
			boolean isMaster = workInfo.getPuid().equals(request.getUid());
			if (workInfo.getIsOpen() <= 0 && !isMaster)
				return null; // is private & not master
		}
		WorksInfoWrapper wrapper = WorksInfoWrapper.of(workInfo);
		if (StringUtils.isNotEmpty(workInfo.getSchoolId())) {
			wrapper.setSchoolName(schoolService.getSchoolName(workInfo.getSchoolId()));
		}
		// set resource info
		boolean isOldData = null != workInfo.getPic() && workInfo.getPic().startsWith("http://");
		if (isOldData) {
			wrapper.setPic(workInfo.getPic()); // old data
		} else {
			wrapper.setPic(AliyunBean.getImagesUrl(workInfo.getPic()));
		}
		// set pictures
		if (StringTools.isNotEmptyAndBlank(workInfo.getPictures())) {
			List<String> pics = GsonTools.gson.fromJson(workInfo.getPictures(), pictureType);
			if (isOldData) {
				wrapper.setPictures(pics);
			} else {
				// 可展示图片
				List<String> pictures = Lists.newArrayList();
				pics.forEach(it -> pictures.add(AliyunBean.getImagesUrl(it)));
				wrapper.setPictures(pictures);
			}
		}

		if (request.getWithCreateInteractPics()) {
			if (StringUtils.isNotEmpty(workInfo.getCreatePics())) {
				List<String> createPics = GsonTools.gson.fromJson(workInfo.getCreatePics(), pictureType);
				wrapper.setCreatePics(
						createPics.stream().map(it -> AliyunBean.getImagesUrl(it)).collect(Collectors.toList()));
			}
			if (StringUtils.isNotEmpty(workInfo.getInteractPics())) {
				List<String> interactPics = GsonTools.gson.fromJson(workInfo.getInteractPics(), pictureType);
				wrapper.setInteractPics(
						interactPics.stream().map(it -> AliyunBean.getImagesUrl(it)).collect(Collectors.toList()));
			}
		}

		if (request.getWithExtended()) {
			// 作品专题
			CourseTomatoTopic topic = courseTomatoTopicService.getTopic(brand.name(), workInfo.getTid());
			if (null != topic)
				wrapper.setTheme(WWorkTopicTheme.of(topic).setPic(AliyunBean.getImagesUrl(topic.getPic())));

			// like count
			wrapper.setLikeCnt(worksLikeService.countForWorkId(workInfo.getId()));
			// comment cnt
			wrapper.setCommentCnt(commentInfoService.countCommentInfo(brand, DbField.works, workInfo.getId()));
			// current user is like
			wrapper.setIsLike(worksLikeService.existForIdAndUid(workInfo.getId(), request.getUid()));
			// 集创随便给一个user,兼容老版本
			String puid = worksAuthorService.getWorksAuthorParent(brand, workInfo.getId());
			wrapper.setUser(userProxy.getSimpleParentInfo(brand, UserRequest.of(request.getUid(), puid)));
		}

		// get student info
		if (request.getWithStudent()) {
			List<String> suids = worksAuthorService.getWorksAuthors(brand, workInfo.getId());
			List<StudentWrapper> authors = userProxy.getStudentSimpleInfo(suids, brand, request.getWithAuthorUser(),
					request.getUid());
			wrapper.setAuthors(authors);
			if (StringUtils.isNotEmpty(workInfo.getSchoolId()) && ArrayMapTools.isNotEmpty(authors)
					&& authors.size() == 1) { // 个创
				StudentWrapper author = authors.get(0);
				Long firstContractTime = studentSchoolService.getFirstContractTime(author.getSuid(),
						workInfo.getSchoolId());
				author.setJoinTime(userProxy.calLearnTimeStr(firstContractTime));
			}
		}

		if (request.getWithTreasure()) {
			WorksTreasure wt = worksTreasureService.getPayedTreasureByWorkId(workInfo.getId());
			if (wt != null) {
				WorksTreasureWrapper wtw = WorksTreasureWrapper.of(wt);
				if (StringUtils.equals(wt.getPuid(), request.getUid())) {
					wtw.setCertPic(wt.getCertPic());
				}
				wrapper.setTreasure(wtw);
				wrapper.setTitle(wt.getTitle());
			}
		}

		return wrapper;
	}
}
