package com.jz.jar.business.proxy.common;

import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.stream.Collectors;

import org.apache.commons.collections.MapUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.jz.common.utils.collection.ArrayMapTools;
import com.jz.common.utils.collection.OrderTool;
import com.jz.common.utils.text.StringTools;
import com.jz.jar.business.bean.AliyunBean;
import com.jz.jar.business.card.ItemCard;
import com.jz.jar.business.request.ImgTxtRequest;
import com.jz.jar.business.request.PlaylistRequest;
import com.jz.jar.business.wrapper.WImageText;
import com.jz.jar.media.enums.BrandEnum;
import com.jz.jar.media.service.PlaylistService;
import com.jz.jar.media.service.TomatoImageTextService;
import com.jz.jar.media.service.UserCollectImageTextService;
import com.jz.jooq.media.tables.pojos.TomatoImageText;

@Lazy
@Service
public class ImageTxtProxy {

	@Autowired
	private TomatoImageTextService tomatoImageTextService;
	@Autowired
	private PlaylistService playlistService;
	@Autowired
	private PlaylistProxy playlistProxy;
	@Autowired
	private UserCollectImageTextService userCollectImageTextService;

	/** 获取图文信息 */
	public WImageText getImageTextInfo(ImgTxtRequest request, BrandEnum brand) {
		TomatoImageText imageText = tomatoImageTextService.getOnlineImageText(request.getIid());
		if (null == imageText)
			return null;
		WImageText wp = WImageText.of(imageText, request.getPid()).setIsCollect(false);
		PlaylistRequest playlistRequest = PlaylistRequest.of(request.getUid(), request.getPid());
		wp.setPlaylist(playlistProxy.getSinglePlaylist(playlistRequest, brand));
		wp.setCollectCnt(userCollectImageTextService.countCollect(imageText.getId()));
		wp.setPic(AliyunBean.getImagesUrl(imageText.getPic()));
		if (StringTools.isNotEmpty(request.getUid()) && request.getIsWithCollect()) {
			wp.setIsCollect(userCollectImageTextService.isCollect(request.getUid(), request.getIid()));
		}
		return wp;
	}

	/** 获取图文列表信息 */
	public List<WImageText> getImageTextWps(ImgTxtRequest request, BrandEnum brand) {
		List<TomatoImageText> its = tomatoImageTextService.findOnlineImageText(request.getIids());
		if (ArrayMapTools.isEmpty(its))
			return null;
		Map<String, Integer> iid2Cnt = userCollectImageTextService.countCollect(request.getIids());
		List<String> ciids = Lists.newArrayList();
		if (StringTools.isNotEmpty(request.getUid()) && request.getIsWithCollect()) {
			List<String> tmp = userCollectImageTextService.filterCollectForUser(request.getUid(), request.getIids());
			if (null != tmp)
				ciids.addAll(tmp);
		}

		List<WImageText> wps = Lists.newArrayList();
		for (TomatoImageText tit : its) {
			WImageText imageText = WImageText.of(tit, request.getPid());
			imageText.setCollectCnt(MapUtils.getInteger(iid2Cnt, tit.getId(), 0));
			imageText.setPic(AliyunBean.getImagesUrl(tit.getPic()));
			imageText.setIsCollect(ciids.contains(tit.getId()));
			wps.add(imageText);
		}
		return OrderTool.keepOrder(request.getIids(), wps, WImageText::getIid);
	}

	public Map<String, WImageText> getImageTxtWpHas(ImgTxtRequest request, BrandEnum brand) {
		List<WImageText> wps = this.getImageTextWps(request, brand);
		if (ArrayMapTools.isEmpty(wps))
			return Maps.newHashMap();
		Map<String, WImageText> iid2Wp = Maps.newLinkedHashMap();
		wps.forEach(it -> iid2Wp.put(it.getIid(), it));
		return iid2Wp;
	}

	/** 获取图文列表卡片 */
	public List<ItemCard> getImageTextCards(ImgTxtRequest request, BrandEnum brand) {
		List<WImageText> its = this.getImageTextWps(request, brand);
		if (ArrayMapTools.isEmpty(its))
			return null;
		return its.stream().map(ItemCard::of).collect(Collectors.toList());
	}

	public Map<String, ItemCard> getImageTxtCardHas(ImgTxtRequest request, BrandEnum brand) {
		List<ItemCard> cards = this.getImageTextCards(request, brand);
		if (ArrayMapTools.isEmpty(cards))
			return Maps.newHashMap();
		Map<String, ItemCard> iid2Card = Maps.newLinkedHashMap();
		cards.forEach(it -> iid2Card.put(it.getImgTxt().getIid(), it));
		return iid2Card;
	}

	private static final Random random = new Random();

	@Transactional
	public void addImageTextWatchCnt(BrandEnum brand, String iid, String pid) {
		// 随机生成200以内的一个播放数
		int randomWatchCnt = random.nextInt(200);
		// 图文播放数
		tomatoImageTextService.addWatchCnt(iid, randomWatchCnt);
		// 当前专辑的播放数
		playlistService.addWatchCnt(pid, brand, randomWatchCnt);
	}
}
