package com.jz.jar.business.proxy.common;

import java.util.List;
import java.util.Map;
import java.util.function.Function;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.jz.common.utils.collection.ArrayMapTools;
import com.jz.common.utils.collection.OrderTool;
import com.jz.jar.business.bean.AliyunBean;
import com.jz.jar.business.wrapper.WebsiteBannerWrapper;
import com.jz.jar.media.enums.BrandEnum;
import com.jz.jar.media.service.MediaBannerService;
import com.jz.jooq.media.tables.pojos.MediaBanner;
import com.jz.jooq.website.tables.pojos.ActivityInfo;
import com.jz.jooq.website.tables.pojos.Art;
import com.jz.jooq.website.tables.pojos.Banner;
import com.jz.jooq.website.tables.pojos.NewsInfo;
import com.jz.website.enums.BannerEnum;
import com.jz.website.enums.BannerLocation;
import com.jz.website.enums.BannerType;
import com.jz.website.service.ActivityService;
import com.jz.website.service.ArtService;
import com.jz.website.service.NewsInfoService;

@Lazy
@Service
public class BannerProxy {

	private static final Logger logger = LoggerFactory.getLogger(BannerProxy.class);

	@Autowired
	private MediaBannerService mediaBannerService;
	@Autowired
	private NewsInfoService newsInfoService;
	@Autowired
	private ActivityService activityService;
	@Autowired
	private ArtService artService;
	
	public List<MediaBanner> getMediaBanner(BrandEnum brand) {
		List<MediaBanner> banners = mediaBannerService.findOnlineBanners(brand);
		if (ArrayMapTools.isEmpty(banners))
			return null;
		banners.forEach(it -> {
			it.setPic(AliyunBean.getImagesUrl(it.getPic()));
		});
		return banners;
	}

	public BannerEnum getWebsiteBannerId(String brand, String school, BannerLocation location) {
		if (location == BannerLocation.study)
			return BannerEnum.ho_study; // 只有总部才有study
		boolean isHeadOffice = brand.equals(school);
		if (location == BannerLocation.home)
			return isHeadOffice ? BannerEnum.ho_home : BannerEnum.school_home;
		if (location == BannerLocation.news)
			return isHeadOffice ? BannerEnum.ho_news : BannerEnum.school_news;
		return null;
	}

	public List<WebsiteBannerWrapper> packageWebsiteBanner(List<Banner> banners) {
		if (ArrayMapTools.isEmpty(banners))
			return null;
		List<String> orderKeys = Lists.newArrayList();
		Map<BannerType, List<String>> type2Ids = Maps.newHashMap();
		banners.forEach(it -> {
			BannerType type = BannerType.valueOf(it.getType());
			if (null == type) {
				logger.error("unknow banner type : [?]", type);
				return;
			}
			if (!type2Ids.containsKey(type))
				type2Ids.put(type, Lists.newArrayList());
			type2Ids.get(type).add(it.getDataId());
			// 添加用于底部保持输入顺序
			orderKeys.add(it.getType() + it.getDataId());
		});
		List<WebsiteBannerWrapper> resultMBanners = Lists.newArrayList();
		type2Ids.forEach((type, ids) -> {
			if (BannerType.news == type) {
				getMBanner(type, newsInfoService.getBannerInfo(ids), NewsInfo::getId, NewsInfo::getTitle,
						NewsInfo::getBannerPic, resultMBanners);
				return;
			}

			if (BannerType.activity == type) {
				getMBanner(type, activityService.getBannerInfo(ids), ActivityInfo::getActivityId,
						ActivityInfo::getTitle, ActivityInfo::getBannerPic, resultMBanners);
				return;
			}

			if (BannerType.art == type) {
				getMBanner(type, artService.getBannerInfo(ids), Art::getId, Art::getTitle, Art::getBannerPic,
						resultMBanners);
				return;
			}
		});
		return OrderTool.keepOrder(orderKeys, resultMBanners, o -> o.getType() + o.getId());
	}

	private <T> void getMBanner(BannerType type, List<T> arrays, Function<T, String> idMapper,
			Function<T, String> titleMapper, Function<T, String> picMapper, List<WebsiteBannerWrapper> banners) {
		arrays.forEach(t -> {
			String pic = AliyunBean.getImagesUrl(picMapper.apply(t));
			banners.add(WebsiteBannerWrapper.of(type.name(), idMapper.apply(t), titleMapper.apply(t), pic));
		});
	}

}
