package com.jz.aliyun.tools;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.codec.binary.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.aliyun.oss.OSSClient;
import com.aliyun.oss.common.utils.BinaryUtil;
import com.aliyun.oss.model.MatchMode;
import com.aliyun.oss.model.PolicyConditions;
import com.aliyuncs.DefaultAcsClient;
import com.aliyuncs.exceptions.ClientException;
import com.aliyuncs.exceptions.ServerException;
import com.aliyuncs.http.MethodType;
import com.aliyuncs.http.ProtocolType;
import com.aliyuncs.profile.DefaultProfile;
import com.aliyuncs.profile.IClientProfile;
import com.aliyuncs.sts.model.v20150401.AssumeRoleRequest;
import com.aliyuncs.sts.model.v20150401.AssumeRoleResponse;
import com.jz.aliyun.beans.AliyunConfig;
import com.jz.aliyun.beans.StsTokenConfig;
import com.jz.aliyun.beans.StsUploadInfo;
import com.jz.common.utils.json.GsonTools;
import com.jz.common.utils.jz.IdTools;
import com.jz.common.utils.security.EncryptionTools;
import com.jz.common.utils.text.StringTools;

/**
 * @Title StsTools
 * @Package com.jz.aliyun.tools
 * @author tangjunfeng
 * @date 2018年7月11日 上午10:09:42
 * @version V1.0
 */
public class StsTools {

	private static final Logger logger = LoggerFactory.getLogger(StsTools.class);
	private static final String REGION_CN_BEIJING = "cn-beijing";
	private static final String STS_API_VERSION = "2015-04-01";
	private static final String roleArn = "acs:ram::1651426276585947:role/ramonlyupload";

	private AliyunConfig config;

	private StsTools(AliyunConfig config) {
		this.config = config;
	}

	public static StsTools getInstance(AliyunConfig config) {
		return new StsTools(config);
	}

	private String getFileId(String identity, String clientFileName, StsTokenConfig tokenConfig) {
		if (tokenConfig.isUseSourceName())
			return clientFileName;
		return StringTools.ternary(tokenConfig.getPrefix())
				+ IdTools.encode(identity + StringTools.RandomString.random(10))
				+ StringTools.ternary(tokenConfig.getSuffix());
	}

	private String getCallbackBody(String callbackBody, String host, String fileId, String savePath, String data,
			String clientFileName) throws UnsupportedEncodingException {
		// 拼接host
		if (StringTools.isNotEmptyAndBlank(host))
			callbackBody += "&host=" + host;
		callbackBody += "&fileId=" + fileId + "&path=" + URLEncoder.encode(savePath, "UTF-8") + "&fileName="
				+ URLEncoder.encode(clientFileName, "UTF-8");
		callbackBody += "&mimeType=${mimeType}&imageInfo.format=${imageInfo.format}&imgHeight=${imageInfo.height}&imgWidth=${imageInfo.width}";
		callbackBody += "&data=" + StringTools.ternary(data);
		return callbackBody;
	}

	/** 获取临时上传token */
	public StsUploadInfo getTemporaryUploadToken(String identity, String clientFileName, StsTokenConfig tokenConfig)
			throws Exception {
		String fileId = this.getFileId(identity, clientFileName, tokenConfig);
		String path = tokenConfig.getSaveDir();
		if (!path.startsWith(tokenConfig.getBasePath()))
			path = tokenConfig.getBasePath() + "/" + path;
		if (!path.endsWith("/"))
			path += "/";
		path += fileId;
		if (StringTools.isNotEmptyAndBlank(tokenConfig.getExt()))
			path += "." + tokenConfig.getExt();

		StsUploadInfo info = StsUploadInfo.of();
		String roleSessionName = EncryptionTools.MD5(identity);
		String policy = this.buildPolicy(tokenConfig.getBasePath(), config.getBucketName());
		AssumeRoleResponse resp = this.assumeRole(config.getAccessKeyId(), config.getAccessKeySecret(), roleArn,
				roleSessionName, policy, ProtocolType.HTTPS, tokenConfig.getTokenExpireSeconds());

		String ossHost = config.getBucketName() + "." + config.getEndpoint();
		
		info.setAccessKeyId(resp.getCredentials().getAccessKeyId())
				.setSecret(resp.getCredentials().getAccessKeySecret()).setHost(ossHost).setKey(path)
				.setSignature(resp.getCredentials().getSecurityToken())
				.setExpire(System.currentTimeMillis() / 1000 + tokenConfig.getTokenExpireSeconds());
		if (StringTools.isNotEmpty(tokenConfig.getCallback())) {
			String callbackBody = this.getCallbackBody(tokenConfig.getCallbackBody(),
					StringTools.ternary(tokenConfig.getCdnHost(), ossHost), fileId, path, tokenConfig.getData(),
					clientFileName);
			logger.info("set callback {}, {}", tokenConfig.getCallback(), callbackBody);
			info.setCallback(this.genCallback(tokenConfig.getCallback(), callbackBody));
		}

		return info;
	}

	public StsUploadInfo getSignUploadTask(String identity, String clientFileName, StsTokenConfig tokenConfig)
			throws UnsupportedEncodingException {
		OSSClient client = new OSSClient(config.getEndpoint(), config.getAccessKeyId(), config.getAccessKeySecret());
		String ossHost = config.getBucketName() + "." + config.getEndpoint();
		ossHost = (tokenConfig.isEnableHttps() ? "https://" : "http://") + ossHost;
		long expireEndTime = System.currentTimeMillis() + tokenConfig.getTokenExpireSeconds() * 1000;
		Date expiration = new Date(expireEndTime);
		PolicyConditions policyConds = new PolicyConditions();
		policyConds.addConditionItem(PolicyConditions.COND_CONTENT_LENGTH_RANGE, 0, 5368709120L);
		policyConds.addConditionItem(MatchMode.StartWith, PolicyConditions.COND_KEY, tokenConfig.getSaveDir());

		String postPolicy = client.generatePostPolicy(expiration, policyConds);
		byte[] binaryData = postPolicy.getBytes("UTF-8");
		String encodedPolicy = BinaryUtil.toBase64String(binaryData);
		String postSignature = client.calculatePostSignature(postPolicy);

		String fileId = this.getFileId(identity, clientFileName, tokenConfig);
		String fileName = fileId;
		if (StringTools.isNotEmptyAndBlank(tokenConfig.getExt()))
			fileName += "." + tokenConfig.getExt();
		String savePath = tokenConfig.getSaveDir() + "/" + fileName;

		StsUploadInfo info = StsUploadInfo.of().setAccessKeyId(config.getAccessKeyId()).setHost(ossHost)
				.setPolicy(encodedPolicy).setSignature(postSignature).setKey(savePath).setFileName(fileName)
				.setExpire(expireEndTime / 1000);
		if (StringTools.isNotEmpty(tokenConfig.getCallback())) {
			String callbackBody = this.getCallbackBody(tokenConfig.getCallbackBody(),
					StringTools.ternary(tokenConfig.getCdnHost(), ossHost), fileId, savePath, tokenConfig.getData(),
					clientFileName);
			logger.info("set callback {}, {}", tokenConfig.getCallback(), callbackBody);
			info.setCallback(this.genCallback(tokenConfig.getCallback(), callbackBody));
		}
		return info;
	}

	private String buildPolicy(String basepath, String bucket) {
		List<String> actions = new ArrayList<>();
		actions.add("oss:PutObject");
		actions.add("oss:InitMultipartUpload");
		actions.add("oss:UploadPart");
		actions.add("oss:ListParts");
		actions.add("oss:CompleteMultipartUpload");
		actions.add("oss:DeleteMultipartUpload");

		List<String> resources = new ArrayList<>();
		resources.add("acs:oss:*:*:" + bucket);
		if ("/".equals(basepath)) { // 可以访问当前bucket下的所有资源
			resources.add("acs:oss:*:*:" + bucket + "/*");
		} else { // 可以访问当前bucket/basePath下的所有资源
			resources.add("acs:oss:*:*:" + bucket + "/" + basepath + "/*");
		}

		Map<String, Object> statement = new HashMap<>();
		statement.put("Action", actions);
		statement.put("Resource", resources);
		statement.put("Effect", "Allow");

		Map<String, Object> policy = new HashMap<>();
		policy.put("Statement", Arrays.asList(statement));
		policy.put("Version", "1");
		return GsonTools.gson.toJson(policy);
	}

	private AssumeRoleResponse assumeRole(String accessKeyId, String accessKeySecret, String roleArn,
			String roleSessionName, String policy, ProtocolType protocolType, long durationSeconds)
			throws ServerException, ClientException {
		// 创建一个 Aliyun Acs Client, 用于发起 OpenAPI 请求
		IClientProfile profile = DefaultProfile.getProfile(REGION_CN_BEIJING, accessKeyId, accessKeySecret);
		DefaultAcsClient client = new DefaultAcsClient(profile);

		// 创建一个 AssumeRoleRequest 并设置请求参数
		final AssumeRoleRequest request = new AssumeRoleRequest();
		request.setVersion(STS_API_VERSION);
		request.setMethod(MethodType.POST);
		request.setProtocol(protocolType);

		request.setRoleArn(roleArn);
		request.setRoleSessionName(roleSessionName);
		request.setPolicy(policy);
		request.setDurationSeconds(durationSeconds);
		// 发起请求，并得到response
		return client.getAcsResponse(request);
	}

	private String genCallback(String callbackUrl, String callbackBody) throws UnsupportedEncodingException {
		Map<String, String> callback = new HashMap<>();
		callback.put("callbackUrl", callbackUrl);
		callback.put("callbackBody", callbackBody);
		callback.put("callbackBodyType", "application/x-www-form-urlencoded");
		return new String(Base64.encodeBase64(GsonTools.gson.toJson(callback).getBytes()));
	}

}
