package com.jz.aliyun.tools;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.net.URLEncoder;
import java.util.Calendar;
import java.util.Collection;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.aliyun.oss.ClientConfiguration;
import com.aliyun.oss.OSSClient;
import com.aliyun.oss.model.CannedAccessControlList;
import com.aliyun.oss.model.CopyObjectRequest;
import com.aliyun.oss.model.ObjectMetadata;
import com.google.common.collect.Maps;
import com.jz.aliyun.beans.AliyunConfig;
import com.jz.common.utils.json.GsonTools;

/**
 * @Title 针对于aliyun的Oss进行相关文件操作
 * @Package com.jz.aliyun.tools
 * @author tangjunfeng
 * @date 2018年7月10日 下午5:42:16
 * @version V1.0
 */
public class OssTools {

	private static final Logger logger = LoggerFactory.getLogger(OssTools.class);

	private AliyunConfig aliyunConfig;

	private OssTools() {
	}

	private OssTools(AliyunConfig aliyunConfig) {
		super();
		this.aliyunConfig = aliyunConfig;
	}

	public static OssTools getInstance(AliyunConfig aliyunConfig) {
		return new OssTools(aliyunConfig);
	}

	/** 上传本地文件 */
	public void upload(String localFilePath, String ossFilePath) throws FileNotFoundException {
		logger.info("aliyun upload file : {} => {}", localFilePath, ossFilePath);
		upload(new FileInputStream(new File(localFilePath)), ossFilePath);
	}

	/** 创建OSSClient实例 */
	private OSSClient getOssClient() {
		System.out.println("aliyunConfig : " + GsonTools.gson.toJson(aliyunConfig));
		ClientConfiguration conf = new ClientConfiguration();
		// 设置是否支持将自定义域名作为Endpoint，默认支持
		conf.setSupportCname(true);
		return new OSSClient(aliyunConfig.getEndpoint(), aliyunConfig.getAccessKeyId(),
				aliyunConfig.getAccessKeySecret(), conf);
	}

	/** 上传网络文件 */
	public void upload(InputStream inputStream, String ossFilePath) throws FileNotFoundException {
		// 创建OSSClient实例
		OSSClient ossClient = this.getOssClient();
		// 上传文件流
		ossClient.putObject(aliyunConfig.getBucketName(), ossFilePath, inputStream);
		// 关闭client
		ossClient.shutdown();
	}

	/** oss上的文件复制 */
	public void copy(String sourcePath, String destinationPath) {
		logger.info("aliyun copy file : {} => {}", sourcePath, destinationPath);
		// 创建OSSClient实例
		OSSClient ossClient = this.getOssClient();
		ossClient.copyObject(aliyunConfig.getBucketName(), sourcePath, aliyunConfig.getBucketName(), destinationPath);
		// 关闭client
		ossClient.shutdown();
	}

	/** oss上的文件复制 */
	public void copyToTargetBucket(String sourcePath, String destinationPath) {
		logger.info("aliyun copy file : {}.{} => {}.{}", aliyunConfig.getSourceBucketName(), sourcePath,
				aliyunConfig.getTargetBucketName(), destinationPath);
		// 创建OSSClient实例
		OSSClient ossClient = this.getOssClient();
		// 复制
		ossClient.copyObject(aliyunConfig.getSourceBucketName(), sourcePath, aliyunConfig.getTargetBucketName(),
				destinationPath);
		// 继承Bucket
		ossClient.setObjectAcl(aliyunConfig.getTargetBucketName(), destinationPath, CannedAccessControlList.Default);
		// 关闭client
		ossClient.shutdown();
	}

	/** oss上的文件删除 */
	public void delete(String filePath) throws FileNotFoundException {
		logger.info("aliyun delete file : {} => {}", aliyunConfig.getBucketName(), filePath);
		// 创建OSSClient实例
		OSSClient ossClient = this.getOssClient();
		ossClient.deleteObject(aliyunConfig.getBucketName(), filePath);
		// 关闭client
		ossClient.shutdown();
	}

	/** oss上的文件移动（like shell : mv） */
	public void move(String sourcePath, String destinationPath) {
		logger.info("aliyun mv file : {}/{} => {}/{}", aliyunConfig.getSourceBucketName(), sourcePath,
				aliyunConfig.getTargetBucketName(), destinationPath);
		// 创建OSSClient实例
		OSSClient ossClient = this.getOssClient();
		try {
			sourcePath = URLEncoder.encode(sourcePath, "UTF-8");
			destinationPath = URLEncoder.encode(destinationPath, "UTF-8");
		} catch (Exception e) {
		}
		// 先复制
		ossClient.copyObject(aliyunConfig.getSourceBucketName(), sourcePath, aliyunConfig.getTargetBucketName(),
				destinationPath);
		// 继承Bucket
		ossClient.setObjectAcl(aliyunConfig.getTargetBucketName(), destinationPath, CannedAccessControlList.Default);
		// 再删除源文件
		ossClient.deleteObject(aliyunConfig.getSourceBucketName(), sourcePath);
		// 关闭client
		ossClient.shutdown();
	}

	/** oss上的文件可访问地址 */
	public String getUrl(String fileUrl, int expireSeconds) {
		// 创建OSSClient实例
		OSSClient ossClient = this.getOssClient();
		// 过期时间为指定秒之后
		Calendar expiration = Calendar.getInstance();
		expiration.add(Calendar.SECOND, expireSeconds);
		URL url = ossClient.generatePresignedUrl(aliyunConfig.getBucketName(), fileUrl, expiration.getTime());
		// 关闭client
		ossClient.shutdown();
		return url.toString();
	}

	/** oss上的文件可访问地址 */
	public Map<String, String> getUrl(Collection<String> fileUrls, int expireSeconds) {
		// 创建OSSClient实例
		OSSClient ossClient = this.getOssClient();
		// 过期时间为指定秒之后
		Calendar expiration = Calendar.getInstance();
		expiration.add(Calendar.SECOND, expireSeconds);
		Map<String, String> source2Target = Maps.newHashMap();
		for (String tempUrl : fileUrls) {
			URL url = ossClient.generatePresignedUrl(aliyunConfig.getBucketName(), tempUrl, expiration.getTime());
			source2Target.put(tempUrl, url.toString());
		}
		// 关闭client
		ossClient.shutdown();
		return source2Target;
	}

	/** oss上的文件设置下载的文件名称 */
	public void updateDownloadName(String sourcePath, String downloadFileName) throws UnsupportedEncodingException {
		// https://help.aliyun.com/document_detail/84840.html?spm=a2c4g.11186623.6.686.127642444xBkgp
		logger.info("aliyun update download file name : {} => {}", sourcePath, downloadFileName);
		// 创建OSSClient实例
		OSSClient ossClient = this.getOssClient();
		String bucketName = aliyunConfig.getSourceBucketName();
		// 设置源文件与目标文件相同，调用ossClient.copyObject方法修改文件元信息
		CopyObjectRequest request = new CopyObjectRequest(bucketName, sourcePath, bucketName, sourcePath);

		ObjectMetadata meta = new ObjectMetadata();
		// 设置内容被下载时的名称
		meta.setContentDisposition("attachment; filename=" + URLEncoder.encode(downloadFileName, "UTF-8"));
		// 设置内容被下载时的编码格式。
		meta.setContentEncoding("UTF-8");
		request.setNewObjectMetadata(meta);

		// 修改元信息。
		ossClient.copyObject(request);
		// 关闭client
		ossClient.shutdown();
	}

	public static void main(String[] args) {
		String url = "playabc/video/00b5d7bac101baac/normal.mp4";
		AliyunConfig config = AliyunConfig.of("video.gymchina.com", "LTAIXIOV1fsWtqzM",
				"aEnYKzdq9wZS5MLrJw6Dj8amkW3XK9", "jz-media");
		System.out.println(OssTools.getInstance(config).getUrl(url, 3600));
	}
}