package com.gymchina.library.netclient.interceptor

import okhttp3.*
import okhttp3.MediaType.Companion.toMediaTypeOrNull
import okio.Buffer
import java.io.IOException
import java.lang.IllegalArgumentException

/**
 * 自定义拦截器
 * Request Header & Params
 *
 * 使用方式：
 * BasicInfoInterceptor interceptor = new BasicInfoInterceptor.Builder()
 * .addParam("from", "android") //添加公共参数到 post 请求体
 * .addQueryParam("version","1")  // 添加公共版本号，加在 URL 后面
 * .addHeaderLine("X-Ping: Pong")  // 示例： 添加公共消息头
 * .addParamsMap(map) // 可以添加 Map 格式的参数
 * .build();
 *
 * Created by wangwh on 2018/12/26.
 */
class BasicInfoInterceptor private constructor() : Interceptor {

    private val queryParamsMap: HashMap<String, String> = hashMapOf() // 添加到 URL 末尾，Get Post 方法都使用
    private val paramsMap: HashMap<String, String> = hashMapOf()      // 添加到公共参数到消息体，适用 Post 请求
    private val headerParamsMap: HashMap<String, String> = hashMapOf()// 公共 Headers 添加
    private val headerLinesList: ArrayList<String> = arrayListOf()  // 消息头 集合形式，一次添加一行

    override fun intercept(chain: Interceptor.Chain): Response {
        var request: Request? = chain.request()
        val requestBuilder = request!!.newBuilder()

        // 1，process header params inject
        val headerBuilder = request.headers.newBuilder()
        // 1.1 以 Entry 添加所有消息头
        if (headerParamsMap.isNotEmpty()) {
            for ((key, value) in headerParamsMap) {
                headerBuilder.add(key, value)
            }
        }
        // 1.2 以 String 形式添加消息头
        if (headerLinesList.isNotEmpty()) {
            for (line in headerLinesList) {
                headerBuilder.add(line)
            }
            requestBuilder.headers(headerBuilder.build())
        }
        //1， process header params end

        // 2， process queryParams inject whatever it's GET or POST
        if (queryParamsMap.isNotEmpty()) request =
                injectParamsIntoUrl(request.url.newBuilder(), requestBuilder, queryParamsMap)

        // 3，process post body inject
        if (paramsMap.isNotEmpty() && canInjectIntoBody(request)) {

            val formBodyBuilder = FormBody.Builder()
            for ((key, value) in paramsMap) {
                formBodyBuilder.add(key, value)
            }
            val formBody = formBodyBuilder.build()
            var postBodyString = bodyToString(request!!.body)
            postBodyString += (if (postBodyString.isNotEmpty()) "&" else "") + bodyToString(formBody)
            requestBuilder.post(
                RequestBody.create(
                    "application/x-www-form-urlencoded;charset=UTF-8".toMediaTypeOrNull(),
                    postBodyString
                )
            )
        }
        // 4，构建新的请求
        request = requestBuilder.build()
        return chain.proceed(request)
    }

    /**
     * 往 Url 中注入参数
     */
    private fun injectParamsIntoUrl(
        httpUrlBuilder: HttpUrl.Builder,
        requestBuilder: Request.Builder,
        paramsMap: Map<String, String>
    ): Request? {
        if (paramsMap.isNotEmpty()) {
            for ((key, value) in paramsMap) {
                httpUrlBuilder.addQueryParameter(key, value)
            }
            requestBuilder.url(httpUrlBuilder.build())
            return requestBuilder.build()
        }
        return null
    }

    /**
     * 确认是否是 post 请求
     *
     * @param request 发出的请求
     * @return true 需要注入公共参数
     */
    private fun canInjectIntoBody(request: Request?): Boolean {
        if (request == null) return false
        if ("POST" == request.method) return false
        val body = request.body ?: return false
        val mediaType = body.contentType() ?: return false
        return "x-www-form-urlencoded" == mediaType.subtype
    }

    /**
     * 将请求体转成字符串
     */
    private fun bodyToString(request: RequestBody?): String {
        try {
            val buffer = Buffer()
            if (request != null)
                request.writeTo(buffer)
            else
                return ""
            return buffer.readUtf8()
        } catch (e: IOException) {
            return "did not work"
        }
    }

    /**
     * 构建拦截器
     */
    class Builder {
        private val interceptor = BasicInfoInterceptor()

        fun build(): BasicInfoInterceptor {
            return interceptor
        }

        /**
         * 添加公共参数到 post 消息体
         */
        fun addParam(key: String, value: String): Builder {
            interceptor.paramsMap[key] = value
            return this
        }

        /**
         * 添加公共参数到 post 消息体
         */
        fun addParamsMap(paramsMap: Map<String, String>): Builder {
            interceptor.paramsMap.putAll(paramsMap)
            return this
        }

        /**
         * 添加公共参数到消息头
         */
        fun addHeaderParam(key: String, value: String): Builder {
            interceptor.headerParamsMap[key] = value
            return this
        }

        /**
         * 添加公共参数集合到消息头
         */
        fun addHeaderParamsMap(headerParamsMap: Map<String, String>): Builder {
            interceptor.headerParamsMap.putAll(headerParamsMap)
            return this
        }

        /**
         * 添加公共参数到消息头
         */
        fun addHeaderLine(headerLine: String): Builder {
            val index = headerLine.indexOf(":")
            if (index == -1) throw IllegalArgumentException("Unexpected header: $headerLine")
            interceptor.headerLinesList.add(headerLine)
            return this
        }

        /**
         * 添加公共参数集合到消息头
         */
        fun addHeaderLinesList(headerLinesList: List<String>): Builder {
            for (headerLine in headerLinesList) {
                val index = headerLine.indexOf(":")
                if (index == -1) throw IllegalArgumentException("Unexpected header: $headerLine")
                interceptor.headerLinesList.add(headerLine)
            }
            return this
        }

        /**
         * 添加公共参数到 URL
         */
        fun addQueryParam(key: String, value: String): Builder {
            interceptor.queryParamsMap[key] = value
            return this
        }

        /**
         * 添加公共参数集合到 URL
         */
        fun addQueryParamsMap(queryParamsMap: Map<String, String>): Builder {
            interceptor.queryParamsMap.putAll(queryParamsMap)
            return this
        }
    }
}