package com.gymchina.library.netclient.callback

import retrofit2.Call
import retrofit2.Response

/**
 * 处理请求响应
 */
abstract class AbsCallback<Content>: retrofit2.Callback<Content> {

    /**
     * 收到HTTP响应时回调
     *
     * 注意：
     * 1，这里的 HTTP响应 包括应用程序级别的异常，包括：404、500
     * 2，通过调用 {@link Response#isSuccessful()} 确定响应是否成功
     */
    override fun onResponse(call: Call<Content>, response: Response<Content>) {
        if (response.isSuccessful) {
            // 响应成功
            onSuccess(response.body())
        } else {
            // 响应失败
            if (handleException(call, response)) {
                // handleException 拦截了 异常状态
                return
            }
            if (onException(call, response)) {
                // onException 处理了异常状态
                return
            }
            onSuccess(null)
        }
    }

    /**
     * 服务器返回状态正常时调用
     *
     * 注意：当不处理异常状态时，会走该回调 content = null
     */
    abstract fun onSuccess(content: Content?)

    /**
     * 在发生网络异常时与服务器通信或意外发生时调用
     *
     * 如：创建请求或处理响应时发生异常
     */
    /*
    override fun onFailure(call: Call<Content>, t: Throwable) {

    }
    */

    /**
     * 与服务器通信，服务器返回状态异常时调用
     *
     * @return
     * true ：表示该函数会拦截 状态异常，不会继续执行 #onException
     * false ： 表示函数不会拦截 异常状态，会继续执行 #onException 或 #onSuccess
     */
    open fun handleException(call: Call<Content>, response: Response<Content>): Boolean {
        return false
    }

    /**
     * 与服务器通信，服务器返回状态异常时调用
     * @return
     *
     * true ：表示该函数接受 异常状态
     * false ： 表示函数不接受 异常状态，交由 #onSuccess(null) 去处理，
     */
    open fun onException(call: Call<Content>, response: Response<Content>): Boolean {
        return false
    }
}