package com.gymchina.library.netclient

import com.gymchina.library.netclient.interceptor.BasicInfoInterceptor
import com.ihsanbal.logging.Level
import com.ihsanbal.logging.LoggingInterceptor
import okhttp3.Interceptor
import okhttp3.OkHttpClient
import okhttp3.internal.platform.Platform
import retrofit2.Converter
import retrofit2.Retrofit
import java.util.HashMap
import java.util.concurrent.TimeUnit

abstract class AbsNetClient {

    companion object {
        const val CONNECT_TIMEOUT: Long = 20L // 单位：秒
        const val WRITE_TIMEOUT: Long = 15L   // 单位：秒
        const val READ_TIMEOUT: Long = 15L    // 单位：秒
    }

    fun <Interface, Api : AbsApi<Interface>> api(api: Api): Interface {
        return createService(api)
    }

    /**
     * 不再缓存Api接口服务对象，这里空实现用于APP端调用
     */
    fun resetClient() {
    }

    ////////////////////////////////////////////////////////////////////////////////

    /**
     * 构建 API Interface
     */
    @Suppress("UNCHECKED_CAST")
    open fun <Interface, Api : AbsApi<Interface>> createService(api: Api): Interface {
        return createRetrofit(api).create(api.getServiceClass()) as Interface
    }

    /**
     * 构建 Retrofit 实例
     */
    protected open fun <Interface, Api : AbsApi<Interface>> createRetrofit(api: Api): Retrofit {
        val builder = Retrofit.Builder().baseUrl(api.getBaseUrl()).client(okHttpClient)
        val converterFactory = converterFactory
        if (!converterFactory.isNullOrEmpty()) {
            for (cf in converterFactory) builder.addConverterFactory(cf)
        }
        return builder.build()
    }

    //////////////////////////// 以下函数调用时都会创建一个新的实例 ////////////////////////////////

    /**
     * 生成 OkHttpClient
     */
    protected open val okHttpClient: OkHttpClient
        get() {
            val builder = OkHttpClient.Builder()
                .connectTimeout(CONNECT_TIMEOUT, TimeUnit.SECONDS)
                .writeTimeout(WRITE_TIMEOUT, TimeUnit.SECONDS)
                .readTimeout(READ_TIMEOUT, TimeUnit.SECONDS)
                .retryOnConnectionFailure(true)

            val allInterceptor = allInterceptor
            if (!allInterceptor.isNullOrEmpty()) {
                for (interceptor in allInterceptor) builder.addInterceptor(interceptor)
            }
            if (isHttpLogEnable) {
                builder.addInterceptor(getHttpInterceptor())
            }
            val networkInterceptor = networkInterceptor
            if (!networkInterceptor.isNullOrEmpty()) {
                for (interceptor in networkInterceptor) builder.addNetworkInterceptor(interceptor)
            }
            return builder.build()
        }

    /**
     * 获取解析器数组
     */
    protected open val converterFactory: MutableList<Converter.Factory>?
        get() = null

    /**
     * 获取网络拦截器数组
     */
    protected open val networkInterceptor: MutableList<Interceptor>?
        get() = null

    /**
     * 获取拦截器数组
     */
    protected open val allInterceptor: MutableList<Interceptor>?
        get() {
            val basicIt = basicInterceptor
            return if (basicIt != null) mutableListOf(basicIt) else null
        }

    /**
     * 配置拦截器，用于添加通用参数和头
     */
    protected open val basicInterceptor: Interceptor?
        get() {
            var builder: BasicInfoInterceptor.Builder? = null
            if (basicQueryParams != null) {
                builder = builder ?: BasicInfoInterceptor.Builder()
                builder.addQueryParamsMap(basicQueryParams!!)
            }
            if (basicHeaderLines != null) {
                builder = builder ?: BasicInfoInterceptor.Builder()
                builder.addHeaderLinesList(basicHeaderLines!!)
            }
            return builder?.build()
        }

    protected open fun getHttpInterceptor() : Interceptor{
        return LoggingInterceptor.Builder()
            .setLevel(Level.BASIC)
            .tag("okhttp")
            .build()
    }

    /**
     * 配置通用请求参数
     */
    protected open val basicQueryParams: HashMap<String, String>?
        get() = null

    protected open val isHttpLogEnable: Boolean get() = false

    /**
     * 配置通用请求头
     */
    protected open val basicHeaderLines: MutableList<String>?
        get() = null
}
