package com.aliyun.opensearch;
/**
 * Created by dengwx on 16/8/29.
 */

import com.aliyun.opensearch.client.ResourceClient;
import com.aliyun.opensearch.sdk.generated.commons.OpenSearchClientException;
import com.aliyun.opensearch.sdk.generated.commons.OpenSearchException;
import com.aliyun.opensearch.sdk.generated.commons.OpenSearchResult;
import com.aliyun.opensearch.sdk.generated.document.Command;
import com.aliyun.opensearch.sdk.generated.document.DocumentConstants;
import com.aliyun.opensearch.sdk.generated.document.DocumentService;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.Map;
import java.util.Map.Entry;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

public class DocumentClient implements DocumentService.Iface {
  private static final Log log = LogFactory.getLog(DocumentClient.class);

  private ResourceClient resourceClient;

  /**
   * 进行提交的数据
   */
  private JSONArray docBuffer = new JSONArray();

  /**
   * 构造函数
   *
   * @param client CloudsearchClient实例。
   */
  public DocumentClient(OpenSearchClient client) {
    this.resourceClient = new ResourceClient("/apps", client);
  }

  /**
   * 添加文档
   *
   * 设置需要添加的属性名称和属性值，用于生成符合文档格式的数据，所有更新结束之后需要调用push(String tableName)方法
   *
   * @param fields 字段名和字段值的map
   * @throws JSONException
   */
  public void add(Map<String, Object> fields) {
    pushOneDoc(fields, Command.ADD);
  }

  /**
   * 更新文档
   *
   * 设置需要更新的属性名称和属性值，用于生成符合文档格式的数据，所有更新结束之后需要调用push(String tableName)方法
   * 标准版不支持update
   *
   * @param fields 字段名和字段值的map
   * @throws JSONException
   */
  public void update(Map<String, Object> fields) {
    pushOneDoc(fields, Command.UPDATE);
  }

  /**
   * 删除文档
   *
   * 设置需要删除的属性名称和属性值，用于生成符合文档格式的数据，所有更新结束之后需要调用push(String tableName)方法
   *
   * @param fields 字段名和字段值的map
   * @throws JSONException
   */
  public void remove(Map<String, Object> fields) {
    pushOneDoc(fields, Command.DELETE);
  }

  private void pushOneDoc(Map<String, Object> fields, Command command) {
    JSONObject JSONFields = new JSONObject();
    for (Entry<String, Object> entry : fields.entrySet()) {
      JSONFields.put(entry.getKey(), entry.getValue());
    }
    JSONObject json = new JSONObject();
    json.put(DocumentConstants.DOC_KEY_CMD, command.toString());
    json.put(DocumentConstants.DOC_KEY_FIELDS, JSONFields);
    docBuffer.put(json);
  }

  /**
   * 执行文档变更操作(1)
   *
   * 针对文档的操作add、update和remove会生成符合文档格式的数据，通过调用此接口用户提交的文档变更才会真正生效。
   *
   * @param tableName 表名称
   * @return 返回的数据
   * @throws OpenSearchException
   */
	public OpenSearchResult commit(String appName, String tableName)
			throws OpenSearchException, OpenSearchClientException {
		OpenSearchResult result;
		try {
			result = doPush(docBuffer, appName, tableName);
		} finally {
			docBuffer = new JSONArray();
		}
		return result;
	}

  /**
   * 执行文档变更操作(2)
   *
   * 通过此接口可以直接将符合文档格式的数据直接推送到指定的表中
   *
   * @param docs 此docs为用户push的数据，此字段为json类型的字符串。
   * @param tableName 操作的表名。
   * @return 请求API并返回相应的结果。
   * @throws OpenSearchException
   * @throws JSONException
   */
  @Override
  public OpenSearchResult push(String docsJson, String appName, String tableName) throws OpenSearchException, OpenSearchClientException {
    JSONArray docs = new JSONArray(docsJson);// verify user input.
    return doPush(docs, appName, tableName);
  }

  private OpenSearchResult doPush(JSONArray jsonArray, String appName, String tableName) throws OpenSearchException, OpenSearchClientException {
    return resourceClient.post(createPushPath(appName, tableName), jsonArray.toString());
  }

  private String createPushPath(String appName, String tableName) {
    return String.format("/%s/%s/actions/bulk", appName, tableName);
  }
}
