/*
 * Copyright (c) 2015-present Alipay.com, https://www.alipay.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package cn.com.antcloud.api.common;

import cn.com.antcloud.api.acapi.AntCloudHttpClient;
import cn.com.antcloud.api.acapi.HttpConfig;
import cn.com.antcloud.api.acapi.StringUtils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Constructor;
import java.lang.reflect.ParameterizedType;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;

public abstract class BaseGwClient<REQ extends BaseClientRequest, RES extends BaseClientResponse> {

    private static final Logger  logger = LoggerFactory.getLogger(BaseGwClient.class);

    private final String         endpoint;
    private final String         accessKey;
    private final String         accessSecret;
    private final boolean        checkSign;
    private final boolean        enableAutoRetry;
    private final int            autoRetryLimit;

    protected AntCloudHttpClient httpClient;
    protected String             securityToken;

    public BaseGwClient(String endpoint, String accessKey, String accessSecret, boolean checkSign,
                        boolean enableAutoRetry, int autoRetryLimit,
                        AntCloudHttpClient httpClient, String securityToken) {
        SDKUtils.checkNotNull(endpoint);
        SDKUtils.checkNotNull(accessKey);
        SDKUtils.checkNotNull(accessSecret);

        this.endpoint = endpoint;
        this.accessKey = accessKey;
        this.accessSecret = accessSecret;
        this.checkSign = checkSign;
        this.enableAutoRetry = enableAutoRetry;
        this.autoRetryLimit = autoRetryLimit;
        this.httpClient = httpClient;
        this.securityToken = securityToken;
    }

    public String getEndpoint() {
        return endpoint;
    }

    public String getAccessKey() {
        return accessKey;
    }

    public String getAccessSecret() {
        return accessSecret;
    }

    public boolean isCheckSign() {
        return checkSign;
    }

    protected HttpUriRequest buildRequest(String endpoint,
                                          Map<String, String> request) throws UnsupportedEncodingException {
        URI uri;
        try {
            URIBuilder builder = new URIBuilder(endpoint);
            uri = builder.build();
        } catch (URISyntaxException e) {
            throw new ClientException(SDKConstants.ResultCodes.PARASE_URL_ERROR, e);
        }

        List<NameValuePair> nameValuePairs = new ArrayList<NameValuePair>();
        for (String key : request.keySet()) {
            nameValuePairs.add(new BasicNameValuePair(key, request.get(key)));
        }
        UrlEncodedFormEntity urlEncodedFormEntity = new UrlEncodedFormEntity(nameValuePairs,
            SDKConstants.DEFAULT_CHARSET);
        HttpPost httpPost = new HttpPost(uri);
        httpPost.setEntity(urlEncodedFormEntity);
        return httpPost;
    }

    public RES execute(REQ request) throws InterruptedException {
        SDKUtils.checkNotNull(request);
        SDKUtils.checkNotNull(request.getMethod(), "method cannot be null");
        SDKUtils.checkNotNull(request.getVersion(), "version cannot be null");
        prepareParameters(request);

        return sendRequest(request);
    }

    private RES sendRequest(REQ request) throws InterruptedException {
        int retried = 0;
        for (;;) {
            try {
                HttpUriRequest httpUriRequest = buildRequest(endpoint, request.getParameters());
                HttpResponse httpResponse = this.httpClient.invoke(httpUriRequest);
                String responseString = EntityUtils.toString(httpResponse.getEntity(),
                    Charset.forName("UTF-8"));

                // 如果数据解析失败，则开启重试逻辑
                JSONObject wholeJson;
                try {
                    wholeJson = JSON.parseObject(responseString);
                } catch (Throwable e) {
                    logger.error("error when parse response as json, response = {}",
                        responseString);
                    if (enableAutoRetry && retried < autoRetryLimit) {
                        // 具备重试条件
                        retried += 1;
                        logger.error("retry send request, retried count = {}", retried);
                        continue;
                    } else {
                        throw new ClientException(SDKConstants.ResultCodes.RESPONSE_FORMAT_ERROR,
                            e);
                    }
                }

                if (wholeJson == null) {
                    logger.error(responseString);
                    throw new ClientException(SDKConstants.ResultCodes.TRANSPORT_ERROR,
                        "Unexpected gateway response: " + responseString);
                }

                JSONObject responseNode = wholeJson.getJSONObject("response");

                if (responseNode == null) {
                    logger.error(responseString);
                    throw new ClientException(SDKConstants.ResultCodes.TRANSPORT_ERROR,
                        "Unexpected gateway response: " + responseString);
                }

                RES response = newResponse();
                response.setData(responseNode);

                if (response.isSuccess() && checkSign) {
                    String sign = wholeJson.getString(SDKConstants.ParamKeys.SIGN);
                    String stringToSign = GwSigns.extractStringToSign(responseString);
                    String calculatedSign;
                    try {
                        calculatedSign = GwSigns.sign(stringToSign,
                            request.getParameter(SDKConstants.ParamKeys.SIGN_TYPE), accessSecret,
                            SDKConstants.SIGN_CHARSET);
                    } catch (Exception e) {
                        throw new ClientException(SDKConstants.ResultCodes.BAD_SIGNATURE,
                            "Invalid signature in response");
                    }

                    if (!calculatedSign.equals(sign)) {
                        throw new ClientException(SDKConstants.ResultCodes.BAD_SIGNATURE,
                            "Invalid signature in response");
                    }
                }
                return response;
            } catch (IOException e) {
                throw new ClientException(SDKConstants.ResultCodes.TRANSPORT_ERROR, e);
            }
        }
    }

    protected RES newResponse() {
        Class<RES> clazz = GenericTypeResolver.resolveTypeArguments(getClass(),
            BaseGwClient.class)[1];
        try {
            return clazz.newInstance();
        } catch (InstantiationException e) {
            throw new IllegalStateException("Cannot create response");
        } catch (IllegalAccessException e) {
            throw new IllegalStateException("Cannot create response");
        }
    }

    private void putParameterIfAbsent(REQ request, String key, String value) {
        if (request.getParameter(key) == null) {
            request.putParameter(key, value);
        }
    }

    private void prepareParameters(REQ request) {
        request.putParameter(SDKConstants.ParamKeys.ACCESS_KEY, accessKey);

        putParameterIfAbsent(request, SDKConstants.ParamKeys.SIGN_TYPE,
            SDKConstants.DEFAULT_SIGN_TYPE);
        putParameterIfAbsent(request, SDKConstants.ParamKeys.REQ_MSG_ID,
            SDKUtils.generateReqMsgId());
        putParameterIfAbsent(request, SDKConstants.ParamKeys.REQ_TIME,
            SDKUtils.formatDate(new Date()));

        String signType = request.getParameter(SDKConstants.ParamKeys.SIGN_TYPE);

        if (!signType.equalsIgnoreCase(SDKConstants.DEFAULT_SIGN_TYPE)
            && !signType.equalsIgnoreCase(SDKConstants.SIGN_TYPE_SHA256)) {
            throw new ClientException(SDKConstants.ResultCodes.INVALID_PARAMETER,
                "wrong sign type");
        }

        // STS token
        if (!StringUtils.isEmpty(securityToken)) {
            putParameterIfAbsent(request, SDKConstants.ParamKeys.SECURITY_TOKEN, securityToken);
        }

        // 基础包版本信息
        putParameterIfAbsent(request, SDKConstants.ParamKeys.BASE_SDK_VERSION,
            SDKConstants.BASE_SDK_VERSION_VALUE);
        try {
            String sign = GwSigns.sign(request.getParameters(),
                request.getParameter(SDKConstants.ParamKeys.SIGN_TYPE), accessSecret,
                SDKConstants.SIGN_CHARSET);
            request.putParameter(SDKConstants.ParamKeys.SIGN, sign);
        } catch (Exception e) {
            throw new ClientException(SDKConstants.ResultCodes.UNKNOWN_ERROR, e);
        }
    }

    protected static class Builder<T, B extends Builder<T, B>> {
        private String             endpoint;
        private String             accessKey;
        private String             accessSecret;
        private boolean            checkSign       = true;
        private boolean            enableAutoRetry = false;
        private int                autoRetryLimit  = 3;
        private AntCloudHttpClient httpClient;
        private String             securityToken;

        public T build() {
            @SuppressWarnings("unchecked")
            Class<T> clazz = (Class<T>) ((ParameterizedType) getClass().getGenericSuperclass()).getActualTypeArguments()[0];

            if (httpClient == null){
                HttpConfig httpConfig = new HttpConfig();
                httpClient = new AntCloudHttpClient(httpConfig);
            }

            try {
                Constructor<T> ctor = clazz.getDeclaredConstructor(String.class, String.class,
                    String.class, boolean.class, boolean.class, int.class,
                    AntCloudHttpClient.class, String.class);
                ctor.setAccessible(true);
                return ctor.newInstance(endpoint, accessKey, accessSecret, checkSign,
                    enableAutoRetry, autoRetryLimit, httpClient, securityToken);
            } catch (Exception e) {
                throw new IllegalStateException(e);
            }
        }

        public B setHttpClient(AntCloudHttpClient httpClient) {
            this.httpClient = httpClient;
            return (B) this;
        }

        public B setEndpoint(String endpoint) {
            this.endpoint = endpoint;
            return (B) this;
        }

        public B setAccess(String accessKey, String accessSecret) {
            this.accessKey = accessKey.trim();
            this.accessSecret = accessSecret.trim();
            return (B) this;
        }

        public B setCheckSign(boolean checkSign) {
            this.checkSign = checkSign;
            return (B) this;
        }

        public B setEnableAutoRetry(boolean enableAutoRetry) {
            this.enableAutoRetry = enableAutoRetry;
            return (B) this;
        }

        public B setAutoRetryLimit(int autoRetryLimit) {
            this.autoRetryLimit = autoRetryLimit;
            return (B) this;
        }

        public B setSecurityToken(String securityToken) {
            this.securityToken = securityToken;
            return (B) this;
        }
    }
}
