/**
 * Copyright (c) 2009-2015, Data Geekery GmbH (http://www.datageekery.com)
 * All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Other licenses:
 * -----------------------------------------------------------------------------
 * Commercial licenses for this work are available. These replace the above
 * ASL 2.0 and offer limited warranties, support, maintenance, and commercial
 * database integrations.
 *
 * For more information, please visit: http://www.jooq.org/licenses
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */
package org.jooq.impl;

import static org.jooq.conf.SettingsTools.executePreparedStatements;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;

import org.jooq.conf.Settings;
import org.jooq.conf.StatementType;
import org.jooq.tools.jdbc.DefaultConnection;

/**
 * A proxy for a JDBC {@link Connection} that handles creation of prepared
 * statements according to the settings' {@link StatementType}
 *
 * @author Lukas Eder
 */
class SettingsEnabledConnection extends DefaultConnection {

    private final Settings settings;

    SettingsEnabledConnection(Connection delegate, Settings settings) {
        super(delegate);

        this.settings = settings;
    }

    // ------------------------------------------------------------------------
    // XXX Creation of PreparedStatements
    // ------------------------------------------------------------------------

    @Override
    public final PreparedStatement prepareStatement(String sql) throws SQLException {
        if (executePreparedStatements(settings)) {
            return getDelegate().prepareStatement(sql);
        }
        else {
            return new SettingsEnabledPreparedStatement(getDelegate(), sql);
        }
    }

    @Override
    public final PreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency)
        throws SQLException {
        if (executePreparedStatements(settings)) {
            return getDelegate().prepareStatement(sql, resultSetType, resultSetConcurrency);
        }
        else {
            return new SettingsEnabledPreparedStatement(getDelegate(), sql, resultSetType, resultSetConcurrency);
        }
    }

    @Override
    public final PreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency,
        int resultSetHoldability) throws SQLException {
        if (executePreparedStatements(settings)) {
            return getDelegate().prepareStatement(sql, resultSetType, resultSetConcurrency, resultSetHoldability);
        }
        else {
            return new SettingsEnabledPreparedStatement(getDelegate(), sql, resultSetType, resultSetConcurrency, resultSetHoldability);
        }
    }

    @Override
    public final PreparedStatement prepareStatement(String sql, int autoGeneratedKeys) throws SQLException {
        if (executePreparedStatements(settings)) {
            return getDelegate().prepareStatement(sql, autoGeneratedKeys);
        }
        else {
            return new SettingsEnabledPreparedStatement(getDelegate(), sql, autoGeneratedKeys);
        }
    }

    @Override
    public final PreparedStatement prepareStatement(String sql, int[] columnIndexes) throws SQLException {
        if (executePreparedStatements(settings)) {
            return getDelegate().prepareStatement(sql, columnIndexes);
        }
        else {
            return new SettingsEnabledPreparedStatement(getDelegate(), sql, columnIndexes);
        }
    }

    @Override
    public final PreparedStatement prepareStatement(String sql, String[] columnNames) throws SQLException {
        if (executePreparedStatements(settings)) {
            return getDelegate().prepareStatement(sql, columnNames);
        }
        else {
            return new SettingsEnabledPreparedStatement(getDelegate(), sql, columnNames);
        }
    }
}
