/**
 * Copyright (c) 2009-2015, Data Geekery GmbH (http://www.datageekery.com)
 * All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Other licenses:
 * -----------------------------------------------------------------------------
 * Commercial licenses for this work are available. These replace the above
 * ASL 2.0 and offer limited warranties, support, maintenance, and commercial
 * database integrations.
 *
 * For more information, please visit: http://www.jooq.org/licenses
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */

package org.jooq;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.Statement;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutorService;

import org.jooq.conf.Settings;
import org.jooq.exception.DataAccessException;
import org.jooq.exception.DataTypeException;
import org.jooq.exception.InvalidResultException;
import org.jooq.exception.MappingException;
import org.jooq.exception.TooManyRowsException;
import org.jooq.impl.DefaultRecordMapper;

/**
 * A query that can return results. Mostly, this is a {@link Select} query used
 * for a <code>SELECT</code> statement.
 * <p>
 * <h3>Lifecycle guarantees</h3> Most methods in this type are based on
 * {@link #fetch()}, which completes the whole {@link ConnectionProvider} and
 * {@link ExecuteListener} lifecycles, eagerly fetching all results into memory.
 * Underlying JDBC {@link ResultSet}s are always closed. Underlying JDBC
 * {@link PreparedStatement}s are closed, unless {@link #keepStatement(boolean)}
 * is set.
 * <p>
 * In order to keep open {@link ResultSet}s and fetch records lazily, use
 * {@link #fetchLazy()} instead and then operate on {@link Cursor}.
 *
 * @author Lukas Eder
 */
public interface ResultQuery<R extends Record> extends Query, Iterable<R> {

    /**
     * Return the result generated by a previous call to execute().
     *
     * @return The result or <code>null</code> if no call to execute() was done
     *         previously.
     */
    Result<R> getResult();

    /**
     * Execute the query and return the generated result.
     * <p>
     * This is the same as calling {@link #execute()} and then
     * {@link #getResult()}
     * <p>
     * The result and its contained records are attached to the original
     * {@link Configuration} by default. Use {@link Settings#isAttachRecords()}
     * to override this behaviour.
     * <h3>Lifecycle guarantees</h3> This method completes the whole
     * {@link ConnectionProvider} and {@link ExecuteListener} lifecycles,
     * eagerly fetching all results into memory. Underlying JDBC
     * {@link ResultSet}s are always closed. Underlying JDBC
     * {@link PreparedStatement}s are closed, unless
     * {@link #keepStatement(boolean)} is set.
     * <p>
     * In order to keep open {@link ResultSet}s and fetch records lazily, use
     * {@link #fetchLazy()} instead and then operate on {@link Cursor}.
     *
     * @return The result.
     * @throws DataAccessException if something went wrong executing the query
     */
    Result<R> fetch() throws DataAccessException;

    /**
     * Execute the query and return the generated result as a JDBC
     * {@link ResultSet}.
     * <p>
     * This is the same as calling {@link #fetchLazy()}.
     * {@link Cursor#resultSet() resultSet()} and will return a
     * {@link ResultSet} wrapping the JDBC driver's <code>ResultSet</code>.
     * Closing this <code>ResultSet</code> may close the producing
     * {@link Statement} or {@link PreparedStatement}, depending on your setting
     * for {@link #keepStatement(boolean)}.
     * <p>
     * You can use this method when you want to use jOOQ for query execution,
     * but not for result fetching. The returned <code>ResultSet</code> can also
     * be used with {@link DSLContext#fetch(ResultSet)}.
     *
     * @return The result.
     * @throws DataAccessException if something went wrong executing the query
     */
    ResultSet fetchResultSet() throws DataAccessException;

    /**
     * Execute the query and return the generated result.
     * <p>
     * This is essentially the same as {@link #fetch()}, except that being
     * declared in {@link Iterable}, this method can be used in Java 5 foreach
     * statements.
     * <p>
     * {@inheritDoc}
     */
    @Override
    Iterator<R> iterator() throws DataAccessException;

    /**
     * Execute the query and "lazily" return the generated result.
     * <p>
     * The returned {@link Cursor} holds a reference to the executed
     * {@link PreparedStatement} and the associated {@link ResultSet}. Data can
     * be fetched (or iterated over) lazily, fetching records from the
     * {@link ResultSet} one by one.
     * <p>
     * Depending on your JDBC driver's default behaviour, this may load the
     * whole database result into the driver's memory. In order to indicate to
     * the driver that you may not want to fetch all records at once, use
     * {@link #fetchLazy(int)}
     * <p>
     * Client code is responsible for closing the cursor after use.
     *
     * @return The resulting cursor.
     * @throws DataAccessException if something went wrong executing the query
     * @see #fetchLazy(int)
     */
    Cursor<R> fetchLazy() throws DataAccessException;

    /**
     * Execute the query and "lazily" return the generated result.
     * <p>
     * The returned {@link Cursor} holds a reference to the executed
     * {@link PreparedStatement} and the associated {@link ResultSet}. Data can
     * be fetched (or iterated over) lazily, fetching records from the
     * {@link ResultSet} one by one.
     * <p>
     * Depending on your JDBC driver's behaviour, this will load only
     * <code>fetchSize</code> records from the database into memory at once. For
     * more details, see also {@link Statement#setFetchSize(int)}
     * <p>
     * Client code is responsible for closing the cursor after use.
     *
     * @return The resulting cursor.
     * @throws DataAccessException if something went wrong executing the query
     * @see #fetchLazy()
     * @see Statement#setFetchSize(int)
     * @deprecated - [#2811] - 3.3.0 - Use {@link #fetchSize(int)} and
     *             {@link #fetchLazy()} instead.
     */
    @Deprecated
    Cursor<R> fetchLazy(int fetchSize) throws DataAccessException;

    /**
     * Execute a query, possibly returning several result sets.
     * <p>
     * Example (Sybase ASE):
     * <p>
     * <code><pre>
     * String sql = "sp_help 'my_table'";</pre></code>
     * <p>
     * The result and its contained records are attached to the original
     * {@link Configuration} by default. Use {@link Settings#isAttachRecords()}
     * to override this behaviour.
     *
     * @return The resulting records
     * @throws DataAccessException if something went wrong executing the query
     */
    List<Result<Record>> fetchMany() throws DataAccessException;

    /**
     * Execute the query and return all values for a field from the generated
     * result.
     * <p>
     * This is the same as calling {@link #fetch()} and then
     * {@link Result#getValues(Field)}
     *
     * @return The resulting values.
     * @throws DataAccessException if something went wrong executing the query
     */
    <T> List<T> fetch(Field<T> field) throws DataAccessException;

    /**
     * Execute the query and return all values for a field from the generated
     * result.
     * <p>
     * This is the same as calling {@link #fetch()} and then
     * {@link Result#getValues(Field, Class)}
     *
     * @return The resulting values.
     * @throws DataAccessException if something went wrong executing the query
     * @see Record#getValue(Field, Class)
     */
    <T> List<T> fetch(Field<?> field, Class<? extends T> type) throws DataAccessException;

    /**
     * Execute the query and return all values for a field from the generated
     * result.
     * <p>
     * This is the same as calling {@link #fetch()} and then
     * {@link Result#getValues(Field, Converter)}
     *
     * @return The resulting values.
     * @throws DataAccessException if something went wrong executing the query
     * @see Record#getValue(Field, Converter)
     */
    <T, U> List<U> fetch(Field<T> field, Converter<? super T, U> converter) throws DataAccessException;

    /**
     * Execute the query and return all values for a field index from the
     * generated result.
     * <p>
     * This is the same as calling {@link #fetch()} and then
     * {@link Result#getValues(int)}
     *
     * @return The resulting values.
     * @throws DataAccessException if something went wrong executing the query
     */
    List<?> fetch(int fieldIndex) throws DataAccessException;

    /**
     * Execute the query and return all values for a field index from the
     * generated result.
     * <p>
     * This is the same as calling {@link #fetch()} and then
     * {@link Result#getValues(int, Class)}
     *
     * @return The resulting values.
     * @throws DataAccessException if something went wrong executing the query
     * @see Record#getValue(int, Class)
     */
    <T> List<T> fetch(int fieldIndex, Class<? extends T> type) throws DataAccessException;

    /**
     * Execute the query and return all values for a field index from the
     * generated result.
     * <p>
     * This is the same as calling {@link #fetch()} and then
     * {@link Result#getValues(int, Converter)}
     *
     * @return The resulting values.
     * @throws DataAccessException if something went wrong executing the query
     * @see Record#getValue(int, Converter)
     */
    <U> List<U> fetch(int fieldIndex, Converter<?, U> converter) throws DataAccessException;

    /**
     * Execute the query and return all values for a field name from the
     * generated result.
     * <p>
     * This is the same as calling {@link #fetch()} and then
     * {@link Result#getValues(String)}
     *
     * @return The resulting values.
     * @throws DataAccessException if something went wrong executing the query
     */
    List<?> fetch(String fieldName) throws DataAccessException;

    /**
     * Execute the query and return all values for a field name from the
     * generated result.
     * <p>
     * This is the same as calling {@link #fetch()} and then
     * {@link Result#getValues(String, Class)}
     *
     * @return The resulting values.
     * @throws DataAccessException if something went wrong executing the query
     * @see Record#getValue(String, Class)
     */
    <T> List<T> fetch(String fieldName, Class<? extends T> type) throws DataAccessException;

    /**
     * Execute the query and return all values for a field name from the
     * generated result.
     * <p>
     * This is the same as calling {@link #fetch()} and then
     * {@link Result#getValues(String, Converter)}
     *
     * @return The resulting values.
     * @throws DataAccessException if something went wrong executing the query
     * @see Record#getValue(String, Converter)
     */
    <U> List<U> fetch(String fieldName, Converter<?, U> converter) throws DataAccessException;

    /**
     * Execute the query and return at most one resulting value for a
     * field from the generated result.
     * <p>
     * This is the same as calling {@link #fetchOne()} and then
     * {@link Record#getValue(Field)}
     *
     * @return The resulting value or <code>null</code> if the query returned no
     *         records.
     * @throws DataAccessException if something went wrong executing the query
     * @throws TooManyRowsException if the query returned more than one record
     */
    <T> T fetchOne(Field<T> field) throws DataAccessException, TooManyRowsException;

    /**
     * Execute the query and return at most one resulting value for a
     * field from the generated result.
     * <p>
     * This is the same as calling {@link #fetchOne()} and then
     * {@link Record#getValue(Field, Class)}
     *
     * @return The resulting value or <code>null</code> if the query returned no
     *         records.
     * @throws DataAccessException if something went wrong executing the query
     * @throws TooManyRowsException if the query returned more than one record
     */
    <T> T fetchOne(Field<?> field, Class<? extends T> type) throws DataAccessException, TooManyRowsException;

    /**
     * Execute the query and return at most one resulting value for a
     * field from the generated result.
     * <p>
     * This is the same as calling {@link #fetchOne()} and then
     * {@link Record#getValue(Field, Converter)}
     *
     * @return The resulting value or <code>null</code> if the query returned no
     *         records.
     * @throws DataAccessException if something went wrong executing the query
     * @throws TooManyRowsException if the query returned more than one record
     */
    <T, U> U fetchOne(Field<T> field, Converter<? super T, U> converter) throws DataAccessException, TooManyRowsException;

    /**
     * Execute the query and return at most one resulting value for a
     * field index from the generated result.
     * <p>
     * This is the same as calling {@link #fetchOne()} and then
     * {@link Record#getValue(int)}
     *
     * @return The resulting value or <code>null</code> if the query returned no
     *         records.
     * @throws DataAccessException if something went wrong executing the query
     * @throws TooManyRowsException if the query returned more than one record
     */
    Object fetchOne(int fieldIndex) throws DataAccessException, TooManyRowsException;

    /**
     * Execute the query and return at most one resulting value for a
     * field index from the generated result.
     * <p>
     * This is the same as calling {@link #fetchOne()} and then
     * {@link Record#getValue(int, Class)}
     *
     * @return The resulting value or <code>null</code> if the query returned no
     *         records.
     * @throws DataAccessException if something went wrong executing the query
     * @throws TooManyRowsException if the query returned more than one record
     */
    <T> T fetchOne(int fieldIndex, Class<? extends T> type) throws DataAccessException, TooManyRowsException;

    /**
     * Execute the query and return at most one resulting value for a
     * field index from the generated result.
     * <p>
     * This is the same as calling {@link #fetchOne()} and then
     * {@link Record#getValue(int, Converter)}
     *
     * @return The resulting value or <code>null</code> if the query returned no
     *         records.
     * @throws DataAccessException if something went wrong executing the query
     * @throws TooManyRowsException if the query returned more than one record
     */
    <U> U fetchOne(int fieldIndex, Converter<?, U> converter) throws DataAccessException, TooManyRowsException;

    /**
     * Execute the query and return at most one resulting value for a
     * field name from the generated result.
     * <p>
     * This is the same as calling {@link #fetchOne()} and then
     * {@link Record#getValue(int)}
     *
     * @return The resulting value or <code>null</code> if the query returned no
     *         records.
     * @throws DataAccessException if something went wrong executing the query
     * @throws TooManyRowsException if the query returned more than one record
     */
    Object fetchOne(String fieldName) throws DataAccessException, TooManyRowsException;

    /**
     * Execute the query and return at most one resulting value for a
     * field name from the generated result.
     * <p>
     * This is the same as calling {@link #fetchOne()} and then
     * {@link Record#getValue(String, Class)}
     *
     * @return The resulting value or <code>null</code> if the query returned no
     *         records.
     * @throws DataAccessException if something went wrong executing the query
     * @throws TooManyRowsException if the query returned more than one record
     */
    <T> T fetchOne(String fieldName, Class<? extends T> type) throws DataAccessException, TooManyRowsException;

    /**
     * Execute the query and return at most one resulting value for a
     * field name from the generated result.
     * <p>
     * This is the same as calling {@link #fetchOne()} and then
     * {@link Record#getValue(String, Converter)}
     *
     * @return The resulting value or <code>null</code> if the query returned no
     *         records.
     * @throws DataAccessException if something went wrong executing the query
     * @throws TooManyRowsException if the query returned more than one record
     */
    <U> U fetchOne(String fieldName, Converter<?, U> converter) throws DataAccessException, TooManyRowsException;

    /**
     * Execute the query and return at most one resulting record.
     * <p>
     * The resulting record is attached to the original {@link Configuration} by
     * default. Use {@link Settings#isAttachRecords()} to override this
     * behaviour.
     *
     * @return The resulting record or <code>null</code> if the query returns no
     *         records.
     * @throws DataAccessException if something went wrong executing the query
     * @throws TooManyRowsException if the query returned more than one record
     */
    R fetchOne() throws DataAccessException, TooManyRowsException;

    /**
     * Execute the query and return at most one resulting value into a
     * custom mapper callback.
     *
     * @return The custom mapped record or <code>null</code> if the query returned no
     *         records.
     * @throws DataAccessException if something went wrong executing the query
     * @throws TooManyRowsException if the query returned more than one record
     */
    <E> E fetchOne(RecordMapper<? super R, E> mapper) throws DataAccessException, TooManyRowsException;

    /**
     * Execute the query and return at most one resulting record as a name/value
     * map.
     *
     * @return The resulting record or <code>null</code> if the query returns no
     *         records.
     * @throws DataAccessException if something went wrong executing the query
     * @throws TooManyRowsException if the query returned more than one record
     * @see Result#intoMaps()
     * @see Record#intoMap()
     */
    Map<String, Object> fetchOneMap() throws DataAccessException, TooManyRowsException;

    /**
     * Execute the query and return at most one resulting record as an array
     * <p>
     * You can access data like this
     * <code><pre>query.fetchOneArray()[fieldIndex]</pre></code>
     *
     * @return The resulting record or <code>null</code> if the query returns no
     *         records.
     * @throws DataAccessException if something went wrong executing the query
     * @throws TooManyRowsException if the query returned more than one record
     */
    Object[] fetchOneArray() throws DataAccessException, TooManyRowsException;

    /**
     * Map resulting records onto a custom type.
     * <p>
     * This is the same as calling <code><pre>
     * E result = null;
     * Record r = q.fetchOne();
     *
     * if (r != null)
     *     result = r.into(type);
     * </pre></code>. See {@link Record#into(Class)} for more details
     *
     * @param <E> The generic entity type.
     * @param type The entity type.
     * @return The resulting record or <code>null</code> if the query returns no
     *         records.
     * @see Record#into(Class)
     * @see Result#into(Class)
     * @throws DataAccessException if something went wrong executing the query
     * @throws MappingException wrapping any reflection or data type conversion
     *             exception that might have occurred while mapping records
     * @throws TooManyRowsException if the query returned more than one record
     * @see DefaultRecordMapper
     */
    <E> E fetchOneInto(Class<? extends E> type) throws DataAccessException, MappingException, TooManyRowsException;

    /**
     * Map resulting records onto a custom record.
     * <p>
     * This is the same as calling <code><pre>
     * Z result = null;
     * Record r = q.fetchOne();
     *
     * if (r != null)
     *     result = r.into(table);
     * </pre></code>. See {@link Record#into(Table)} for more details
     * <p>
     * The resulting record is attached to the original {@link Configuration} by
     * default. Use {@link Settings#isAttachRecords()} to override this
     * behaviour.
     *
     * @param <Z> The generic table record type.
     * @param table The table type.
     * @return The resulting record or <code>null</code> if the query returns no
     *         records.
     * @see Record#into(Table)
     * @see Result#into(Table)
     * @throws DataAccessException if something went wrong executing the query
     * @throws TooManyRowsException if the query returned more than one record
     */
    <Z extends Record> Z fetchOneInto(Table<Z> table) throws DataAccessException, TooManyRowsException;

    /**
     * Execute the query and return at most one resulting value for a
     * field from the generated result.
     * <p>
     * This is the same as calling {@link #fetchOne()} and then
     * {@link Record#getValue(Field)}
     *
     * @return The resulting value or <code>null</code> if the query returned no
     *         records.
     * @throws DataAccessException if something went wrong executing the query
     */
    <T> T fetchAny(Field<T> field) throws DataAccessException;

    /**
     * Execute the query and return at most one resulting value for a
     * field from the generated result.
     * <p>
     * This is the same as calling {@link #fetchOne()} and then
     * {@link Record#getValue(Field, Class)}
     *
     * @return The resulting value or <code>null</code> if the query returned no
     *         records.
     * @throws DataAccessException if something went wrong executing the query
     */
    <T> T fetchAny(Field<?> field, Class<? extends T> type) throws DataAccessException;

    /**
     * Execute the query and return at most one resulting value for a
     * field from the generated result.
     * <p>
     * This is the same as calling {@link #fetchOne()} and then
     * {@link Record#getValue(Field, Converter)}
     *
     * @return The resulting value or <code>null</code> if the query returned no
     *         records.
     * @throws DataAccessException if something went wrong executing the query
     */
    <T, U> U fetchAny(Field<T> field, Converter<? super T, U> converter) throws DataAccessException;

    /**
     * Execute the query and return at most one resulting value for a
     * field index from the generated result.
     * <p>
     * This is the same as calling {@link #fetchOne()} and then
     * {@link Record#getValue(int)}
     *
     * @return The resulting value or <code>null</code> if the query returned no
     *         records.
     * @throws DataAccessException if something went wrong executing the query
     */
    Object fetchAny(int fieldIndex) throws DataAccessException;

    /**
     * Execute the query and return at most one resulting value for a
     * field index from the generated result.
     * <p>
     * This is the same as calling {@link #fetchOne()} and then
     * {@link Record#getValue(int, Class)}
     *
     * @return The resulting value or <code>null</code> if the query returned no
     *         records.
     * @throws DataAccessException if something went wrong executing the query
     */
    <T> T fetchAny(int fieldIndex, Class<? extends T> type) throws DataAccessException;

    /**
     * Execute the query and return at most one resulting value for a
     * field index from the generated result.
     * <p>
     * This is the same as calling {@link #fetchOne()} and then
     * {@link Record#getValue(int, Converter)}
     *
     * @return The resulting value or <code>null</code> if the query returned no
     *         records.
     * @throws DataAccessException if something went wrong executing the query
     */
    <U> U fetchAny(int fieldIndex, Converter<?, U> converter) throws DataAccessException;

    /**
     * Execute the query and return at most one resulting value for a
     * field name from the generated result.
     * <p>
     * This is the same as calling {@link #fetchOne()} and then
     * {@link Record#getValue(int)}
     *
     * @return The resulting value or <code>null</code> if the query returned no
     *         records.
     * @throws DataAccessException if something went wrong executing the query
     */
    Object fetchAny(String fieldName) throws DataAccessException;

    /**
     * Execute the query and return at most one resulting value for a
     * field name from the generated result.
     * <p>
     * This is the same as calling {@link #fetchOne()} and then
     * {@link Record#getValue(String, Class)}
     *
     * @return The resulting value or <code>null</code> if the query returned no
     *         records.
     * @throws DataAccessException if something went wrong executing the query
     */
    <T> T fetchAny(String fieldName, Class<? extends T> type) throws DataAccessException;

    /**
     * Execute the query and return at most one resulting value for a
     * field name from the generated result.
     * <p>
     * This is the same as calling {@link #fetchOne()} and then
     * {@link Record#getValue(String, Converter)}
     *
     * @return The resulting value or <code>null</code> if the query returned no
     *         records.
     * @throws DataAccessException if something went wrong executing the query
     */
    <U> U fetchAny(String fieldName, Converter<?, U> converter) throws DataAccessException;

    /**
     * Execute the query and return at most one resulting record.
     * <p>
     * The resulting record is attached to the original {@link Configuration} by
     * default. Use {@link Settings#isAttachRecords()} to override this
     * behaviour.
     *
     * @return The first resulting record or <code>null</code> if the query
     *         returns no records.
     * @throws DataAccessException if something went wrong executing the query
     */
    R fetchAny() throws DataAccessException;

    /**
     * Execute the query and return at most one resulting record as a name/value
     * map.
     *
     * @return The resulting record or <code>null</code> if the query returns no
     *         records.
     * @throws DataAccessException if something went wrong executing the query
     * @see Result#intoMaps()
     * @see Record#intoMap()
     */
    Map<String, Object> fetchAnyMap() throws DataAccessException;

    /**
     * Execute the query and return at most one resulting record as an array
     * <p>
     * You can access data like this
     * <code><pre>query.fetchAnyArray()[fieldIndex]</pre></code>
     *
     * @return The resulting record or <code>null</code> if the query returns no
     *         records.
     * @throws DataAccessException if something went wrong executing the query
     */
    Object[] fetchAnyArray() throws DataAccessException;

    /**
     * Map resulting records onto a custom type.
     * <p>
     * This is the same as calling <code><pre>
     * E result = null;
     * Record r = q.fetchAny();
     *
     * if (r != null)
     *     result = r.into(type);
     * </pre></code>. See {@link Record#into(Class)} for more details
     *
     * @param <E> The generic entity type.
     * @param type The entity type.
     * @return The resulting record or <code>null</code> if the query returns no
     *         records.
     * @see Record#into(Class)
     * @see Result#into(Class)
     * @throws DataAccessException if something went wrong executing the query
     * @throws MappingException wrapping any reflection or data type conversion
     *             exception that might have occurred while mapping records
     * @see DefaultRecordMapper
     */
    <E> E fetchAnyInto(Class<? extends E> type) throws DataAccessException, MappingException;

    /**
     * Map resulting records onto a custom record.
     * <p>
     * This is the same as calling <code><pre>
     * Z result = null;
     * Record r = q.fetchOne();
     *
     * if (r != null)
     *     result = r.into(table);
     * </pre></code>. See {@link Record#into(Table)} for more details
     * <p>
     * The resulting record is attached to the original {@link Configuration} by
     * default. Use {@link Settings#isAttachRecords()} to override this
     * behaviour.
     *
     * @param <Z> The generic table record type.
     * @param table The table type.
     * @return The resulting record or <code>null</code> if the query returns no
     *         records.
     * @see Record#into(Table)
     * @see Result#into(Table)
     * @throws DataAccessException if something went wrong executing the query
     */
    <Z extends Record> Z fetchAnyInto(Table<Z> table) throws DataAccessException;

    /**
     * Execute the query and return the generated result as a list of name/value
     * maps.
     *
     * @return The result.
     * @throws DataAccessException if something went wrong executing the query
     * @throws InvalidResultException if the key field returned two or more
     *             equal values from the result set.
     * @see Result#intoMaps()
     * @see Record#intoMap()
     */
    List<Map<String, Object>> fetchMaps() throws DataAccessException;

    /**
     * Execute the query and return a {@link Map} with one of the result's
     * columns as key and the corresponding records as value.
     * <p>
     * An exception is thrown, if the key turns out to be non-unique in the
     * result set. Use {@link #fetchGroups(Field)} instead, if your keys are
     * non-unique
     * <p>
     * The resulting records are attached to the original {@link Configuration}
     * by default. Use {@link Settings#isAttachRecords()} to override this
     * behaviour.
     *
     * @param <K> The key's generic field type
     * @param key The key field. Client code must assure that this field is
     *            unique in the result set.
     * @return A Map containing the results
     * @throws DataAccessException if something went wrong executing the query
     * @throws InvalidResultException if the key field returned two or more
     *             equal values from the result set.
     * @see Result#intoMap(Field)
     */
    <K> Map<K, R> fetchMap(Field<K> key) throws DataAccessException;

    /**
     * Execute the query and return a {@link Map} with one of the result's
     * columns as key and the corresponding records as value.
     * <p>
     * An exception is thrown, if the key turns out to be non-unique in the
     * result set. Use {@link #fetchGroups(int)} instead, if your keys are
     * non-unique
     * <p>
     * The resulting records are attached to the original {@link Configuration}
     * by default. Use {@link Settings#isAttachRecords()} to override this
     * behaviour.
     *
     * @param keyFieldIndex The key field. Client code must assure that this
     *            field is unique in the result set.
     * @return A Map containing the results
     * @throws DataAccessException if something went wrong executing the query
     * @throws InvalidResultException if the key field returned two or more
     *             equal values from the result set.
     * @see Result#intoMap(int)
     */
    Map<?, R> fetchMap(int keyFieldIndex) throws DataAccessException;

    /**
     * Execute the query and return a {@link Map} with one of the result's
     * columns as key and the corresponding records as value.
     * <p>
     * An exception is thrown, if the key turns out to be non-unique in the
     * result set. Use {@link #fetchGroups(String)} instead, if your keys are
     * non-unique
     * <p>
     * The resulting records are attached to the original {@link Configuration}
     * by default. Use {@link Settings#isAttachRecords()} to override this
     * behaviour.
     *
     * @param keyFieldName The key field. Client code must assure that this
     *            field is unique in the result set.
     * @return A Map containing the results
     * @throws DataAccessException if something went wrong executing the query
     * @throws InvalidResultException if the key field returned two or more
     *             equal values from the result set.
     * @see Result#intoMap(String)
     */
    Map<?, R> fetchMap(String keyFieldName) throws DataAccessException;

    /**
     * Execute the query and return a {@link Map} with one of the result's
     * columns as key and another one of the result's columns as value
     * <p>
     * An exception is thrown, if the key turns out to be non-unique in the
     * result set. Use {@link #fetchGroups(Field, Field)} instead, if your keys
     * are non-unique
     *
     * @param <K> The key's generic field type
     * @param <V> The value's generic field type
     * @param key The key field. Client code must assure that this field is
     *            unique in the result set.
     * @param value The value field
     * @return A Map containing the results
     * @throws DataAccessException if something went wrong executing the query
     * @throws InvalidResultException if the key field returned two or more
     *             equal values from the result set.
     * @see Result#intoMap(Field, Field)
     */
    <K, V> Map<K, V> fetchMap(Field<K> key, Field<V> value) throws DataAccessException;

    /**
     * Execute the query and return a {@link Map} with one of the result's
     * columns as key and another one of the result's columns as value
     * <p>
     * An exception is thrown, if the key turns out to be non-unique in the
     * result set. Use {@link #fetchGroups(int, int)} instead, if your keys are
     * non-unique
     *
     * @param keyFieldIndex The key field. Client code must assure that this
     *            field is unique in the result set.
     * @param valueFieldIndex The value field
     * @return A Map containing the results
     * @throws DataAccessException if something went wrong executing the query
     * @throws InvalidResultException if the key field returned two or more
     *             equal values from the result set.
     * @see Result#intoMap(int, int)
     */
    Map<?, ?> fetchMap(int keyFieldIndex, int valueFieldIndex) throws DataAccessException;

    /**
     * Execute the query and return a {@link Map} with one of the result's
     * columns as key and another one of the result's columns as value
     * <p>
     * An exception is thrown, if the key turns out to be non-unique in the
     * result set. Use {@link #fetchGroups(String, String)} instead, if your keys
     * are non-unique
     *
     * @param keyFieldName The key field. Client code must assure that this
     *            field is unique in the result set.
     * @param valueFieldName The value field
     * @return A Map containing the results
     * @throws DataAccessException if something went wrong executing the query
     * @throws InvalidResultException if the key field returned two or more
     *             equal values from the result set.
     * @see Result#intoMap(String, String)
     */
    Map<?, ?> fetchMap(String keyFieldName, String valueFieldName) throws DataAccessException;

    /**
     * Execute the query and return a {@link Map} with keys as a map key and the
     * corresponding record as value.
     * <p>
     * An exception is thrown, if the keys turn out to be non-unique in the
     * result set. Use {@link #fetchGroups(Field[])} instead, if your keys are
     * non-unique.
     *
     * @param keys The keys. Client code must assure that keys are unique in the
     *            result set.
     * @return A Map containing the results.
     * @throws DataAccessException if something went wrong executing the query
     * @throws InvalidResultException if the key list is non-unique in the
     *             result set.
     * @see Result#intoMap(Field[])
     */
    Map<Record, R> fetchMap(Field<?>[] keys) throws DataAccessException;

    /**
     * Execute the query and return a {@link Map} with keys as a map key and the
     * corresponding record as value.
     * <p>
     * An exception is thrown, if the keys turn out to be non-unique in the
     * result set. Use {@link #fetchGroups(int[])} instead, if your keys are
     * non-unique.
     *
     * @param keyFieldIndexes The keys. Client code must assure that keys are
     *            unique in the result set.
     * @return A Map containing the results.
     * @throws DataAccessException if something went wrong executing the query
     * @throws InvalidResultException if the key list is non-unique in the
     *             result set.
     * @see Result#intoMap(int[])
     */
    Map<Record, R> fetchMap(int[] keyFieldIndexes) throws DataAccessException;

    /**
     * Execute the query and return a {@link Map} with keys as a map key and the
     * corresponding record as value.
     * <p>
     * An exception is thrown, if the keys turn out to be non-unique in the
     * result set. Use {@link #fetchGroups(String[])} instead, if your keys are
     * non-unique.
     *
     * @param keyFieldNames The keys. Client code must assure that keys are
     *            unique in the result set.
     * @return A Map containing the results.
     * @throws DataAccessException if something went wrong executing the query
     * @throws InvalidResultException if the key list is non-unique in the
     *             result set.
     * @see Result#intoMap(String[])
     */
    Map<Record, R> fetchMap(String[] keyFieldNames) throws DataAccessException;

    /**
     * Execute the query and return a {@link Map} with results grouped by the
     * given keys and mapped into the given entity type.
     * <p>
     * An {@link InvalidResultException} is thrown, if the keys are non-unique
     * in the result set. Use {@link #fetchGroups(Field[], Class)} instead, if
     * your keys are non-unique.
     *
     * @param keys The keys. Client code must assure that keys are unique in the
     *            result set. If this is <code>null</code> or an empty array,
     *            the resulting map will contain at most one entry.
     * @param type The entity type.
     * @return A Map containing the results.
     * @throws DataAccessException if something went wrong executing the query
     * @throws InvalidResultException if the keys are non-unique in the result
     *             set.
     * @throws MappingException wrapping any reflection or data type conversion
     *             exception that might have occurred while mapping records
     * @see Result#intoMap(Field[], Class)
     * @see DefaultRecordMapper
     */
    <E> Map<List<?>, E> fetchMap(Field<?>[] keys, Class<? extends E> type) throws DataAccessException, MappingException;

    /**
     * Execute the query and return a {@link Map} with results grouped by the
     * given keys and mapped into the given entity type.
     * <p>
     * An {@link InvalidResultException} is thrown, if the keys are non-unique
     * in the result set. Use {@link #fetchGroups(int[], Class)} instead, if
     * your keys are non-unique.
     *
     * @param keyFieldIndexes The keys. Client code must assure that keys are
     *            unique in the result set. If this is <code>null</code> or an
     *            empty array, the resulting map will contain at most one entry.
     * @param type The entity type.
     * @return A Map containing the results.
     * @throws DataAccessException if something went wrong executing the query
     * @throws InvalidResultException if the keys are non-unique in the result
     *             set.
     * @throws MappingException wrapping any reflection or data type conversion
     *             exception that might have occurred while mapping records
     * @see Result#intoMap(int[], Class)
     * @see DefaultRecordMapper
     */
    <E> Map<List<?>, E> fetchMap(int[] keyFieldIndexes, Class<? extends E> type) throws DataAccessException,
        MappingException;

    /**
     * Execute the query and return a {@link Map} with results grouped by the
     * given keys and mapped into the given entity type.
     * <p>
     * An {@link InvalidResultException} is thrown, if the keys are non-unique
     * in the result set. Use {@link #fetchGroups(String[], Class)} instead, if
     * your keys are non-unique.
     *
     * @param keyFieldNames The keys. Client code must assure that keys are
     *            unique in the result set. If this is <code>null</code> or an
     *            empty array, the resulting map will contain at most one entry.
     * @param type The entity type.
     * @return A Map containing the results.
     * @throws DataAccessException if something went wrong executing the query
     * @throws InvalidResultException if the keys are non-unique in the result
     *             set.
     * @throws MappingException wrapping any reflection or data type conversion
     *             exception that might have occurred while mapping records
     * @see Result#intoMap(String[], Class)
     * @see DefaultRecordMapper
     */
    <E> Map<List<?>, E> fetchMap(String[] keyFieldNames, Class<? extends E> type) throws DataAccessException,
        MappingException;

    /**
     * Execute the query and return a {@link Map} with results grouped by the
     * given keys and mapped by the given mapper.
     * <p>
     * An {@link InvalidResultException} is thrown, if the keys are non-unique
     * in the result set. Use {@link #fetchGroups(Field[], RecordMapper)}
     * instead, if your keys are non-unique.
     *
     * @param keys The keys. Client code must assure that keys are unique in the
     *            result set. If this is <code>null</code> or an empty array,
     *            the resulting map will contain at most one entry.
     * @param mapper The mapper callback.
     * @return A Map containing the results.
     * @throws DataAccessException if something went wrong executing the query
     * @throws InvalidResultException if the keys are non-unique in the result
     *             set.
     * @throws MappingException wrapping any reflection or data type conversion
     *             exception that might have occurred while mapping records
     * @see Result#intoMap(Field[], Class)
     * @see DefaultRecordMapper
     */
    <E> Map<List<?>, E> fetchMap(Field<?>[] keys, RecordMapper<? super R, E> mapper) throws DataAccessException,
        MappingException;

    /**
     * Execute the query and return a {@link Map} with results grouped by the
     * given keys and mapped by the given mapper.
     * <p>
     * An {@link InvalidResultException} is thrown, if the keys are non-unique
     * in the result set. Use {@link #fetchGroups(int[], RecordMapper)} instead,
     * if your keys are non-unique.
     *
     * @param keyFieldIndexes The keys. Client code must assure that keys are
     *            unique in the result set. If this is <code>null</code> or an
     *            empty array, the resulting map will contain at most one entry.
     * @param mapper The mapper callback.
     * @return A Map containing the results.
     * @throws DataAccessException if something went wrong executing the query
     * @throws InvalidResultException if the keys are non-unique in the result
     *             set.
     * @throws MappingException wrapping any reflection or data type conversion
     *             exception that might have occurred while mapping records
     * @see Result#intoMap(int[], Class)
     * @see DefaultRecordMapper
     */
    <E> Map<List<?>, E> fetchMap(int[] keyFieldIndexes, RecordMapper<? super R, E> mapper) throws DataAccessException,
        MappingException;

    /**
     * Execute the query and return a {@link Map} with results grouped by the
     * given keys and mapped by the given mapper.
     * <p>
     * An {@link InvalidResultException} is thrown, if the keys are non-unique
     * in the result set. Use {@link #fetchGroups(String[], RecordMapper)}
     * instead, if your keys are non-unique.
     *
     * @param keyFieldNames The keys. Client code must assure that keys are
     *            unique in the result set. If this is <code>null</code> or an
     *            empty array, the resulting map will contain at most one entry.
     * @param mapper The mapper callback.
     * @return A Map containing the results.
     * @throws DataAccessException if something went wrong executing the query
     * @throws InvalidResultException if the keys are non-unique in the result
     *             set.
     * @throws MappingException wrapping any reflection or data type conversion
     *             exception that might have occurred while mapping records
     * @see Result#intoMap(String[], Class)
     * @see DefaultRecordMapper
     */
    <E> Map<List<?>, E> fetchMap(String[] keyFieldNames, RecordMapper<? super R, E> mapper) throws DataAccessException,
        MappingException;

    /**
     * Execute the query and return a {@link Map} with table as a map key and the
     * corresponding record as value.
     * <p>
     * An exception is thrown, if the keys turn out to be non-unique in the
     * result set. Use {@link #fetchGroups(Table)} instead, if your keys are
     * non-unique.
     *
     * @param table The key table. Client code must assure that keys are unique
     *            in the result set. May not be <code>null</code>.
     * @return A Map containing the results.
     * @throws DataAccessException if something went wrong executing the query
     * @throws InvalidResultException if the key list is non-unique in the
     *             result set.
     * @see Result#intoMap(Table)
     */
    <S extends Record> Map<S, R> fetchMap(Table<S> table) throws DataAccessException;

    /**
     * Execute the query and return a {@link Map} with results grouped by the
     * given table and mapped into the given entity type.
     * <p>
     * An {@link InvalidResultException} is thrown, if the keys are non-unique
     * in the result set. Use {@link #fetchGroups(Table, Class)} instead, if
     * your keys are non-unique.
     *
     * @param table The key table. Client code must assure that keys are unique
     *            in the result set. May not be <code>null</code>.
     * @param type The entity type.
     * @return A Map containing the results.
     * @throws DataAccessException if something went wrong executing the query
     * @throws InvalidResultException if the keys are non-unique in the result
     *             set.
     * @throws MappingException wrapping any reflection or data type conversion
     *             exception that might have occurred while mapping records
     * @see Result#intoMap(Table, Class)
     * @see DefaultRecordMapper
     */
    <E, S extends Record> Map<S, E> fetchMap(Table<S> table, Class<? extends E> type) throws DataAccessException,
        MappingException;

    /**
     * Execute the query and return a {@link Map} with results grouped by the
     * given table and mapped by the given mapper.
     * <p>
     * An {@link InvalidResultException} is thrown, if the keys are non-unique
     * in the result set. Use {@link #fetchGroups(Table, RecordMapper)} instead,
     * if your keys are non-unique.
     *
     * @param table The key table. Client code must assure that keys are unique
     *            in the result set. May not be <code>null</code>.
     * @param mapper The mapper callback.
     * @return A Map containing the results.
     * @throws DataAccessException if something went wrong executing the query
     * @throws InvalidResultException if the keys are non-unique in the result
     *             set.
     * @throws MappingException wrapping any reflection or data type conversion
     *             exception that might have occurred while mapping records
     * @see Result#intoMap(Table, Class)
     * @see DefaultRecordMapper
     */
    <E, S extends Record> Map<S, E> fetchMap(Table<S> table, RecordMapper<? super R, E> mapper)
        throws DataAccessException, MappingException;

    /**
     * Execute the query and return a {@link Map} with results grouped by the
     * given key and mapped into the given entity type.
     * <p>
     * An exception is thrown, if the key turn out to be non-unique in the
     * result set. Use {@link #fetchGroups(Field, Class)} instead, if your key
     * is non-unique.
     *
     * @param key The key. Client code must assure that key is unique in the
     *            result set.
     * @param type The entity type.
     * @return A Map containing the result.
     * @throws DataAccessException if something went wrong executing the query
     * @throws InvalidResultException if the key is non-unique in the result
     *             set.
     * @see Result#intoMap(Field, Class)
     */
    <K, E> Map<K, E> fetchMap(Field<K> key, Class<? extends E> type) throws DataAccessException;

    /**
     * Execute the query and return a {@link Map} with results grouped by the
     * given key and mapped into the given entity type.
     * <p>
     * An exception is thrown, if the key turn out to be non-unique in the
     * result set. Use {@link #fetchGroups(int, Class)} instead, if your key
     * is non-unique.
     *
     * @param keyFieldIndex The key. Client code must assure that key is unique
     *            in the result set.
     * @param type The entity type.
     * @return A Map containing the result.
     * @throws DataAccessException if something went wrong executing the query
     * @throws InvalidResultException if the key is non-unique in the result
     *             set.
     * @see Result#intoMap(int, Class)
     */
    <E> Map<?, E> fetchMap(int keyFieldIndex, Class<? extends E> type) throws DataAccessException;

    /**
     * Execute the query and return a {@link Map} with results grouped by the
     * given key and mapped into the given entity type.
     * <p>
     * An exception is thrown, if the key turn out to be non-unique in the
     * result set. Use {@link #fetchGroups(String, Class)} instead, if your key
     * is non-unique.
     *
     * @param keyFieldName The key. Client code must assure that key is unique
     *            in the result set.
     * @param type The entity type.
     * @return A Map containing the result.
     * @throws DataAccessException if something went wrong executing the query
     * @throws InvalidResultException if the key is non-unique in the result
     *             set.
     * @see Result#intoMap(String, Class)
     */
    <E> Map<?, E> fetchMap(String keyFieldName, Class<? extends E> type) throws DataAccessException;

    /**
     * Execute the query and return a {@link Map} with results grouped by the
     * given key and mapped by the given mapper.
     * <p>
     * An exception is thrown, if the key turn out to be non-unique in the
     * result set. Use {@link #fetchGroups(Field, Class)} instead, if your key
     * is non-unique.
     *
     * @param key The key. Client code must assure that key is unique in the
     *            result set.
     * @param mapper The mapper callback.
     * @return A Map containing the result.
     * @throws DataAccessException if something went wrong executing the query
     * @throws InvalidResultException if the key is non-unique in the result
     *             set.
     * @see Result#intoMap(Field, Class)
     */
    <K, E> Map<K, E> fetchMap(Field<K> key, RecordMapper<? super R, E> mapper) throws DataAccessException;

    /**
     * Execute the query and return a {@link Map} with results grouped by the
     * given key and mapped by the given mapper.
     * <p>
     * An exception is thrown, if the key turn out to be non-unique in the
     * result set. Use {@link #fetchGroups(int, Class)} instead, if your key is
     * non-unique.
     *
     * @param keyFieldIndex The key. Client code must assure that key is unique
     *            in the result set.
     * @param mapper The mapper callback.
     * @return A Map containing the result.
     * @throws DataAccessException if something went wrong executing the query
     * @throws InvalidResultException if the key is non-unique in the result
     *             set.
     * @see Result#intoMap(int, Class)
     */
    <E> Map<?, E> fetchMap(int keyFieldIndex, RecordMapper<? super R, E> mapper) throws DataAccessException;

    /**
     * Execute the query and return a {@link Map} with results grouped by the
     * given key and mapped by the given mapper.
     * <p>
     * An exception is thrown, if the key turn out to be non-unique in the
     * result set. Use {@link #fetchGroups(String, Class)} instead, if your key
     * is non-unique.
     *
     * @param keyFieldName The key. Client code must assure that key is unique
     *            in the result set.
     * @param mapper The mapper callback.
     * @return A Map containing the result.
     * @throws DataAccessException if something went wrong executing the query
     * @throws InvalidResultException if the key is non-unique in the result
     *             set.
     * @see Result#intoMap(String, Class)
     */
    <E> Map<?, E> fetchMap(String keyFieldName, RecordMapper<? super R, E> mapper) throws DataAccessException;

    /**
     * Execute the query and return a {@link Map} with one of the result's
     * columns as key and a list of corresponding records as value.
     * <p>
     * Unlike {@link #fetchMap(Field)}, this method allows for non-unique keys
     * in the result set.
     * <p>
     * The resulting records are attached to the original {@link Configuration}
     * by default. Use {@link Settings#isAttachRecords()} to override this
     * behaviour.
     *
     * @param <K> The key's generic field type
     * @param key The key field.
     * @return A Map containing the results
     * @throws DataAccessException if something went wrong executing the query
     * @see Result#intoGroups(Field)
     */
    <K> Map<K, Result<R>> fetchGroups(Field<K> key) throws DataAccessException;

    /**
     * Execute the query and return a {@link Map} with one of the result's
     * columns as key and a list of corresponding records as value.
     * <p>
     * Unlike {@link #fetchMap(int)}, this method allows for non-unique keys in
     * the result set.
     * <p>
     * The resulting records are attached to the original {@link Configuration}
     * by default. Use {@link Settings#isAttachRecords()} to override this
     * behaviour.
     *
     * @param keyFieldIndex The key field index.
     * @return A Map containing the results
     * @throws DataAccessException if something went wrong executing the query
     * @see Result#intoGroups(int)
     */
    Map<?, Result<R>> fetchGroups(int keyFieldIndex) throws DataAccessException;

    /**
     * Execute the query and return a {@link Map} with one of the result's
     * columns as key and a list of corresponding records as value.
     * <p>
     * Unlike {@link #fetchMap(String)}, this method allows for non-unique keys
     * in the result set.
     * <p>
     * The resulting records are attached to the original {@link Configuration}
     * by default. Use {@link Settings#isAttachRecords()} to override this
     * behaviour.
     *
     * @param keyFieldName The key field name.
     * @return A Map containing the results
     * @throws DataAccessException if something went wrong executing the query
     * @see Result#intoGroups(String)
     */
    Map<?, Result<R>> fetchGroups(String keyFieldName) throws DataAccessException;

    /**
     * Execute the query and return a {@link Map} with one of the result's
     * columns as key and another one of the result's columns as value
     * <p>
     * Unlike {@link #fetchMap(Field, Field)}, this method allows for non-unique
     * keys in the result set.
     *
     * @param <K> The key's generic field type
     * @param <V> The value's generic field type
     * @param key The key field.
     * @param value The value field
     * @return A Map containing the results
     * @throws DataAccessException if something went wrong executing the query
     * @see Result#intoGroups(Field, Field)
     */
    <K, V> Map<K, List<V>> fetchGroups(Field<K> key, Field<V> value) throws DataAccessException;

    /**
     * Execute the query and return a {@link Map} with one of the result's
     * columns as key and another one of the result's columns as value
     * <p>
     * Unlike {@link #fetchMap(int, int)}, this method allows for non-unique
     * keys in the result set.
     *
     * @param keyFieldIndex The key field index.
     * @param valueFieldIndex The value field index.
     * @return A Map containing the results
     * @throws DataAccessException if something went wrong executing the query
     * @see Result#intoGroups(int, int)
     */
    Map<?, List<?>> fetchGroups(int keyFieldIndex, int valueFieldIndex) throws DataAccessException;

    /**
     * Execute the query and return a {@link Map} with one of the result's
     * columns as key and another one of the result's columns as value
     * <p>
     * Unlike {@link #fetchMap(String, String)}, this method allows for
     * non-unique keys in the result set.
     *
     * @param keyFieldName The key field name.
     * @param valueFieldName The value field name.
     * @return A Map containing the results
     * @throws DataAccessException if something went wrong executing the query
     * @see Result#intoGroups(String, String)
     */
    Map<?, List<?>> fetchGroups(String keyFieldName, String valueFieldName) throws DataAccessException;

    /**
     * Execute the query and return a {@link Map} with the result grouped by the
     * given keys.
     * <p>
     * Unlike {@link #fetchMap(Field[])}, this method allows for non-unique keys
     * in the result set.
     *
     * @param keys The keys used for result grouping. If this is
     *            <code>null</code> or an empty array, the resulting map will
     *            contain at most one entry.
     * @return A Map containing grouped results
     * @throws DataAccessException if something went wrong executing the query
     * @see Result#intoGroups(Field[])
     */
    Map<Record, Result<R>> fetchGroups(Field<?>[] keys) throws DataAccessException;

    /**
     * Execute the query and return a {@link Map} with the result grouped by the
     * given keys.
     * <p>
     * Unlike {@link #fetchMap(int[])}, this method allows for non-unique keys
     * in the result set.
     *
     * @param keyFieldIndexes The keys used for result grouping. If this is
     *            <code>null</code> or an empty array, the resulting map will
     *            contain at most one entry.
     * @return A Map containing grouped results
     * @throws DataAccessException if something went wrong executing the query
     * @see Result#intoGroups(int[])
     */
    Map<Record, Result<R>> fetchGroups(int[] keyFieldIndexes) throws DataAccessException;

    /**
     * Execute the query and return a {@link Map} with the result grouped by the
     * given keys.
     * <p>
     * Unlike {@link #fetchMap(String[])}, this method allows for non-unique
     * keys in the result set.
     *
     * @param keyFieldNames The keys used for result grouping. If this is
     *            <code>null</code> or an empty array, the resulting map will
     *            contain at most one entry.
     * @return A Map containing grouped results
     * @throws DataAccessException if something went wrong executing the query
     * @see Result#intoGroups(String[])
     */
    Map<Record, Result<R>> fetchGroups(String[] keyFieldNames) throws DataAccessException;

    /**
     * Execute the query and return a {@link Map} with results grouped by the
     * given keys and mapped into the given entity type.
     * <p>
     * Unlike {@link #fetchMap(Field[], Class)}, this method allows for
     * non-unique keys in the result set.
     *
     * @param keys The keys. If this is <code>null</code> or an empty array, the
     *            resulting map will contain at most one entry.
     * @param type The entity type.
     * @return A Map containing grouped results
     * @throws DataAccessException if something went wrong executing the query
     * @throws MappingException wrapping any reflection or data type conversion
     *             exception that might have occurred while mapping records
     * @see Result#intoGroups(Field[], Class)
     * @see DefaultRecordMapper
     */
    <E> Map<Record, List<E>> fetchGroups(Field<?>[] keys, Class<? extends E> type) throws MappingException;

    /**
     * Execute the query and return a {@link Map} with results grouped by the
     * given keys and mapped into the given entity type.
     * <p>
     * Unlike {@link #fetchMap(int[], Class)}, this method allows for
     * non-unique keys in the result set.
     *
     * @param keyFieldIndexes The keys. If this is <code>null</code> or an empty
     *            array, the resulting map will contain at most one entry.
     * @param type The entity type.
     * @return A Map containing grouped results
     * @throws DataAccessException if something went wrong executing the query
     * @throws MappingException wrapping any reflection or data type conversion
     *             exception that might have occurred while mapping records
     * @see Result#intoGroups(int[], Class)
     * @see DefaultRecordMapper
     */
    <E> Map<Record, List<E>> fetchGroups(int[] keyFieldIndexes, Class<? extends E> type) throws MappingException;

    /**
     * Execute the query and return a {@link Map} with results grouped by the
     * given keys and mapped into the given entity type.
     * <p>
     * Unlike {@link #fetchMap(String[], Class)}, this method allows for
     * non-unique keys in the result set.
     *
     * @param keyFieldNames The keys. If this is <code>null</code> or an empty
     *            array, the resulting map will contain at most one entry.
     * @param type The entity type.
     * @return A Map containing grouped results
     * @throws DataAccessException if something went wrong executing the query
     * @throws MappingException wrapping any reflection or data type conversion
     *             exception that might have occurred while mapping records
     * @see Result#intoGroups(String[], Class)
     * @see DefaultRecordMapper
     */
    <E> Map<Record, List<E>> fetchGroups(String[] keyFieldNames, Class<? extends E> type) throws MappingException;

    /**
     * Execute the query and return a {@link Map} with results grouped by the
     * given keys and mapped by the given mapper.
     * <p>
     * Unlike {@link #fetchMap(Field[], RecordMapper)}, this method allows for
     * non-unique keys in the result set.
     *
     * @param keys The keys. If this is <code>null</code> or an empty array, the
     *            resulting map will contain at most one entry.
     * @param mapper The mapper callback.
     * @return A Map containing grouped results
     * @throws DataAccessException if something went wrong executing the query
     * @throws MappingException wrapping any reflection or data type conversion
     *             exception that might have occurred while mapping records
     * @see Result#intoGroups(Field[], Class)
     * @see DefaultRecordMapper
     */
    <E> Map<Record, List<E>> fetchGroups(Field<?>[] keys, RecordMapper<? super R, E> mapper) throws MappingException;

    /**
     * Execute the query and return a {@link Map} with results grouped by the
     * given keys and mapped by the given mapper.
     * <p>
     * Unlike {@link #fetchMap(int[], RecordMapper)}, this method allows for
     * non-unique keys in the result set.
     *
     * @param keyFieldIndexes The keys. If this is <code>null</code> or an empty
     *            array, the resulting map will contain at most one entry.
     * @param mapper The mapper callback.
     * @return A Map containing grouped results
     * @throws DataAccessException if something went wrong executing the query
     * @throws MappingException wrapping any reflection or data type conversion
     *             exception that might have occurred while mapping records
     * @see Result#intoGroups(int[], Class)
     * @see DefaultRecordMapper
     */
    <E> Map<Record, List<E>> fetchGroups(int[] keyFieldIndexes, RecordMapper<? super R, E> mapper)
        throws MappingException;

    /**
     * Execute the query and return a {@link Map} with results grouped by the
     * given keys and mapped by the given mapper.
     * <p>
     * Unlike {@link #fetchMap(String[], RecordMapper)}, this method allows for
     * non-unique keys in the result set.
     *
     * @param keyFieldNames The keys. If this is <code>null</code> or an empty
     *            array, the resulting map will contain at most one entry.
     * @param mapper The mapper callback.
     * @return A Map containing grouped results
     * @throws DataAccessException if something went wrong executing the query
     * @throws MappingException wrapping any reflection or data type conversion
     *             exception that might have occurred while mapping records
     * @see Result#intoGroups(String[], Class)
     * @see DefaultRecordMapper
     */
    <E> Map<Record, List<E>> fetchGroups(String[] keyFieldNames, RecordMapper<? super R, E> mapper)
        throws MappingException;

    /**
     * Execute the query and return a {@link Map} with the result grouped by the
     * given table.
     * <p>
     * Unlike {@link #fetchMap(Table)}, this method allows for non-unique keys
     * in the result set.
     *
     * @param table The key table. May not be <code>null</code>.
     * @return A Map containing grouped results
     * @throws DataAccessException if something went wrong executing the query
     * @see Result#intoGroups(Field[])
     */
    <S extends Record> Map<S, Result<R>> fetchGroups(Table<S> table) throws DataAccessException;

    /**
     * Execute the query and return a {@link Map} with results grouped by the
     * given table and mapped into the given entity type.
     * <p>
     * Unlike {@link #fetchMap(Table, Class)}, this method allows for non-unique
     * keys in the result set.
     *
     * @param table The key table. May not be <code>null</code>.
     * @param type The entity type.
     * @return A Map containing grouped results
     * @throws DataAccessException if something went wrong executing the query
     * @throws MappingException wrapping any reflection or data type conversion
     *             exception that might have occurred while mapping records
     * @see Result#intoGroups(Table, Class)
     * @see DefaultRecordMapper
     */
    <E, S extends Record> Map<S, List<E>> fetchGroups(Table<S> table, Class<? extends E> type)
        throws DataAccessException, MappingException;

    /**
     * Execute the query and return a {@link Map} with results grouped by the
     * given table and mapped by the given mapper.
     * <p>
     * Unlike {@link #fetchMap(Table, RecordMapper)}, this method allows for
     * non-unique keys in the result set.
     *
     * @param table The key table. May not be <code>null</code>.
     * @param mapper The mapper callback.
     * @return A Map containing grouped results
     * @throws DataAccessException if something went wrong executing the query
     * @throws MappingException wrapping any reflection or data type conversion
     *             exception that might have occurred while mapping records
     * @see Result#intoGroups(Table, Class)
     * @see DefaultRecordMapper
     */
    <E, S extends Record> Map<S, List<E>> fetchGroups(Table<S> table, RecordMapper<? super R, E> mapper)
        throws DataAccessException, MappingException;

    /**
     * Return a {@link Map} with results grouped by the given key and mapped
     * into the given entity type.
     *
     * @param <K> The key's generic field type
     * @param <E> The generic entity type.
     * @param key The key field.
     * @param type The entity type.
     * @throws DataAccessException if something went wrong executing the query
     * @throws MappingException wrapping any reflection or data type conversion
     *             exception that might have occurred while mapping records
     * @see Result#intoGroups(Field, Class)
     * @see DefaultRecordMapper
     */
    <K, E> Map<K, List<E>> fetchGroups(Field<K> key, Class<? extends E> type) throws DataAccessException,
        MappingException;

    /**
     * Return a {@link Map} with results grouped by the given key and mapped
     * into the given entity type.
     *
     * @param keyFieldIndex The key field index.
     * @param type The entity type.
     * @throws DataAccessException if something went wrong executing the query
     * @throws MappingException wrapping any reflection or data type conversion
     *             exception that might have occurred while mapping records
     * @see Result#intoGroups(int, Class)
     * @see DefaultRecordMapper
     */
    <E> Map<?, List<E>> fetchGroups(int keyFieldIndex, Class<? extends E> type) throws DataAccessException,
        MappingException;

    /**
     * Return a {@link Map} with results grouped by the given key and mapped
     * into the given entity type.
     *
     * @param keyFieldName The key field name.
     * @param type The entity type.
     * @throws DataAccessException if something went wrong executing the query
     * @throws MappingException wrapping any reflection or data type conversion
     *             exception that might have occurred while mapping records
     * @see Result#intoGroups(String, Class)
     * @see DefaultRecordMapper
     */
    <E> Map<?, List<E>> fetchGroups(String keyFieldName, Class<? extends E> type) throws DataAccessException,
        MappingException;

    /**
     * Return a {@link Map} with results grouped by the given key and mapped by
     * the given mapper.
     *
     * @param <K> The key's generic field type
     * @param <E> The generic entity type.
     * @param key The key field.
     * @param mapper The mapper callback.
     * @throws DataAccessException if something went wrong executing the query
     * @throws MappingException wrapping any reflection or data type conversion
     *             exception that might have occurred while mapping records
     * @see Result#intoGroups(Field, Class)
     * @see DefaultRecordMapper
     */
    <K, E> Map<K, List<E>> fetchGroups(Field<K> key, RecordMapper<? super R, E> mapper) throws DataAccessException,
        MappingException;

    /**
     * Return a {@link Map} with results grouped by the given key and mapped by
     * the given mapper.
     *
     * @param keyFieldIndex The key field index.
     * @param mapper The mapper callback.
     * @throws DataAccessException if something went wrong executing the query
     * @throws MappingException wrapping any reflection or data type conversion
     *             exception that might have occurred while mapping records
     * @see Result#intoGroups(int, Class)
     * @see DefaultRecordMapper
     */
    <E> Map<?, List<E>> fetchGroups(int keyFieldIndex, RecordMapper<? super R, E> mapper) throws DataAccessException,
        MappingException;

    /**
     * Return a {@link Map} with results grouped by the given key and mapped by
     * the given mapper.
     *
     * @param keyFieldName The key field name.
     * @param mapper The mapper callback.
     * @throws DataAccessException if something went wrong executing the query
     * @throws MappingException wrapping any reflection or data type conversion
     *             exception that might have occurred while mapping records
     * @see Result#intoGroups(String, Class)
     * @see DefaultRecordMapper
     */
    <E> Map<?, List<E>> fetchGroups(String keyFieldName, RecordMapper<? super R, E> mapper) throws DataAccessException,
        MappingException;

    /**
     * Execute the query and return the generated result as an Object matrix.
     * <p>
     * You can access data like this
     * <code><pre>query.fetchArray()[recordIndex][fieldIndex]</pre></code>
     *
     * @return The result.
     * @throws DataAccessException if something went wrong executing the query
     * @see Result#intoArrays()
     */
    Object[][] fetchArrays() throws DataAccessException;

    /**
     * Execute the query and return the generated result as an array of records.
     * <p>
     *
     * @return The result.
     * @throws DataAccessException if something went wrong executing the query
     * @see Result#toArray(Object[])
     */
    R[] fetchArray() throws DataAccessException;

    /**
     * Execute the query and return all values for a field index from the
     * generated result.
     * <p>
     * You can access data like this
     * <code><pre>query.fetchArray(fieldIndex)[recordIndex]</pre></code>
     *
     * @return The resulting values. This may be an array type more concrete
     *         than <code>Object[]</code>, depending on whether jOOQ has any
     *         knowledge about <code>fieldIndex</code>'s actual type.
     * @throws DataAccessException if something went wrong executing the query
     * @see Result#intoArray(int)
     */
    Object[] fetchArray(int fieldIndex) throws DataAccessException;

    /**
     * Execute the query and return all values for a field index from the
     * generated result.
     * <p>
     * You can access data like this
     * <code><pre>query.fetchArray(fieldIndex)[recordIndex]</pre></code>
     *
     * @return The resulting values.
     * @throws DataAccessException if something went wrong executing the query
     * @see Result#intoArray(int, Class)
     */
    <T> T[] fetchArray(int fieldIndex, Class<? extends T> type) throws DataAccessException;

    /**
     * Execute the query and return all values for a field index from the
     * generated result.
     * <p>
     * You can access data like this
     * <code><pre>query.fetchArray(fieldIndex)[recordIndex]</pre></code>
     *
     * @return The resulting values.
     * @throws DataAccessException if something went wrong executing the query
     * @see Result#intoArray(int, Converter)
     */
    <U> U[] fetchArray(int fieldIndex, Converter<?, U> converter) throws DataAccessException;

    /**
     * Execute the query and return all values for a field name from the
     * generated result.
     * <p>
     * You can access data like this
     * <code><pre>query.fetchArray(fieldName)[recordIndex]</pre></code>
     *
     * @return The resulting values. This may be an array type more concrete
     *         than <code>Object[]</code>, depending on whether jOOQ has any
     *         knowledge about <code>fieldName</code>'s actual type.
     * @throws DataAccessException if something went wrong executing the query
     * @see Result#intoArray(String)
     */
    Object[] fetchArray(String fieldName) throws DataAccessException;

    /**
     * Execute the query and return all values for a field name from the
     * generated result.
     * <p>
     * You can access data like this
     * <code><pre>query.fetchArray(fieldName)[recordIndex]</pre></code>
     *
     * @return The resulting values.
     * @throws DataAccessException if something went wrong executing the query
     * @see Result#intoArray(String, Converter)
     */
    <T> T[] fetchArray(String fieldName, Class<? extends T> type) throws DataAccessException;

    /**
     * Execute the query and return all values for a field name from the
     * generated result.
     * <p>
     * You can access data like this
     * <code><pre>query.fetchArray(fieldName)[recordIndex]</pre></code>
     *
     * @return The resulting values.
     * @throws DataAccessException if something went wrong executing the query
     * @see Result#intoArray(String, Class)
     */
    <U> U[] fetchArray(String fieldName, Converter<?, U> converter) throws DataAccessException;

    /**
     * Execute the query and return all values for a field from the generated
     * result.
     * <p>
     * You can access data like this
     * <code><pre>query.fetchArray(field)[recordIndex]</pre></code>
     *
     * @return The resulting values.
     * @throws DataAccessException if something went wrong executing the query
     * @see Result#intoArray(Field)
     */
    <T> T[] fetchArray(Field<T> field) throws DataAccessException;

    /**
     * Execute the query and return all values for a field from the generated
     * result.
     * <p>
     * You can access data like this
     * <code><pre>query.fetchArray(field)[recordIndex]</pre></code>
     *
     * @return The resulting values.
     * @throws DataAccessException if something went wrong executing the query
     * @see Result#intoArray(Field, Class)
     */
    <T> T[] fetchArray(Field<?> field, Class<? extends T> type) throws DataAccessException;

    /**
     * Execute the query and return all values for a field from the generated
     * result.
     * <p>
     * You can access data like this
     * <code><pre>query.fetchArray(field)[recordIndex]</pre></code>
     *
     * @return The resulting values.
     * @throws DataAccessException if something went wrong executing the query
     * @see Result#intoArray(Field, Converter)
     */
    <T, U> U[] fetchArray(Field<T> field, Converter<? super T, U> converter) throws DataAccessException;

    /**
     * Execute the query and return all values for a field index from the
     * generated result.
     *
     * @return The resulting values.
     * @throws DataAccessException if something went wrong executing the query
     * @see Result#intoArray(int)
     */
    Set<?> fetchSet(int fieldIndex) throws DataAccessException;

    /**
     * Execute the query and return all values for a field index from the
     * generated result.
     *
     * @return The resulting values.
     * @throws DataAccessException if something went wrong executing the query
     * @see Result#intoArray(int, Class)
     */
    <T> Set<T> fetchSet(int fieldIndex, Class<? extends T> type) throws DataAccessException;

    /**
     * Execute the query and return all values for a field index from the
     * generated result.
     *
     * @return The resulting values.
     * @throws DataAccessException if something went wrong executing the query
     * @see Result#intoArray(int, Converter)
     */
    <U> Set<U> fetchSet(int fieldIndex, Converter<?, U> converter) throws DataAccessException;

    /**
     * Execute the query and return all values for a field name from the
     * generated result.
     *
     * @return The resulting values.
     * @throws DataAccessException if something went wrong executing the query
     * @see Result#intoArray(String)
     */
    Set<?> fetchSet(String fieldName) throws DataAccessException;

    /**
     * Execute the query and return all values for a field name from the
     * generated result.
     *
     * @return The resulting values.
     * @throws DataAccessException if something went wrong executing the query
     * @see Result#intoArray(String, Converter)
     */
    <T> Set<T> fetchSet(String fieldName, Class<? extends T> type) throws DataAccessException;

    /**
     * Execute the query and return all values for a field name from the
     * generated result.
     *
     * @return The resulting values.
     * @throws DataAccessException if something went wrong executing the query
     * @see Result#intoArray(String, Class)
     */
    <U> Set<U> fetchSet(String fieldName, Converter<?, U> converter) throws DataAccessException;

    /**
     * Execute the query and return all values for a field from the generated
     * result.
     *
     * @return The resulting values.
     * @throws DataAccessException if something went wrong executing the query
     * @see Result#intoArray(Field)
     */
    <T> Set<T> fetchSet(Field<T> field) throws DataAccessException;

    /**
     * Execute the query and return all values for a field from the generated
     * result.
     *
     * @return The resulting values.
     * @throws DataAccessException if something went wrong executing the query
     * @see Result#intoArray(Field, Class)
     */
    <T> Set<T> fetchSet(Field<?> field, Class<? extends T> type) throws DataAccessException;

    /**
     * Execute the query and return all values for a field from the generated
     * result.
     *
     * @return The resulting values.
     * @throws DataAccessException if something went wrong executing the query
     * @see Result#intoArray(Field, Converter)
     */
    <T, U> Set<U> fetchSet(Field<T> field, Converter<? super T, U> converter) throws DataAccessException;

    /**
     * Map resulting records onto a custom type.
     * <p>
     * This is the same as calling <code>fetch().into(type)</code>. See
     * {@link Record#into(Class)} for more details
     *
     * @param <E> The generic entity type.
     * @param type The entity type.
     * @see Record#into(Class)
     * @see Result#into(Class)
     * @throws DataAccessException if something went wrong executing the query
     * @throws MappingException wrapping any reflection or data type conversion
     *             exception that might have occurred while mapping records
     * @see DefaultRecordMapper
     */
    <E> List<E> fetchInto(Class<? extends E> type) throws DataAccessException, MappingException;

    /**
     * Map resulting records onto a custom record.
     * <p>
     * This is the same as calling <code>fetch().into(table)</code>. See
     * {@link Record#into(Table)} for more details
     * <p>
     * The result and its contained records are attached to the original
     * {@link Configuration} by default. Use {@link Settings#isAttachRecords()}
     * to override this behaviour.
     *
     * @param <Z> The generic table record type.
     * @param table The table type.
     * @see Record#into(Table)
     * @see Result#into(Table)
     * @throws DataAccessException if something went wrong executing the query
     */
    <Z extends Record> Result<Z> fetchInto(Table<Z> table) throws DataAccessException;

    /**
     * Fetch results into a custom handler callback.
     * <p>
     * The resulting records are attached to the original {@link Configuration}
     * by default. Use {@link Settings#isAttachRecords()} to override this
     * behaviour.
     *
     * @param handler The handler callback
     * @return Convenience result, returning the parameter handler itself
     * @throws DataAccessException if something went wrong executing the query
     */
    <H extends RecordHandler<? super R>> H fetchInto(H handler) throws DataAccessException;

    /**
     * Fetch results into a custom mapper callback.
     *
     * @param mapper The mapper callback
     * @return The custom mapped records
     * @throws DataAccessException if something went wrong executing the query
     */
    <E> List<E> fetch(RecordMapper<? super R, E> mapper) throws DataAccessException;

    /**
     * Fetch results asynchronously.
     * <p>
     * This method wraps fetching of records in a
     * {@link java.util.concurrent.Future}, such that you can access the actual
     * records at a future instant. This is especially useful when
     * <ul>
     * <li>You want to load heavy data in the background, for instance when the
     * user logs in and accesses a pre-calculated dashboard screen, before they
     * access the heavy data.</li>
     * <li>You want to parallelise several independent OLAP queries before
     * merging all data into a single report</li>
     * <li>...</li>
     * </ul>
     * <p>
     * This will internally create a "single thread executor", that is shut down
     * at the end of the {@link FutureResult}'s lifecycle. Use
     * {@link #fetchLater(ExecutorService)} instead, if you want control over
     * your executing threads.
     * <p>
     * The result and its contained records are attached to the original
     * {@link Configuration} by default. Use {@link Settings#isAttachRecords()}
     * to override this behaviour.
     *
     * @return A future result
     * @throws DataAccessException if something went wrong executing the query
     * @deprecated - 3.2.0 - [#2581] - This method will be removed in jOOQ 4.0
     */
    @Deprecated
    FutureResult<R> fetchLater() throws DataAccessException;

    /**
     * Fetch results asynchronously.
     * <p>
     * This method wraps fetching of records in a
     * {@link java.util.concurrent.Future}, such that you can access the actual
     * records at a future instant. This is especially useful when
     * <ul>
     * <li>You want to load heavy data in the background, for instance when the
     * user logs in and accesses a pre-calculated dashboard screen, before they
     * access the heavy data.</li>
     * <li>You want to parallelise several independent OLAP queries before
     * merging all data into a single report</li>
     * <li>...</li>
     * </ul>
     * <p>
     * Use this method rather than {@link #fetchLater()}, in order to keep
     * control over thread lifecycles, if you manage threads in a J2EE container
     * or with Spring, for instance.
     * <p>
     * The result and its contained records are attached to the original
     * {@link Configuration} by default. Use {@link Settings#isAttachRecords()}
     * to override this behaviour.
     *
     * @param executor A custom executor
     * @return A future result
     * @throws DataAccessException if something went wrong executing the query
     * @deprecated - 3.2.0 - [#2581] - This method will be removed in jOOQ 4.0
     */
    @Deprecated
    FutureResult<R> fetchLater(ExecutorService executor) throws DataAccessException;

    /**
     * The record type produced by this query.
     */
    Class<? extends R> getRecordType();

    /**
     * {@inheritDoc}
     */
    @Override
    ResultQuery<R> bind(String param, Object value) throws IllegalArgumentException, DataTypeException;

    /**
     * {@inheritDoc}
     */
    @Override
    ResultQuery<R> bind(int index, Object value) throws IllegalArgumentException, DataTypeException;

    // ------------------------------------------------------------------------
    // JDBC methods
    // ------------------------------------------------------------------------

    /**
     * {@inheritDoc}
     */
    @Override
    ResultQuery<R> queryTimeout(int timeout);

    /**
     * {@inheritDoc}
     */
    @Override
    ResultQuery<R> keepStatement(boolean keepStatement);

    /**
     * Specify the maximum number of rows returned by the underlying
     * {@link Statement}.
     * <p>
     * This is not the same as setting a <code>LIMIT .. OFFSET</code> clause
     * onto the statement, where the result set is restricted within the
     * database.
     *
     * @see Statement#setMaxRows(int)
     */
    ResultQuery<R> maxRows(int rows);

    /**
     * Specify the fetch size of the underlying {@link Statement}.
     * <p>
     * Regardless of this setting, {@link #fetchLazy()} is the only way in jOOQ
     * not to fetch all data in memory. However, you may influence how your JDBC
     * driver interacts with your database through specifying a fetch size.
     * <p>
     * Note that some databases (in particular PostgreSQL) do not like fetch
     * sizes being combined with
     * <code>{@link Connection#getAutoCommit()} == true</code>. For more
     * information, see <a href=
     * "http://jdbc.postgresql.org/documentation/head/query.html#query-with-cursor"
     * >this page here</a>
     *
     * @see Statement#setFetchSize(int)
     */
    ResultQuery<R> fetchSize(int rows);

    /**
     * Specify the <code>ResultSet</code> concurrency of <code>ResultSet</code>
     * objects created by jOOQ.
     * <p>
     * This will affect the way you may perceive <code>ResultSet</code> objects
     * obtained from any of these methods:
     * <ul>
     * <li>{@link ResultQuery#fetchResultSet()}</li>
     * <li>{@link Cursor#resultSet()}</li>
     * <li>{@link ExecuteContext#resultSet()}</li>
     * </ul>
     *
     * @see Statement#getResultSetConcurrency()
     */
    ResultQuery<R> resultSetConcurrency(int resultSetConcurrency);

    /**
     * Specify the <code>ResultSet</code> type of <code>ResultSet</code>
     * objects created by jOOQ.
     * <p>
     * This will affect the way you may perceive <code>ResultSet</code> objects
     * obtained from any of these methods:
     * <ul>
     * <li>{@link ResultQuery#fetchResultSet()}</li>
     * <li>{@link Cursor#resultSet()}</li>
     * <li>{@link ExecuteContext#resultSet()}</li>
     * </ul>
     *
     * @see Statement#getResultSetType()
     */
    ResultQuery<R> resultSetType(int resultSetType);

    /**
     * Specify the <code>ResultSet</code> holdability of <code>ResultSet</code>
     * objects created by jOOQ.
     * <p>
     * This will affect the way you may perceive <code>ResultSet</code> objects
     * obtained from any of these methods:
     * <ul>
     * <li>{@link ResultQuery#fetchResultSet()}</li>
     * <li>{@link Cursor#resultSet()}</li>
     * <li>{@link ExecuteContext#resultSet()}</li>
     * </ul>
     *
     * @see Statement#getResultSetHoldability()
     */
    ResultQuery<R> resultSetHoldability(int resultSetHoldability);

    /**
     * Specify a set of fields whose values should be interned.
     * <p>
     * Unlike {@link Result}'s <code>intern()</code> methods, this already
     * interns values right after fetching them from a JDBC result set. See
     * {@link Result#intern(int...)} for more details.
     *
     * @param fields The fields whose values should be interned
     * @return The same result query
     * @see Result#intern(Field...)
     * @see String#intern()
     */
    ResultQuery<R> intern(Field<?>... fields);

    /**
     * Specify a set of field indexes whose values should be interned.
     * <p>
     * Unlike {@link Result}'s <code>intern()</code> methods, this already
     * interns values right after fetching them from a JDBC result set. See
     * {@link Result#intern(int...)} for more details.
     *
     * @param fieldIndexes The field indexes whose values should be interned
     * @return The same result query
     * @see Result#intern(int...)
     * @see String#intern()
     */
    ResultQuery<R> intern(int... fieldIndexes);

    /**
     * Specify a set of field names whose values should be interned.
     * <p>
     * Unlike {@link Result}'s <code>intern()</code> methods, this already
     * interns values right after fetching them from a JDBC result set. See
     * {@link Result#intern(int...)} for more details.
     *
     * @param fieldNames The field names whose values should be interned
     * @return The same result query
     * @see Result#intern(String...)
     * @see String#intern()
     */
    ResultQuery<R> intern(String... fieldNames);

}
