/**
 * Copyright (c) 2009-2015, Data Geekery GmbH (http://www.datageekery.com)
 * All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Other licenses:
 * -----------------------------------------------------------------------------
 * Commercial licenses for this work are available. These replace the above
 * ASL 2.0 and offer limited warranties, support, maintenance, and commercial
 * database integrations.
 *
 * For more information, please visit: http://www.jooq.org/licenses
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */
package org.jooq;

import java.util.Collection;

import org.jooq.impl.DSL;

/**
 * This type is used for the {@link Delete}'s DSL API.
 * <p>
 * Example: <code><pre>
 * DSLContext create = DSL.using(configuration);
 *
 * create.delete(table)
 *       .where(field1.greaterThan(100))
 *       .execute();
 * </pre></code>
 *
 * @author Lukas Eder
 */
public interface DeleteWhereStep<R extends Record> extends DeleteFinalStep<R> {

    /**
     * Add conditions to the query, connecting them with each other with
     * {@link Operator#AND}.
     */
    @Support
    DeleteConditionStep<R> where(Condition... conditions);

    /**
     * Add conditions to the query, connecting them with each other with
     * {@link Operator#AND}.
     */
    @Support
    DeleteConditionStep<R> where(Collection<? extends Condition> conditions);

    /**
     * Add conditions to the query.
     */
    @Support
    DeleteConditionStep<R> where(Field<Boolean> condition);

    /**
     * Add conditions to the query.
     * <p>
     * <b>NOTE</b>: When inserting plain SQL into jOOQ objects, you must
     * guarantee syntax integrity. You may also create the possibility of
     * malicious SQL injection. Be sure to properly use bind variables and/or
     * escape literals when concatenated into SQL clauses!
     *
     * @see DSL#condition(String)
     */
    @Support
    @PlainSQL
    DeleteConditionStep<R> where(String sql);

    /**
     * Add conditions to the query.
     * <p>
     * <b>NOTE</b>: When inserting plain SQL into jOOQ objects, you must
     * guarantee syntax integrity. You may also create the possibility of
     * malicious SQL injection. Be sure to properly use bind variables and/or
     * escape literals when concatenated into SQL clauses!
     *
     * @see DSL#condition(String, Object...)
     */
    @Support
    @PlainSQL
    DeleteConditionStep<R> where(String sql, Object... bindings);

    /**
     * Add conditions to the query.
     * <p>
     * <b>NOTE</b>: When inserting plain SQL into jOOQ objects, you must
     * guarantee syntax integrity. You may also create the possibility of
     * malicious SQL injection. Be sure to properly use bind variables and/or
     * escape literals when concatenated into SQL clauses!
     *
     * @see DSL#condition(String, QueryPart...)
     */
    @Support
    @PlainSQL
    DeleteConditionStep<R> where(String sql, QueryPart... parts);

    /**
     * Add an EXISTS clause to the query.
     */
    @Support
    DeleteConditionStep<R> whereExists(Select<?> select);

    /**
     * Add a NOT EXISTS clause to the query.
     */
    @Support
    DeleteConditionStep<R> whereNotExists(Select<?> select);
}
