/*
 * Copyright (c) 2015-present Alipay.com, https://www.alipay.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package cn.com.antcloud.api.common;

import java.lang.reflect.Type;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

public class BaseClientRequest {

    private final Map<String, String> parameters;

    /**
     * Constructor.
     */
    public BaseClientRequest() {
        this.parameters = new HashMap<String, String>();
    }

    /**
     * Put parameter.
     *
     * @param key   the key
     * @param value the value
     */
    public void putParameter(String key, String value) {
        SDKUtils.checkNotNull(key);

        parameters.put(key, value);
    }

    /**
     * Remove parameter.
     *
     * @param key the key
     */
    public void removeParameter(String key) {
        SDKUtils.checkNotNull(key);
        parameters.remove(key);
    }

    /**
     * Gets parameter.
     *
     * @param key the key
     * @return the parameter
     */
    public String getParameter(String key) {
        return parameters.get(key);
    }

    /**
     * Put parameters.
     *
     * @param params the params
     */
    public void putParameters(Map<String, String> params) {
        SDKUtils.checkNotNull(params);
        if (params.containsKey(null)) {
            throw new IllegalArgumentException("Null key is not allowed");
        }

        parameters.putAll(params);
    }

    /**
     * Put parameters from object.
     *
     * @param o the o
     */
    public void putParametersFromObject(Object o) {
        Map<String, String> params = GwKeyValues.toMap(o);
        putParameters(params);
    }

    /**
     * Gets parameters as object.
     *
     * @param <T>   the type parameter
     * @param clazz the clazz
     * @return the parameters as object
     */
    public <T> T getParametersAsObject(Class<T> clazz) {
        return GwKeyValues.toObject(parameters, clazz);
    }

    /**
     * Gets parameters as object.
     *
     * @param <T>  the type parameter
     * @param type the type
     * @return the parameters as object
     */
    public <T> T getParametersAsObject(Type type) {
        return GwKeyValues.toObject(parameters, type);
    }

    /**
     * Gets parameters.
     *
     * @return the parameters
     */
    public Map<String, String> getParameters() {
        return Collections.unmodifiableMap(parameters);
    }

    /**
     * Gets method.
     *
     * @return the method
     */
    public String getMethod() {
        return getParameter(SDKConstants.ParamKeys.METHOD);
    }

    /**
     * Sets method.
     *
     * @param method the method
     */
    public void setMethod(String method) {
        putParameter(SDKConstants.ParamKeys.METHOD, method);
    }

    /**
     * Gets version.
     *
     * @return the version
     */
    public String getVersion() {
        return getParameter(SDKConstants.ParamKeys.VERSION);
    }

    /**
     * Sets version.
     *
     * @param version the version
     */
    public void setVersion(String version) {
        putParameter(SDKConstants.ParamKeys.VERSION, version);
    }

    /**
     * Gets req msg id.
     *
     * @return the req msg id
     */
    public String getReqMsgId() {
        return getParameter(SDKConstants.ParamKeys.REQ_MSG_ID);
    }

    /**
     * Sets req msg id.
     *
     * @param reqMsgId the req msg id
     */
    public void setReqMsgId(String reqMsgId) {
        putParameter(SDKConstants.ParamKeys.REQ_MSG_ID, reqMsgId);
    }

    /**
     * Equals boolean.
     *
     * @param o the o
     * @return the boolean
     */
    @Override
    public boolean equals(Object o) {
        if (this == o)
            return true;
        if (o == null || getClass() != o.getClass())
            return false;

        BaseClientRequest that = (BaseClientRequest) o;

        return parameters != null ? parameters.equals(that.parameters) : that.parameters == null;
    }

    /**
     * Hash code int.
     *
     * @return the int
     */
    @Override
    public int hashCode() {
        return parameters != null ? parameters.hashCode() : 0;
    }
}
