package com.jz.website.cache.ram;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.PostConstruct;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.jz.common.utils.collection.ArrayMapTools;
import com.jz.jooq.website.tables.pojos.NewsTopic;
import com.jz.website.service.NewsTopicService;

/**
 * @Title NewsTopicRAMCache
 * @Package com.jz.website.cache.ram
 * @author tangjunfeng
 * @date 2018年11月15日 下午5:49:12
 * @version V1.0
 */
@Lazy
@Component
public class NewsTopicRAMCache {
	
	@Autowired
	private NewsTopicService newsTopicService;
	// id对应实体
	private static Map<Integer, NewsTopic> id2Topic = Maps.newHashMap();
	// 父级对应子集
	private static Map<Integer, List<Integer>> parent2Children = Maps.newLinkedHashMap();
	// root节点
	private static Set<Integer> roots = Sets.newHashSet();
	// 分部拥有
	private static Set<Integer> branchPossess = Sets.newHashSet();
	// 总部拥有(目前需求是总部拥有所有的)
	private Set<Integer> headOfficePossess = Sets.newHashSet();
	
	// 资讯中心、学术国际
	public static final Integer newsTopic = 10001, studyTopic = 10000;
	
	@PostConstruct
	public void loadTopic2Cache() {
		List<NewsTopic> topics = newsTopicService.findAll();
		if (ArrayMapTools.isEmpty(topics))
			return;
		topics.forEach(it -> {
			// id -> instance
			id2Topic.put(it.getId(), it);
			// parent -> children
			if (null == it.getParent()) {
				roots.add(it.getId());
			} else {
				if (!parent2Children.containsKey(it.getParent()))
					parent2Children.put(it.getParent(), Lists.newArrayList());
				parent2Children.get(it.getParent()).add(it.getId());
			}
			if (it.getHo() == 0) { // 都有
				branchPossess.add(it.getId());
				headOfficePossess.add(it.getId());
			} else if (it.getHo() == 1) { // 总部独享
				headOfficePossess.add(it.getId());
			}

		});
	}
	
	/** 是否root节点 */
	public static boolean isRoot(Integer tid) {
		return null != tid && roots.contains(tid);
	}
	
	/** 当前节点是否有效 */
	public static boolean isValid(Integer tid) {
		return null != tid && id2Topic.containsKey(tid);
	}
	
	/** 获取当前节点下所有的子节点 */
	public static List<NewsTopic> getChildren(Integer tid) {
		if (null == tid || !parent2Children.containsKey(tid))
			return null;
		return parent2Children.get(tid).stream().map(id2Topic::get).collect(Collectors.toList());
	}
	
	/** 是否分校拥有 */
	public static boolean isBranchPossess(Integer tid) {
		return null != tid && branchPossess.contains(tid);
	}
	
	/** 获取当前节点信息 */
	public static NewsTopic getTopic(Integer tid) {
		if (null == tid || !id2Topic.containsKey(tid))
			return null;
		return id2Topic.get(tid);
	}
}
