package com.jz.qcloud.tools;

import java.io.File;
import java.io.FileNotFoundException;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Lists;
import com.jz.qcloud.beans.QcloudConfig;
import com.qcloud.cos.COSClient;
import com.qcloud.cos.ClientConfig;
import com.qcloud.cos.auth.BasicCOSCredentials;
import com.qcloud.cos.auth.COSCredentials;
import com.qcloud.cos.http.HttpProtocol;
import com.qcloud.cos.model.COSObjectSummary;
import com.qcloud.cos.model.GetObjectRequest;
import com.qcloud.cos.model.ListObjectsRequest;
import com.qcloud.cos.model.ObjectListing;
import com.qcloud.cos.model.PutObjectRequest;
import com.qcloud.cos.region.Region;

/**
 * @Title 针对于腾讯云对象存储进行相关文件操作
 * @author shaolianjie
 */
public class CosTools {

	private static final Logger logger = LoggerFactory.getLogger(CosTools.class);

	private QcloudConfig qcloudConfig;

	private CosTools() {
	}

	private CosTools(QcloudConfig qcloudConfig) {
		super();
		this.qcloudConfig = qcloudConfig;
	}

	public static CosTools getInstance(QcloudConfig qcloudConfig) {
		return new CosTools(qcloudConfig);
	}

	private COSClient createCOSClient() {
		// 1 初始化用户身份信息（secretId, secretKey）。
		// SECRETID和SECRETKEY请登录访问管理控制台 https://console.cloud.tencent.com/cam/capi
		// 进行查看和管理
		String secretId = qcloudConfig.getAccessKeyId();
		String secretKey = qcloudConfig.getAccessKeySecret();
		COSCredentials cred = new BasicCOSCredentials(secretId, secretKey);
		// 2 设置 bucket 的地域, COS 地域的简称请参照
		// https://cloud.tencent.com/document/product/436/6224
		// clientConfig 中包含了设置 region, https(默认 http), 超时, 代理等 set 方法, 使用可参见源码或者常见问题
		// Java SDK 部分。
		Region region = new Region(qcloudConfig.getRegion());
		ClientConfig clientConfig = new ClientConfig(region);
		// 这里建议设置使用 https 协议
		// 从 5.6.54 版本开始，默认使用了 https
		clientConfig.setHttpProtocol(HttpProtocol.https);
		// 3 生成 cos 客户端。
		return new COSClient(cred, clientConfig);
	}

	/** 上传本地文件 */
	public void upload(String localFilePath, String cosFilePath) throws FileNotFoundException {
		logger.info("upload file : {} => {}", localFilePath, cosFilePath);
		COSClient cosClient = createCOSClient();
		// 指定要上传的文件
		File localFile = new File(localFilePath);
		// 指定文件将要存放的存储桶
		String bucketName = qcloudConfig.getBucketName();
		// 指定文件上传到 COS 上的路径，即对象键。例如对象键为folder/picture.jpg，则表示将文件 picture.jpg 上传到 folder
		// 路径下
		PutObjectRequest putObjectRequest = new PutObjectRequest(bucketName, cosFilePath, localFile);
		cosClient.putObject(putObjectRequest);
		cosClient.shutdown();
	}

	/** cos上通过文件夹获取文件列表 */
	public List<String> listFiles(String dirPath) {
		COSClient cosClient = createCOSClient();
		ListObjectsRequest listObjectsRequest = new ListObjectsRequest();
		listObjectsRequest.setBucketName(qcloudConfig.getBucketName());
		if (!dirPath.endsWith("/")) {
			dirPath += "/";
		}
		listObjectsRequest.setPrefix(dirPath);
		// deliter表示分隔符, 设置为/表示列出当前目录下的object, 设置为空表示列出所有的object
		listObjectsRequest.setDelimiter("/");
		// 设置最大遍历出多少个对象, 一次listobject最大支持1000
		listObjectsRequest.setMaxKeys(200);
		ObjectListing objectListing = cosClient.listObjects(listObjectsRequest);
		// object summary表示所有列出的object列表
		List<COSObjectSummary> cosObjectSummaries = objectListing.getObjectSummaries();
		List<String> list = Lists.newArrayList();
		for (COSObjectSummary cosObjectSummary : cosObjectSummaries) {
			// 文件的路径key
			String key = cosObjectSummary.getKey();
			// 文件的etag
			// String etag = cosObjectSummary.getETag();
			// 文件的长度
			// long fileSize = cosObjectSummary.getSize();
			// 文件的存储类型
			// String storageClasses = cosObjectSummary.getStorageClass();
			list.add(key);
		}
		cosClient.shutdown();
		return list;
	}

	public void downloadFile(String filePath, String localPath) throws Throwable {
		COSClient cosClient = createCOSClient();
		File downFile = new File(localPath);
		GetObjectRequest getObjectRequest = new GetObjectRequest(qcloudConfig.getBucketName(), filePath);
		cosClient.getObject(getObjectRequest, downFile);
		// 关闭client
		cosClient.shutdown();
	}

	/** oss上的文件删除 */
	public void delete(String filePath) throws FileNotFoundException {
		logger.info("delete file : {} => {}", filePath);
		COSClient cosClient = createCOSClient();
		cosClient.deleteObject(qcloudConfig.getBucketName(), filePath);
		// 关闭client
		cosClient.shutdown();
	}

//	public static void main(String[] args) throws Throwable {
//		String bucket = "image-1301722750";
//		QcloudConfig config = QcloudConfig.of("AKIDkF4fYaBNdleMRGu42mHWAbnhKgofCVW9",
//				"UhceQu60enfxzkEhw83g271kHYaLo0uz", bucket);
//		CosTools.getInstance(config).upload("C:/Users/邵连杰/Desktop/test.jpg", "test/test2.jpg");
//		for(String item:CosTools.getInstance(config).listFiles("test")) {
//			System.out.println(item);
//		}
//		CosTools.getInstance(config).downloadFile("test/test1.jpg", "C:/Users/邵连杰/Desktop/test1.jpg");
//		CosTools.getInstance(config).delete("test/test2.jpg");
//	}
}