package com.jz.jar.oa.service;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.jz.common.utils.collection.ArrayMapTools;
import com.jz.jar.oa.repository.OADepartmentRepository;
import com.jz.jar.oa.repository.OAPositionRepository;
import com.jz.jar.oa.wrapper.DepartmentWrapper;
import com.jz.jar.oa.wrapper.OAUserWrapper;
import com.jz.jar.oa.wrapper.PositionWrapper;
import com.jz.jooq.oa.tables.pojos.Department;
import com.jz.jooq.oa.tables.pojos.Position;
import com.jz.jooq.oa.tables.pojos.User;

@Lazy
@Service
public class OADepartmentService {

	@Autowired
	private OADepartmentRepository departmentRepository;
	@Autowired
	private OAPositionRepository positionRepository;
	@Autowired
	private OAUserService oaUserService;

	public Collection<DepartmentWrapper> getAllDepartmentWithPosition() {
		List<Position> positions = positionRepository.getAllPosition();
		Map<String, List<PositionWrapper>> parent2PosMap = Maps.newHashMap();
		positions.forEach(p -> {
			if (parent2PosMap.containsKey(p.getDeptId())) {
				parent2PosMap.get(p.getDeptId()).add(PositionWrapper.of(p));
			} else {
				parent2PosMap.put(p.getDeptId(), Lists.newArrayList(PositionWrapper.of(p)));
			}
		});
		List<Department> departments = departmentRepository.getAllDepartment(); // 这里是level正序排好的
		Map<String, DepartmentWrapper> lv1DeparmentMap = Maps.newLinkedHashMap(); // 保证有序
		Map<String, DepartmentWrapper> lv2DeparmentMap = Maps.newLinkedHashMap(); // lv2deptId:lv2dept
		Set<String> uids = Sets.newHashSet();
		Set<String> deptIds = Sets.newHashSet();
		Map<String, User> leadersMap = Maps.newHashMap();
		Map<String, Integer> deptStaff = Maps.newHashMap();
		if (ArrayMapTools.isNotEmpty(departments)) {
			departments.forEach(it -> {
				uids.add(it.getLeader());
				deptIds.add(it.getId());
			});
		}
		leadersMap = oaUserService.mutiGetUserMap(uids);
		deptStaff = oaUserService.getStaffNumByDeptIds(deptIds);
		for (Department d : departments) {
			DepartmentWrapper wrapper = DepartmentWrapper.of(d);
			if (leadersMap.containsKey(d.getLeader())) {
				OAUserWrapper leader = OAUserWrapper.ofSimple(leadersMap.get(d.getLeader()));
				wrapper.setLeader(leader);
			}
			if (deptStaff.containsKey(d.getId())) {
				wrapper.setStaffNum(deptStaff.get(d.getId()));
			} else {
				wrapper.setStaffNum(0);
			}
			wrapper.setPositions(parent2PosMap.get(d.getId()));
			if (d.getLevel() == 2) {
				lv2DeparmentMap.put(d.getId(), wrapper);
			} else if (d.getLevel() == 3 && lv2DeparmentMap.containsKey(d.getParentId())) { // lv2
				lv2DeparmentMap.get(d.getParentId()).addChild(wrapper);
				int lv3 = wrapper.getStaffNum().intValue();
				int lv2 = lv2DeparmentMap.get(d.getParentId()).getStaffNum().intValue();
				lv2DeparmentMap.get(d.getParentId()).setStaffNum(lv2 + lv3);
			}
		}
		for (Department d : departments) {
			DepartmentWrapper wrapper = DepartmentWrapper.of(d);
			if (leadersMap.containsKey(d.getLeader())) {
				OAUserWrapper leader = OAUserWrapper.ofSimple(leadersMap.get(d.getLeader()));
				wrapper.setLeader(leader);
			}
			if (deptStaff.containsKey(d.getId())) {
				wrapper.setStaffNum(deptStaff.get(d.getId()));
			} else {
				wrapper.setStaffNum(0);
			}
			wrapper.setPositions(parent2PosMap.get(d.getId()));
			if (d.getLevel() == 1) {
				lv1DeparmentMap.put(d.getId(), wrapper);
			} else { // lv2
				if (!lv1DeparmentMap.containsKey(d.getParentId())) {
					// dirty
					continue;
				}
				lv1DeparmentMap.get(d.getParentId()).addChild(lv2DeparmentMap.get(d.getId()));
				int lv2 = lv2DeparmentMap.get(d.getId()).getStaffNum().intValue();
				int lv1 = lv1DeparmentMap.get(d.getParentId()).getStaffNum().intValue();
				lv1DeparmentMap.get(d.getParentId()).setStaffNum(lv2 + lv1);
			}
		}
		return lv1DeparmentMap.values();
	}

	public Collection<DepartmentWrapper> getAllDepartmentSimple() {
		List<Department> departments = departmentRepository.getAllDepartment(); // 这里是level正序排好的
		Map<String, DepartmentWrapper> lv1DeparmentMap = Maps.newLinkedHashMap(); // 保证有序
		Map<String, DepartmentWrapper> lv2DeparmentMap = Maps.newLinkedHashMap(); // lv2deptId:lv2dept
		Set<String> deptIds = Sets.newHashSet();
		if (ArrayMapTools.isNotEmpty(departments)) {
			departments.forEach(it -> {
				deptIds.add(it.getId());
			});
		}
		for (Department d : departments) {
			DepartmentWrapper wrapper = DepartmentWrapper.ofSimple(d);
			if (d.getLevel() == 2) {
				lv2DeparmentMap.put(d.getId(), wrapper);
			} else if (d.getLevel() == 3 && lv2DeparmentMap.containsKey(d.getParentId())) { // lv2
				lv2DeparmentMap.get(d.getParentId()).addChild(wrapper);
			}
		}
		for (Department d : departments) {
			DepartmentWrapper wrapper = DepartmentWrapper.ofSimple(d);
			if (d.getLevel() == 1) {
				lv1DeparmentMap.put(d.getId(), wrapper);
			} else { // lv2
				if (!lv1DeparmentMap.containsKey(d.getParentId())) {
					// dirty
					continue;
				}
				lv1DeparmentMap.get(d.getParentId()).addChild(lv2DeparmentMap.get(d.getId()));
			}
		}
		return lv1DeparmentMap.values();
	}

	public Map<String, DepartmentWrapper> mutiGetDepartmentWithParentName(Collection<String> deptIds) {
		List<Department> currentDepartments = departmentRepository.mutiGetDepartment(deptIds);
		Set<String> parentIds = currentDepartments.stream().filter(it -> StringUtils.isNotEmpty(it.getParentId()))
				.map(it -> it.getParentId()).collect(Collectors.toSet());
		Map<String, String> parentNameMap = Maps.newHashMap();
		Set<String> rootIds = Sets.newHashSet();
		List<Department> parents = Lists.newArrayList();
		if (ArrayMapTools.isNotEmpty(parentIds)) { // 2级部门
			parents = departmentRepository.mutiGetDepartment(parentIds);
			parents.forEach(it -> {
				parentNameMap.put(it.getId(), it.getName());
				rootIds.add(it.getParentId());
			});
		}
		Map<String, String> parent2root = Maps.newHashMap();
		if (ArrayMapTools.isNotEmpty(rootIds)) { // 3级部门需要这块儿
			List<Department> roots = departmentRepository.mutiGetDepartment(rootIds);
			for (Department d : parents) {
				roots.forEach(it -> {
					if (StringUtils.equals(d.getParentId(), it.getId())) {
						parent2root.put(d.getId(), it.getName());
					}
				});
			}
		}

		Map<String, DepartmentWrapper> rsMap = Maps.newHashMap();
		for (Department d : currentDepartments) {
			DepartmentWrapper wrapper = DepartmentWrapper.of(d);
			wrapper.setParentName(MapUtils.getString(parentNameMap, d.getParentId()));
			wrapper.setRootName(MapUtils.getString(parent2root, d.getParentId()));
			rsMap.put(d.getId(), wrapper);
		}
		return rsMap;
	}

	public DepartmentWrapper getDepartmentWrapperWithParentName(String deptId) {
		Department d = departmentRepository.getDepartment(deptId);
		if (d == null) {
			return null;
		}
		DepartmentWrapper wrapper = DepartmentWrapper.of(d);
		if (StringUtils.isNotEmpty(d.getParentId())) {
			Department parent = departmentRepository.getDepartment(d.getParentId());
			if (parent != null) {
				wrapper.setParentName(parent.getName());
				if (StringUtils.isNotEmpty(parent.getParentId())) {
					Department root = departmentRepository.getDepartment(parent.getParentId());
					wrapper.setRootName(root.getName());
				}
			}
		}
		return wrapper;
	}

	public List<Department> multiGetSimpleDepartment(Collection<String> deptIds) {
		return departmentRepository.multiGetSimpleDepartment(deptIds);
	}

	public Map<String, Department> multiGetSimpleDepartmentHas(Collection<String> deptIds) {
		List<Department> departments = this.multiGetSimpleDepartment(deptIds);
		if (ArrayMapTools.isEmpty(departments))
			return Maps.newHashMap();
		return departments.stream().collect(Collectors.toMap(it -> it.getId(), it -> it));
	}

	public boolean checkExist(String id) {
		return departmentRepository.checkExist(id);
	}

	public boolean checkExist(String parentId, String name) {
		return departmentRepository.checkExist(parentId, name);
	}

	public Department getDepartment(String deptId) {
		return departmentRepository.getDepartment(deptId);
	}

	public void createDepartment(String id, String name, String parentId, Integer level, String leaderId) {
		departmentRepository.createDepartment(id, name, parentId, level, leaderId);
	}

	public void updateDepartment(String deptId, String name, String leaderId) {
		departmentRepository.updateDepartment(deptId, name, leaderId);
	}

	public void deleteDepartment(String deptId) {
		departmentRepository.deleteDepartment(deptId);
	}

	public void updateLeader(String currentlyLeader, String newleader) {
		departmentRepository.updateLeader(currentlyLeader, newleader);
	}

	public boolean checkExistChildren(String deptId) {
		return departmentRepository.checkExistChildren(deptId);
	}

}
