package com.jz.jar.oa.repository;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.jooq.Condition;
import org.jooq.impl.DSL;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Repository;

import com.google.common.collect.Maps;
import com.jz.common.utils.collection.ArrayMapTools;
import com.jz.jooq.oa.Tables;
import com.jz.jooq.oa.tables.pojos.User;

@Lazy
@Repository
public class OAUserRepository extends OABaseRepository {

	private static final com.jz.jooq.oa.tables.User U = Tables.USER;
	private static final String COMPANYDELETED = "companyDeleted";
	private static final String DEPTDELETED = "deptDeleted";
	private static final String POSITIONDELETED = "positionDeleted";
	private static final String WORKADDRDELETED = "workAddrDeleted";

	public void insUpdateUser(String uid, String chineseName, String englishName, String phone, String companyId,
			String workAddrId, String deptId, String positionId, String superiorId, String joinDate, String relations,
			Integer insurance, Integer childFree, Integer trainAgreement, Integer annualLeaveDays, String certificate) {
		oaCtx.insertInto(U).set(U.UID, uid).set(U.CHINESE_NAME, chineseName).set(U.ENGLISH_NAME, englishName)
				.set(U.PHONE, phone).set(U.COMPANY_ID, companyId).set(U.WORK_ADDR_ID, workAddrId).set(U.DEPT_ID, deptId)
				.set(U.POSITION_ID, positionId).set(U.SUPERIOR_ID, superiorId).set(U.JOIN_DATE, joinDate)
				.set(U.RELATIONS, relations).set(U.INSURANCE, insurance).set(U.CHILD_FREE, childFree)
				.set(U.TRAIN_AGREEMENT, trainAgreement).set(U.ANNUAL_LEAVE_DAYS, annualLeaveDays)
				.set(U.CERTIFICATE, certificate).onDuplicateKeyUpdate().set(U.CHINESE_NAME, chineseName)
				.set(U.ENGLISH_NAME, englishName).set(U.PHONE, phone).set(U.COMPANY_ID, companyId)
				.set(U.WORK_ADDR_ID, workAddrId).set(U.DEPT_ID, deptId).set(U.POSITION_ID, positionId)
				.set(U.SUPERIOR_ID, superiorId).set(U.JOIN_DATE, joinDate).set(U.RELATIONS, relations)
				.set(U.INSURANCE, insurance).set(U.CHILD_FREE, childFree).set(U.TRAIN_AGREEMENT, trainAgreement)
				.set(U.ANNUAL_LEAVE_DAYS, annualLeaveDays).set(U.CERTIFICATE, certificate).execute();
	}

	public void updateUserStatus(String uid, int status) {
		oaCtx.update(U).set(U.STATUS, status).where(U.UID.eq(uid)).execute();
	}

	public User getUser(String uid) {
		return oaCtx.selectFrom(U).where(U.UID.eq(uid).and(U.STATUS.eq(1))).fetchAnyInto(User.class);
	}
	
	public User getUserInfo(String uid) {
		return oaCtx.selectFrom(U).where(U.UID.eq(uid)).fetchAnyInto(User.class);
	}

	public List<User> mutGetUsers(Collection<String> uids) {
		return oaCtx.selectFrom(U).where(U.UID.in(uids).and(U.STATUS.eq(1))).fetchInto(User.class);
	}
	
	public List<User> mutGetAllUsers(Collection<String> uids) {
		return oaCtx.selectFrom(U).where(U.UID.in(uids)).fetchInto(User.class);
	}
	
	public List<User> multiGetSimpleUser(Collection<String> uids) {
		return oaCtx.select(U.UID, U.CHINESE_NAME).from(U).where(U.UID.in(uids).and(U.STATUS.eq(1)))
				.fetchInto(User.class);
	}

	public User getSimpleUser(String uid) {
		return oaCtx.select(U.UID, U.SUPERIOR_ID, U.CHINESE_NAME, U.POSITION_ID).from(U)
				.where(U.UID.eq(uid).and(U.STATUS.eq(1))).fetchAnyInto(User.class);
	}

	public boolean isIncumbency(String uid) {
		return oaCtx.fetchExists(U, U.UID.eq(uid).and(U.STATUS.eq(1)));
	}

	public int cntDeptUser(String deptId) {
		return oaCtx.fetchCount(U, U.DEPT_ID.eq(deptId).and(U.STATUS.eq(1)));
	}

	public void updateUserDept(String deptId) {
		oaCtx.update(U).set(U.DEPT_ID, DEPTDELETED).where(U.DEPT_ID.eq(deptId)).execute();
	}

	public int cntPositionUser(String id) {
		return oaCtx.fetchCount(U, U.POSITION_ID.eq(id).and(U.STATUS.eq(1)));
	}

	public void updateUserPosition(String id) {
		oaCtx.update(U).set(U.POSITION_ID, POSITIONDELETED).where(U.POSITION_ID.eq(id)).execute();
	}

	public Map<String, Integer> getStaffNumByCompanyIds(Set<String> companyIds) {
		List<Map<String, Object>> list = oaCtx.select(U.COMPANY_ID.as("companyId"), DSL.count().as("cnt")).from(U)
				.where(U.COMPANY_ID.in(companyIds)).groupBy(U.COMPANY_ID).fetchMaps();
		Map<String, Integer> rsMap = Maps.newHashMap();
		for (Map<String, Object> map : list) {
			rsMap.put(MapUtils.getString(map, "companyId"), MapUtils.getIntValue(map, "cnt"));
		}
		return rsMap;
	}

	public int cntCompanyUser(String id) {
		return oaCtx.fetchCount(U, U.COMPANY_ID.eq(id).and(U.STATUS.eq(1)));
	}

	public void updateUserCompany(String id) {
		oaCtx.update(U).set(U.COMPANY_ID, COMPANYDELETED).where(U.COMPANY_ID.eq(id)).execute();
	}

	public Map<String, Integer> getStaffNumByWorkAddressIds(Set<String> workAddressIds) {
		List<Map<String, Object>> list = oaCtx.select(U.WORK_ADDR_ID.as("workAddressId"), DSL.count().as("cnt")).from(U)
				.where(U.WORK_ADDR_ID.in(workAddressIds)).groupBy(U.WORK_ADDR_ID).fetchMaps();
		Map<String, Integer> rsMap = Maps.newHashMap();
		for (Map<String, Object> map : list) {
			rsMap.put(MapUtils.getString(map, "workAddressId"), MapUtils.getIntValue(map, "cnt"));
		}
		return rsMap;
	}

	public int cntWorkAddressUser(String id) {
		return oaCtx.fetchCount(U, U.WORK_ADDR_ID.eq(id).and(U.STATUS.eq(1)));
	}

	public void updateUserWorkAddress(String id) {
		oaCtx.update(U).set(U.WORK_ADDR_ID, WORKADDRDELETED).where(U.WORK_ADDR_ID.eq(id)).execute();
	}

	public Map<String, Integer> getStaffNumByDeptIds(Set<String> deptIds) {
		List<Map<String, Object>> list = oaCtx.select(U.DEPT_ID.as("deptId"), DSL.count().as("cnt")).from(U)
				.where(U.DEPT_ID.in(deptIds)).groupBy(U.DEPT_ID).fetchMaps();
		Map<String, Integer> rsMap = Maps.newHashMap();
		for (Map<String, Object> map : list) {
			rsMap.put(MapUtils.getString(map, "deptId"), MapUtils.getIntValue(map, "cnt"));
		}
		return rsMap;
	}

	public List<User> getUserInfoPage(String companyId, String keyword, int start, int size) {
		return oaCtx.selectFrom(U).where(this.searchWhereCondition(companyId, keyword)).and(U.STATUS.eq(1))
				.orderBy(U.JOIN_DATE.asc(), U.UID.desc()).limit(start, size).fetchInto(User.class);
	}

	private Condition searchWhereCondition(String companyId, String keyword) {
		Condition condition = DSL.trueCondition();
		if (StringUtils.isNotEmpty(companyId)) {
			condition = condition.and(U.COMPANY_ID.eq(companyId));
		}
		if (StringUtils.isNotEmpty(keyword)) {
			condition = condition
					.and(U.CHINESE_NAME.like("%" + keyword + "%").or(U.ENGLISH_NAME.like("%" + keyword + "%")));
		}
		return condition;
	}

	public User getUserByPosition(String positionId) {
		return oaCtx.selectFrom(U).where(U.POSITION_ID.eq(positionId)).fetchAnyInto(User.class);
	}

	public List<String> getDeptIdsForCompany(String companyId) {
		return oaCtx.selectDistinct(U.DEPT_ID).from(U).where(U.COMPANY_ID.eq(companyId).and(U.STATUS.eq(1)))
				.fetchInto(String.class);
	}

	public List<User> getUsersForDepartment(String companyId, String deptId) {
		return oaCtx.select(U.UID, U.CHINESE_NAME).from(U)
				.where(U.COMPANY_ID.eq(companyId).and(U.STATUS.eq(1)).and(U.DEPT_ID.eq(deptId))).fetchInto(User.class);
	}

	public User getUserByPhone(String phone) {
		return oaCtx.selectFrom(U).where(U.PHONE.eq(phone).and(U.STATUS.eq(1))).fetchAnyInto(User.class);
	}

	public String getSuperiorUid(String uid) {
		return oaCtx.select(U.SUPERIOR_ID).from(U).where(U.UID.eq(uid).and(U.STATUS.eq(1))).fetchAnyInto(String.class);
	}

	public void updateSuperior(String currentlySuperior, String newSuperior) {
		oaCtx.update(U).set(U.SUPERIOR_ID, newSuperior).where(U.SUPERIOR_ID.eq(currentlySuperior)).execute();
	}

	public List<User> getUsersForWorkAddress(String workAddrId, int start, int size) {
		return oaCtx.select(U.UID, U.CHINESE_NAME, U.ENGLISH_NAME).from(U)
				.where(U.WORK_ADDR_ID.eq(workAddrId).and(U.STATUS.eq(1))).orderBy(U.JOIN_DATE.asc(), U.UID.desc())
				.limit(start, size).fetchInto(User.class);
	}

	private Condition getConditions(String uid, String departmentId, String companyId, String workAddrId,
			String keyword, Map<String, String> filterMaps) {
		Condition condition = DSL.trueCondition();
		if (ArrayMapTools.isNotEmpty(filterMaps))
			condition = condition.and(U.WORK_ADDR_ID.in(filterMaps.keySet())).and(U.COMPANY_ID.in(filterMaps.values()));
		if (StringUtils.isNotEmpty(departmentId)) {
			condition = condition.and(U.DEPT_ID.eq(departmentId));
		}
		if (StringUtils.isNotEmpty(companyId)) {
			condition = condition.and(U.COMPANY_ID.eq(companyId));
		}
		if (StringUtils.isNotEmpty(workAddrId)) {
			condition = condition.and(U.WORK_ADDR_ID.eq(workAddrId));
		}
		if (StringUtils.isNotEmpty(keyword)) {
			condition = condition.and(U.CHINESE_NAME.like("%" + keyword + "%")
					.or(U.ENGLISH_NAME.like("%" + keyword + "%")).or(U.PHONE.like("%" + keyword + "%")));
		}
		return condition;
	}

	public List<String> filterUidsByConditionsLimit(String uid, String departmentId, String companyId,
			String workAddrId, String keyword, Map<String, String> filterMaps, int start, int size) {
		return oaCtx.select(U.UID).from(U)
				.where(this.getConditions(uid, departmentId, companyId, workAddrId, keyword, filterMaps)
						.and(U.STATUS.eq(1)))
				.orderBy(U.JOIN_DATE.asc(), U.UID.desc()).limit(start, size).fetchInto(String.class);
	}

	public int cntUsersByConditions(String uid, String departmentId, String companyId, String workAddrId,
			String keyword, Map<String, String> filterMaps) {
		return oaCtx.fetchCount(U,
				this.getConditions(uid, departmentId, companyId, workAddrId, keyword, filterMaps).and(U.STATUS.eq(1)));
	}

	public int cntUsers(List<String> filterUids) {
		return oaCtx.fetchCount(U, U.UID.in(filterUids));
	}

	public List<String> filterRegularEmployee(String date) {
		return oaCtx.select(U.UID).from(U).where(U.STATUS.eq(1).and(U.JOIN_DATE.ge(date))).fetchInto(String.class);
	}

	public void addFloatAnnualLeaveDays(List<String> filterUids) {
		oaCtx.update(U).set(U.FLOAT_ANNUAL_LEAVE_DAYS, U.FLOAT_ANNUAL_LEAVE_DAYS.add(1)).where(U.UID.in(filterUids))
				.execute();
	}

	public int cntUsers(String uid, int status, String departmentId, String companyId, String workAddrId,
			String keyword) {
		return oaCtx.fetchCount(U,
				this.getConditions(uid, departmentId, companyId, workAddrId, keyword, Maps.newHashMap())
						.and(U.STATUS.eq(status)));
	}

	public List<User> filterUsersByConditionsLimit(String uid, int status, String departmentId, String companyId,
			String workAddrId, String keyword, int start, int size) {
		return oaCtx.selectFrom(U)
				.where(U.STATUS.eq(status)
						.and(this.getConditions(uid, departmentId, companyId, workAddrId, keyword, Maps.newHashMap())))
				.orderBy(U.JOIN_DATE.desc(), U.UID.asc()).limit(start, size).fetchInto(User.class);
	}

	public List<User> getRegularEmployees() {
		return oaCtx.selectFrom(U).where(U.STATUS.eq(1)).fetchInto(User.class);
	}

	public List<User> getHrList(List<String> hrs) {
		return oaCtx.select(U.UID, U.CHINESE_NAME, U.ENGLISH_NAME).from(U).where(U.STATUS.eq(1).and(U.UID.in(hrs)))
				.fetchInto(User.class);
	}

}
