package com.jz.jar.media.repository;

import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import org.jooq.Condition;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Repository;

import com.jz.common.utils.collection.ArrayMapTools;
import com.jz.jar.media.enums.BrandEnum;
import com.jz.jar.media.enums.SourceStatus;
import com.jz.jooq.media.Tables;
import com.jz.jooq.media.tables.pojos.UploadVideo;
import com.jz.jooq.media.tables.records.UploadVideoRecord;

@Lazy
@Repository
public class UploadVideoRepository extends MediaBaseRepository {

	private static final com.jz.jooq.media.tables.UploadVideo UV = Tables.UPLOAD_VIDEO;

	public void saveToWaiting(String wid, String uid, String name, String app) {
		long timeMillis = System.currentTimeMillis();
		mediaCtx.insertInto(UV, UV.WID, UV.UID, UV.APP, UV.NAME, UV.STATUS, UV.CREATE_TIME, UV.LAST_UPDATE)
				.values(wid, uid, app, name, SourceStatus.waiting.getCode(), timeMillis, timeMillis)
				.onDuplicateKeyIgnore().execute();
	}

	public void updateVideoInfo(UploadVideoRecord record) {
		mediaCtx.update(UV).set(record).where(UV.WID.eq(record.getWid())).execute();
	}

	private Condition getWhereCondition(String uid, Collection<SourceStatus> status, BrandEnum brand) {
		List<Integer> ss = status.stream().map(it -> it.getCode()).collect(Collectors.toList());
		return UV.STATUS.in(ss).and(UV.UID.eq(uid)).and(UV.APP.eq(brand.name()));
	}

	public int countVideos(String uid, Collection<SourceStatus> status, BrandEnum brand) {
		return mediaCtx.fetchCount(UV, getWhereCondition(uid, status, brand));
	}

	public List<String> getVideos(String uid, Collection<SourceStatus> status, BrandEnum brand, int start, int size) {
		return mediaCtx.select(UV.WID).from(UV).where(getWhereCondition(uid, status, brand))
				.orderBy(UV.STATUS.desc(), UV.CREATE_TIME.desc()).limit(start, size).fetchInto(String.class);
	}

	public List<UploadVideo> getVideoInfo(Collection<String> wids) {
		return mediaCtx.selectFrom(UV).where(UV.WID.in(wids)).fetchInto(UploadVideo.class);
	}

	private Condition getOnlineNotUidWhereCondition(String uid, BrandEnum brand) {
		return UV.STATUS.eq(SourceStatus.online.getCode()).and(UV.APP.eq(brand.name())).and(UV.UID.ne(uid));
	}

	public int countOnlineVideosForNotUid(String uid, BrandEnum brand) {
		return mediaCtx.fetchCount(UV, getOnlineNotUidWhereCondition(uid, brand));
	}

	public List<String> getOnlineVideosForNotUid(String uid, BrandEnum brand, int start, int size) {
		return mediaCtx.select(UV.WID).from(UV).where(getOnlineNotUidWhereCondition(uid, brand))
				.orderBy(UV.CREATE_TIME.desc()).limit(start, size).fetchInto(String.class);
	}

	public List<String> getOnlineVideos(String uid, Collection<String> wids) {
		return mediaCtx.select(UV.WID).from(UV)
				.where(UV.WID.in(wids).and(UV.STATUS.eq(SourceStatus.online.getCode())).and(UV.UID.eq(uid)))
				.fetchInto(String.class);
	}

	public void deleteVideos(String uid, Collection<String> wids) {
		mediaCtx.deleteFrom(UV).where(UV.WID.in(wids).and(UV.UID.eq(uid))).execute();
	}

	public UploadVideo getOnlineVideo(String uid, String wid) {
		List<UploadVideo> videos = mediaCtx.selectFrom(UV)
				.where(UV.WID.eq(wid).and(UV.STATUS.eq(SourceStatus.online.getCode())).and(UV.UID.eq(uid)))
				.fetchInto(UploadVideo.class);
		return ArrayMapTools.isNotEmpty(videos) ? videos.get(0) : null;
	}

	public List<String> getOnlineVideoIds(String uid) {
		return mediaCtx.select(UV.WID).from(UV)
				.where(UV.STATUS.eq(SourceStatus.online.getCode()).and(UV.UID.eq(uid))).fetchInto(String.class);
	}

	public UploadVideo getVideo(String wid) {
		return mediaCtx.selectFrom(UV).where(UV.WID.eq(wid)).fetchAnyInto(UploadVideo.class);
	}
}
