package com.jz.jar.media.repository;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.jooq.Condition;
import org.jooq.TableOnConditionStep;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Repository;

import com.jz.common.utils.text.StringTools;
import com.jz.jar.media.enums.BrandEnum;
import com.jz.jar.media.enums.SourceStatus;
import com.jz.jooq.media.Tables;
import com.jz.jooq.media.tables.pojos.ArtInfo;
import com.jz.jooq.media.tables.records.ArtInfoRecord;

@Lazy
@Repository
public class ArtInfoRepository extends MediaBaseRepository implements BaseConditionRepository<ArtInfoRecord> {

	private static final com.jz.jooq.media.tables.ArtInfo AI = Tables.ART_INFO;
	private static final com.jz.jooq.media.tables.ArtThemeRelation ATR = Tables.ART_THEME_RELATION;
	
	public boolean existArtInfo(String id, BrandEnum brand) {
		return mediaCtx.fetchExists(AI, AI.ID.eq(id).and(AI.BRAND.eq(brand.name())));
	}
	
	public ArtInfo getArtInfo(String id, BrandEnum brand) {
		return mediaCtx.selectFrom(AI).where(AI.ID.eq(id).and(AI.BRAND.eq(brand.name()))).fetchAnyInto(ArtInfo.class);
	}

	public ArtInfo getSimpleArtInfo(String id, BrandEnum brand) {
		return mediaCtx.select(AI.ID, AI.SCHOOL_ID, AI.LESSON_ID, AI.PUID, AI.SUID, AI.IS_OPEN).from(AI)
				.where(AI.ID.eq(id).and(AI.BRAND.eq(brand.name()))).fetchAnyInto(ArtInfo.class);
	}
	
	public List<ArtInfo> findSimpleArtInfo(Collection<String> artIds) {
		return mediaCtx.select(AI.ID, AI.PUID, AI.RESOURCE, AI.IS_OPEN).from(AI).where(AI.ID.in(artIds))
				.fetchInto(ArtInfo.class);
	}
	
	public void updateOpenInfo(String id, BrandEnum brand, int isOpen) {
		mediaCtx.update(AI).set(AI.IS_OPEN, isOpen).set(AI.LAST_UPDATE, System.currentTimeMillis())
				.where(AI.ID.eq(id).and(AI.BRAND.eq(brand.name()))).execute();
	}
	
	/** 关注列表 */
	private Condition getCondition(BrandEnum brand, Collection<String> uids) {
		return AI.BRAND.eq(brand.name()).and(AI.PUID.in(uids)).and(AI.IS_OPEN.ge(1))
				.and(AI.STATUS.eq(SourceStatus.online.getCode()));
	}

	public int countOpenNewArt(BrandEnum brand, Collection<String> uids, long timestamp) {
		return mediaCtx.fetchCount(AI, getCondition(brand, uids).and(AI.CREATE_TIME.gt(timestamp)));
	}

	public int countOpenArt(BrandEnum brand, Collection<String> uids) {
		return mediaCtx.fetchCount(AI, getCondition(brand, uids));
	}

	public List<String> findOpenArtIds(BrandEnum brand, Collection<String> uids, int start, int size) {
		return mediaCtx.select(AI.ID).from(AI).where(getCondition(brand, uids)).orderBy(AI.CREATE_TIME.desc())
				.limit(start, size).fetchInto(String.class);
	}

	/** 推荐列表 */
	private Condition getRecommonCondition(BrandEnum brand) {
		return AI.BRAND.eq(brand.name()).and(AI.IS_OPEN.ge(1)).and(AI.STATUS.eq(SourceStatus.online.getCode()));
	}
	
	public int countRecommonArts(BrandEnum brand) {
		return mediaCtx.fetchCount(AI, getRecommonCondition(brand));
	}
	
	public List<String> findRecommonArtIds(BrandEnum brand, int start, int size) {
		return mediaCtx.select(AI.ID).from(AI).where(getRecommonCondition(brand)).orderBy(AI.CREATE_TIME.desc())
				.limit(start, size).fetchInto(String.class);
	}
	
	/** 获取存在作品的学校及课程 */
	public List<ArtInfo> findExistArtInfo(Map<String, Collection<String>> school2LessonIds, String suid) {
		Condition condition = getOrWhereCondition(AI.SCHOOL_ID, AI.LESSON_ID, school2LessonIds);
		return mediaCtx.select(AI.SCHOOL_ID, AI.LESSON_ID, AI.ID).from(AI).where(condition.and(AI.SUID.eq(suid)))
				.fetchInto(ArtInfo.class);
	}
	
	/** 学员作品列表 */
	private Condition getStudentCondition(BrandEnum brand, String puid, String suid) {
		Condition condition = AI.BRAND.eq(brand.name()).and(AI.PUID.eq(puid));
		if (StringTools.isNotEmpty(suid))
			return condition.and(AI.SUID.eq(suid));
		return condition;
	}

	public int countStudentArts(BrandEnum brand, String puid, String suid) {
		return mediaCtx.fetchCount(AI, getStudentCondition(brand, puid, suid));
	}

	public List<String> findStudentArtIds(BrandEnum brand, String puid, String suid, int start, int size) {
		return mediaCtx.select(AI.ID).from(AI).where(getStudentCondition(brand, puid, suid))
				.orderBy(AI.CREATE_TIME.desc()).limit(start, size).fetchInto(String.class);
	}
	
	public List<String> findAllStudentArtIds(BrandEnum brand, String puid, String suid) {
		return mediaCtx.select(AI.ID).from(AI).where(getStudentCondition(brand, puid, suid)).fetchInto(String.class);
	}
	
	/********************************************************************/
	/*********************** art theme relation *************************/
	/********************************************************************/
	
	private TableOnConditionStep getJoinTable() {
		return ATR.join(AI).on(ATR.ART_ID.eq(AI.ID));
	}
	
	private Condition getRelationCondition(BrandEnum brand, String themeId) {
		return ATR.THEME_ID.eq(themeId).and(AI.BRAND.eq(brand.name())).and(AI.IS_OPEN.ge(1))
				.and(AI.STATUS.eq(SourceStatus.online.getCode()));
	}
	
	public int countArts(BrandEnum brand, String themeId) {
		return mediaCtx.fetchCount(getJoinTable(), getRelationCondition(brand, themeId));
	}

	public List<String> getArtIds(BrandEnum brand, String themeId, boolean isHot, int start, int size) {
		return mediaCtx.select(ATR.ART_ID).from(getJoinTable()).where(getRelationCondition(brand, themeId))
				.orderBy(isHot ? ATR.LIKE_CNT.desc() : ATR.UPLOAD_TIME.desc()).limit(start, size)
				.fetchInto(String.class);
	}
	
	public List<String> getAllThemeIds(String artId) {
		return mediaCtx.select(ATR.THEME_ID).from(ATR).where(ATR.ART_ID.eq(artId)).fetchInto(String.class);
	}
	
	public void addLikeCnt(String artId, int cnt) {
		mediaCtx.update(ATR).set(ATR.LIKE_CNT, ATR.LIKE_CNT.add(cnt)).where(ATR.ART_ID.eq(artId)).execute();
	}
}
