package com.jz.jar.media.repository;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.jooq.Condition;
import org.jooq.impl.DSL;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Repository;

import com.google.common.collect.Lists;
import com.jz.common.utils.collection.ArrayMapTools;
import com.jz.common.utils.text.StringTools;
import com.jz.jar.media.enums.BrandEnum;
import com.jz.jooq.media.Tables;
import com.jz.jooq.media.tables.pojos.WorksAuthor;
import com.jz.jooq.media.tables.records.WorksAuthorRecord;

@Lazy
@Repository
public class WorksAuthorRepository extends MediaBaseRepository implements BaseConditionRepository<WorksAuthorRecord> {

	private static final com.jz.jooq.media.tables.WorksAuthor WA = Tables.WORKS_AUTHOR;

	private Condition getWorkIdCondition(BrandEnum brand, String workId) {
		return WA.WORK_ID.eq(workId).and(WA.BRAND.eq(brand.name()));
	}

	public List<String> getWorksAuthors(BrandEnum brand, String workId) {
		return mediaCtx.select(WA.SUID).from(WA).where(getWorkIdCondition(brand, workId)).fetchInto(String.class);
	}

	public List<String> getWorksAuthorParents(BrandEnum brand, String workId) {
		return mediaCtx.select(WA.PUID).from(WA).where(getWorkIdCondition(brand, workId)).fetchInto(String.class);
	}

	public boolean isWorksAuthor(BrandEnum brand, String workId, String suid) {
		return mediaCtx.fetchExists(WA, getWorkIdCondition(brand, workId).and(WA.SUID.eq(suid)));
	}

	public boolean isWorksAuthorParent(BrandEnum brand, String workId, String puid) {
		return mediaCtx.fetchExists(WA, getWorkIdCondition(brand, workId).and(WA.PUID.eq(puid)));
	}

	public String getWorksAuthor(BrandEnum brand, String workId) {
		return mediaCtx.select(WA.SUID).from(WA).where(getWorkIdCondition(brand, workId)).fetchAnyInto(String.class);
	}

	public String getWorksAuthorParent(BrandEnum brand, String workId) {
		return mediaCtx.select(WA.PUID).from(WA).where(getWorkIdCondition(brand, workId)).fetchAnyInto(String.class);
	}

	public List<WorksAuthor> findWorkAuthor(BrandEnum brand, Collection<String> workIds) {
		return mediaCtx.selectFrom(WA).where(WA.WORK_ID.in(workIds).and(WA.BRAND.eq(brand.name())))
				.fetchInto(WorksAuthor.class);
	}

	/** 学员作品列表 */
	private Condition getStudentCondition(BrandEnum brand, Collection<String> puids, String suid, Integer isOpen) {
		if (ArrayMapTools.isEmpty(puids)) {
			throw new RuntimeException("puids empty");
		}
		Condition condition = WA.BRAND.eq(brand.name()).and(WA.PUID.in(puids));
		if (StringTools.isNotEmptyAndBlank(suid)) {
			condition = condition.and(WA.SUID.eq(suid));
		}
		if (null != isOpen) {
			condition = condition.and(WA.IS_OPEN.eq(isOpen));
		}
		return condition;
	}

	public int countStudentWorks(BrandEnum brand, Collection<String> puids, String suid, Integer isOpen) {
		return mediaCtx.fetchCount(WA, getStudentCondition(brand, puids, suid, isOpen));
	}

	public List<String> findStudentWorksId(BrandEnum brand, Collection<String> puids, String suid, Integer isOpen,
			int start, int size) {
		return mediaCtx.select(WA.WORK_ID).from(WA).where(getStudentCondition(brand, puids, suid, isOpen))
				.orderBy(WA.ID.desc()).limit(start, size).fetchInto(String.class);
	}

	public List<String> findAllStudentWorksId(BrandEnum brand, Collection<String> puids, String suid, Integer isOpen) {
		return mediaCtx.select(WA.WORK_ID).from(WA).where(getStudentCondition(brand, puids, suid, isOpen))
				.fetchInto(String.class);
	}

	/** 获取存在作品的学校及课程 */
	public List<WorksAuthor> findExistWorksInfo(Map<String, Collection<String>> school2LessonIds, String suid) {
		Condition condition = getOrWhereCondition(WA.SCHOOL_ID, WA.LESSON_ID, school2LessonIds);
		return mediaCtx.select(WA.SCHOOL_ID, WA.LESSON_ID, WA.WORK_ID, WA.IS_COLLECTIVE).from(WA)
				.where(condition.and(WA.SUID.eq(suid))).fetchInto(WorksAuthor.class);
	}

	public String getWorkIdByLesson(String schoolId, String lessonId, String suid) {
		return mediaCtx.select(WA.WORK_ID).from(WA)
				.where(WA.SCHOOL_ID.eq(schoolId).and(WA.LESSON_ID.eq(lessonId)).and(WA.SUID.eq(suid)))
				.orderBy(WA.IS_COLLECTIVE.asc(), WA.ID.desc()).limit(1).fetchAnyInto(String.class);
	}

	public void updateOpenInfo(String workId, int isOpen) {
		mediaCtx.update(WA).set(WA.IS_OPEN, isOpen).where(WA.WORK_ID.eq(workId)).execute();
	}

	public void createWorkAuthor(String brandId, String id, String schoolId, String lessonId, String puid, String suid,
			int isCollective) {
		mediaCtx.insertInto(WA, WA.BRAND, WA.WORK_ID, WA.SCHOOL_ID, WA.LESSON_ID, WA.PUID, WA.SUID, WA.IS_COLLECTIVE)
				.values(brandId, id, schoolId, lessonId, puid, suid, isCollective).onDuplicateKeyIgnore().execute();
	}

	public int countStudentWorksBySuids(BrandEnum brand, Collection<String> suids, Integer isOpen) {
		return mediaCtx.fetchCount(WA, this.genBySuidsCondition(brand, suids, isOpen));
	}

	public List<String> findStudentWorksIdBySuids(BrandEnum brand, Collection<String> suids, Integer isOpen, int start,
			int size) {
		return mediaCtx.select(WA.WORK_ID).from(WA).where(this.genBySuidsCondition(brand, suids, isOpen))
				.orderBy(WA.ID.desc()).limit(start, size).fetchInto(String.class);
	}

	public List<String> findAllStudentWorksIdBySuids(BrandEnum brand, Collection<String> suids, Integer isOpen) {
		return mediaCtx.select(WA.WORK_ID).from(WA).where(this.genBySuidsCondition(brand, suids, isOpen))
				.fetchInto(String.class);
	}

	/** 学员作品列表 */
	private Condition genBySuidsCondition(BrandEnum brand, Collection<String> suids, Integer isOpen) {
		if (ArrayMapTools.isEmpty(suids)) {
			throw new RuntimeException("suids empty");
		}
		Condition condition = WA.BRAND.eq(brand.name()).and(WA.SUID.in(suids));
		if (null != isOpen) {
			condition = condition.and(WA.IS_OPEN.eq(isOpen));
		}
		return condition;
	}

	public List<Map<String, Object>> getStudentWorksCnt(BrandEnum brand, List<String> suids) {
		return mediaCtx.select(WA.SUID.as("suid"), DSL.count(WA.WORK_ID).as("worksCnt")).from(WA)
				.where(WA.BRAND.eq(brand.name()).and(WA.SUID.in(suids))).groupBy(WA.SUID).fetchMaps();
	}

	public List<WorksAuthor> filterWorksAuthor(String suid, Map<String, Collection<String>> school2Lessons) {
		List<Condition> conditions = Lists.newArrayList();
		school2Lessons.forEach((schoolId, lessonIds) -> {
			conditions.add(WA.SCHOOL_ID.eq(schoolId).and(WA.LESSON_ID.in(lessonIds)));
		});
		return mediaCtx.selectFrom(WA)
				.where(DSL.or(conditions).and(WA.IS_OPEN.eq(1).or(WA.IS_OPEN.eq(0).and(WA.SUID.eq(suid)))))
				.fetchInto(WorksAuthor.class);
	}
}
