package com.jz.jar.media.repository;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.jooq.Record2;
import org.jooq.Record3;
import org.jooq.Result;
import org.jooq.impl.DSL;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Repository;

import com.google.common.collect.Maps;
import com.jz.common.utils.collection.ArrayMapTools;
import com.jz.jooq.media.Tables;
import com.jz.jooq.media.tables.pojos.PlaylistRelation;

@Lazy
@Repository
public class PlaylistRelationRepository extends MediaBaseRepository {

	private static final com.jz.jooq.media.tables.PlaylistRelation PR = Tables.PLAYLIST_RELATION;

	/** 获取所有的级别和分类 */
	public List<PlaylistRelation> findLevelTopics() {
		return mediaCtx.select(PR.LID, PR.CID).from(PR).groupBy(PR.LID, PR.CID).fetchInto(PlaylistRelation.class);
	}

	/** 获取当前专辑所有的cid */
	public List<String> getTopicsForPlaylist(String pid) {
		return mediaCtx.selectDistinct(PR.CID).from(PR).where(PR.PID.eq(pid)).fetchInto(String.class);
	}

	/** 获取当前级别及分类下所有的专辑数量 */
	public int countPlaylistForLevelAndTopic(String lid, Collection<String> cidArray) {
		return mediaCtx.select(DSL.countDistinct(PR.PID)).from(PR).where(PR.LID.eq(lid).and(PR.CID.in(cidArray)))
				.fetchAnyInto(int.class);
	}

	/** 获取当前级别及分类下所有的专辑id */
	public List<String> findPlaylistForLevelAndTopic(String lid, Collection<String> cidArray, int start, int size) {
		return mediaCtx.selectDistinct(PR.PID).from(PR).where(PR.LID.eq(lid).and(PR.CID.in(cidArray)))
				.orderBy(PR.PID.asc()).limit(start, size).fetchInto(String.class);
	}
	
	/** 是否专项练习专辑 */
	public boolean isSpecialPlaylist(String lid, String pid) {
		return mediaCtx.fetchExists(PR, PR.PID.eq(pid).and(PR.LID.eq(lid)).and(PR.IS_SPECIAL.eq(1)));
	}
	
	/** 获取专项练习的分类及专辑数量 */
	public Map<String, Integer> findSpecialTopicToCount(String lid) {
		Result<Record2<String, Integer>> result = mediaCtx.select(PR.CID, DSL.count(PR.CID)).from(PR)
				.where(PR.LID.eq(lid).and(PR.IS_SPECIAL.eq(1))).groupBy(PR.CID).fetch();
		if (null == result || result.isEmpty())
			return null;
		Map<String, Integer> topicToCount = Maps.newHashMap();
		result.forEach(it -> topicToCount.put(it.value1(), it.value2()));
		return topicToCount;
	}
	
	/** 获取专项练习的专辑数量 */
	public int countSpecialPlaylist(String lid, String cid) {
		return mediaCtx.fetchCount(PR, PR.LID.eq(lid).and(PR.CID.eq(cid)).and(PR.IS_SPECIAL.eq(1)));
	}
	
	/** 获取专项练习的专辑列表 */
	public List<String> findSpecialPlaylist(String lid, String cid) {
		return mediaCtx.select(PR.PID).from(PR).where(PR.LID.eq(lid).and(PR.CID.eq(cid)).and(PR.IS_SPECIAL.eq(1)))
				.orderBy(PR.SPECIAL_SEQ.desc()).fetchInto(String.class);
	}
	
	/** 根据级别和专辑id获取分类 */
	public List<String> findPlaylistTopic(String lid, Collection<String> pidArray) {
		return mediaCtx.selectDistinct(PR.CID).from(PR).where(PR.PID.in(pidArray).and(PR.LID.eq(lid))).fetchInto(String.class);
	}
	
	/** 根据级别和分类列表获取最大播放数的专辑id */
	public Map<String, String> findMaxWatchCntPlaylistForCids(String lid, Collection<String> cidArray) {
		Result<Record3<String, String, Integer>> result = mediaCtx.select(PR.PID, PR.CID, DSL.max(PR.WATCH_CNT))
				.from(PR).where(PR.LID.eq(lid).and(PR.CID.in(cidArray))).groupBy(PR.CID).fetch();
		if (ArrayMapTools.isEmpty(result))
			return Maps.newHashMap();
		Map<String, String> has = Maps.newHashMap();
		result.forEach(it -> has.put(it.value1(), it.value2()));
		return has;
	}
}
