package com.jz.jar.media.proxy;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.collections.MapUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import com.google.common.collect.Lists;
import com.jz.common.utils.collection.ArrayMapTools;
import com.jz.common.utils.collection.OrderTool;
import com.jz.common.utils.json.GsonTools;
import com.jz.common.utils.text.StringTools;
import com.jz.jar.media.beans.ArtResource;
import com.jz.jar.media.enums.BrandEnum;
import com.jz.jar.media.request.ArtRequest;
import com.jz.jar.media.request.UserRequest;
import com.jz.jar.media.service.ArtInfoService;
import com.jz.jar.media.service.ArtLikeService;
import com.jz.jar.media.tool.AliyunBean;
import com.jz.jar.media.tool.PictureCompress;
import com.jz.jar.media.wrapper.ArtInfoWrapper;
import com.jz.jar.media.wrapper.UserWrapper;
import com.jz.jooq.media.tables.pojos.ArtInfo;

/**
 * @Title ArtInfoProxy
 * @Package com.jz.jar.media.proxy
 * @author tangjunfeng
 * @date 2019年1月12日 下午5:43:51
 * @version V1.0
 */
@Lazy
@Service
public class ArtInfoProxy {

	@Autowired
	private ArtInfoService artInfoService;
	@Autowired
	private ArtLikeService artLikeService;
	@Autowired
	private UserProxy userProxy;

	public List<ArtInfoWrapper> getSimpleArtInfo(BrandEnum brand, ArtRequest request) {
		List<ArtInfo> artInfos = artInfoService.findSimpleArtInfo(request.getArtIds());
		if (ArrayMapTools.isEmpty(artInfos))
			return null;
		// get user info
		Set<String> puids = artInfos.stream().map(ArtInfo::getPuid).collect(Collectors.toSet());
		Map<String, UserWrapper> uid2User = userProxy.getSimpleParentInfoHas(brand, UserRequest.of(puids));
		// get art like cnt
		Map<String, Integer> likeCount = artLikeService.getLikeCount(request.getArtIds());
		// get like art ids
		List<String> likes = artLikeService.screenLike(request.getArtIds(), request.getUid());
		// package wrapper
		List<ArtInfoWrapper> wrappers = Lists.newArrayList();
		for (ArtInfo artInfo : artInfos) {
			// get show resources
			ArtResource resource = GsonTools.gson.fromJson(artInfo.getResource(), ArtResource.class);
			if (null == resource)
				continue;
			ArtInfoWrapper wrapper = ArtInfoWrapper.of(artInfo);
			// set resource
			wrapper.setType(resource.getType()).setWid(resource.getWid());
			wrapper.setPic(AliyunBean.getImagesUrl(resource.getPic(), PictureCompress._260x146));
			// set user info
			wrapper.setUser(uid2User.get(artInfo.getPuid()));
			// set like cnt
			wrapper.setLikeCnt(MapUtils.getInteger(likeCount, artInfo.getId(), 0));
			// set is like
			wrapper.setIsLike(likes.contains(artInfo.getId()));
			wrappers.add(wrapper);
		}
		return OrderTool.keepOrder(request.getArtIds(), wrappers, ArtInfoWrapper::getId);
	}

	public ArtInfoWrapper getArtInfo(BrandEnum brand, ArtRequest request) {
		ArtInfo artInfo = artInfoService.getArtInfo(request.getArtId(), brand);
		if (null == artInfo || StringTools.isEmpty(artInfo.getResource()))
			return null;
		boolean isMaster = artInfo.getPuid().equals(request.getUid());
		if (artInfo.getIsOpen() <= 0 && !isMaster)
			return null; // is private & not master
		// get show resources
		ArtResource resource = GsonTools.gson.fromJson(artInfo.getResource(), ArtResource.class);
		if (null == resource)
			return null;
		ArtInfoWrapper wrapper = ArtInfoWrapper.of(artInfo);
		// set resource info
		wrapper.setType(resource.getType()).setWid(resource.getWid());
		wrapper.setPic(AliyunBean.getImagesUrl(resource.getPic(), PictureCompress._510x280));
		// set pictures
		if (null != resource.getPictures()) {
			List<String> pictures = Lists.newArrayList();
			resource.getPictures().forEach(it -> pictures.add(AliyunBean.getImagesUrl(it, PictureCompress._510x280)));
			wrapper.setPictures(pictures);
		}
		// like count
		wrapper.setLikeCnt(artLikeService.countForArtId(artInfo.getId()));
		// current user is like
		wrapper.setIsLike(artLikeService.existForIdAndUid(artInfo.getId(), request.getUid()));
		// get user info
		wrapper.setUser(userProxy.getSimpleParentInfo(brand, UserRequest.of(request.getUid(), artInfo.getPuid())));
		// get student info
		if (request.getWithStudent())
			wrapper.setStudent(userProxy.getStudentInfo(artInfo.getSuid(), true, brand));
		return wrapper;
	}
}
